/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import shohaku.core.lang.Eval;

/**
 * 集合から要素を切り出すヘルパーメソッド群を提供します。
 */
public class HCut {

    /*
     * CharSequence
     */

    /**
     * 最初の文字を切り出します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @return 最初の文字
     */
    public static char begin(CharSequence cs) {
        return cs.charAt(0);
    }

    /**
     * 最後の文字を切り出します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @return 最後の文字
     */
    public static char end(CharSequence cs) {
        return cs.charAt(cs.length() - 1);
    }

    /*
     * trim
     */

    /**
     * 文字シーケンスの前方と後方から空白文字を削除した文字シーケンスを返却します。<br>
     * 第二引数が 0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param direction
     *            0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除する
     * @return 前方と後方から空白文字を削除した文字シーケンス
     */
    public static CharSequence trim(CharSequence cs, int direction) {
        int len = cs.length();
        int st = 0;
        if (0 <= direction) {
            while ((st < len) && (cs.charAt(st) <= ' ')) {
                st++;
            }
        }
        if (0 >= direction) {
            while ((st < len) && (cs.charAt(len - 1) <= ' ')) {
                len--;
            }
        }
        if (cs instanceof String) {
            return ((st > 0) || (len < cs.length())) ? cs.subSequence(st, len) : cs;
        } else {
            return cs.subSequence(st, len);
        }
    }

    /**
     * 文字シーケンスの前方と後方からJavaの基準による空白文字を削除した文字シーケンスを返却します。<br>
     * 第二引数が 0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param direction
     *            0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除する
     * @return 前方と後方からJavaの基準による空白文字を削除した文字シーケンス
     */
    public static CharSequence trimWhite(CharSequence cs, int direction) {
        int len = cs.length();
        int st = 0;
        if (0 <= direction) {
            while ((st < len) && Character.isWhitespace(cs.charAt(st))) {
                st++;
            }
        }
        if (0 >= direction) {
            while ((st < len) && Character.isWhitespace(cs.charAt(len - 1))) {
                len--;
            }
        }
        if (cs instanceof String) {
            return ((st > 0) || (len < cs.length())) ? cs.subSequence(st, len) : cs;
        } else {
            return cs.subSequence(st, len);
        }
    }

    /**
     * 文字シーケンスの前方と後方から指定の文字を削除した文字シーケンスを返却します。<br>
     * 第二引数が 0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param chars
     *            削除する文字
     * @param direction
     *            0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除する
     * @return 前方と後方から指定の文字を削除した文字シーケンス
     */
    public static CharSequence trim(CharSequence cs, char chars, int direction) {
        int len = cs.length();
        int st = 0;
        if (0 <= direction) {
            while ((st < len) && chars == cs.charAt(st)) {
                st++;
            }
        }
        if (0 >= direction) {
            while ((st < len) && chars == cs.charAt(len - 1)) {
                len--;
            }
        }
        if (cs instanceof String) {
            return ((st > 0) || (len < cs.length())) ? cs.subSequence(st, len) : cs;
        } else {
            return cs.subSequence(st, len);
        }
    }

    /**
     * 文字シーケンスの前方と後方から指定の文字を削除した文字シーケンスを返却します。<br>
     * 第二引数が 0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param chars
     *            削除する文字の配列
     * @param direction
     *            0 の場合は両端から削除、正数の場合は前方のみ削除、負数の場合は後方のみ削除する
     * @return 前方と後方から指定の文字を削除した文字シーケンス
     */
    public static CharSequence trim(CharSequence cs, char[] chars, int direction) {
        int len = cs.length();
        int st = 0;
        if (0 <= direction) {
            while ((st < len) && (Eval.isContains(chars, cs.charAt(st)))) {
                st++;
            }
        }
        if (0 >= direction) {
            while ((st < len) && (Eval.isContains(chars, cs.charAt(len - 1)))) {
                len--;
            }
        }
        if (cs instanceof String) {
            return ((st > 0) || (len < cs.length())) ? cs.subSequence(st, len) : cs;
        } else {
            return cs.subSequence(st, len);
        }
    }

    /*
     * cut
     */

    /**
     * 文字シーケンスの両端から指定の文字数を削除した文字シーケンスを返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param beginSize
     *            前方から切り出す文字数
     * @param endSize
     *            後方から切り出す文字数
     * @return 両端から指定の文字数を削除した文字シーケンス
     */
    public static CharSequence cut(CharSequence cs, int beginSize, int endSize) {
        return cs.subSequence(beginSize, (cs.length() - endSize));
    }

    /**
     * 文字シーケンスの前方から指定の文字数と、指定の文字が発見された位置を後方位置として切り出した文字シーケンスを返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param beginSize
     *            前方から切り出す文字数
     * @param endChar
     *            後方から切り出す位置を検索する文字
     * @return 両端から指定の文字数と文字の発見位置を切り出した文字シーケンス
     */
    public static CharSequence cutIndexOf(CharSequence cs, int beginSize, char endChar) {
        int endSize = HSeek.indexOf(cs, beginSize, endChar, 1);
        endSize = (-1 < endSize) ? endSize : cs.length();
        return cs.subSequence(beginSize, endSize);
    }

    /**
     * 文字シーケンスの指定の文字が発見された位置を前方位置として、指定の文字数を切り出した文字シーケンスを返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param beginChar
     *            前方から切り出す位置を検索する文字
     * @param endSize
     *            後方から切り出す文字数
     * @return 両端から指定の文字数と文字の発見位置を切り出した文字シーケンス
     */
    public static CharSequence cutIndexOf(CharSequence cs, char beginChar, int endSize) {
        int beginSize = HSeek.indexOf(cs, 0, (cs.length() - 1 - endSize), beginChar, 1);
        beginSize = (-1 < beginSize) ? beginSize : 0;
        return cs.subSequence(beginSize, (cs.length() - endSize));
    }

    /**
     * 文字シーケンスの指定の文字が発見された位置を前方位置として、後方から指定の文字数を切り出した文字シーケンスを返却します。
     * 
     * @param cs
     *            処理対象の文字シーケンス
     * @param beginChar
     *            前方から切り出す位置を検索する文字
     * @param endSize
     *            後方から切り出す文字数
     * @return 両端から指定の文字の発見位置と文字数を切り出した文字シーケンス
     */
    public static CharSequence cutLastIndexOf(CharSequence cs, char beginChar, int endSize) {
        int beginSize = HSeek.indexOf(cs, 0, (cs.length() - 1 - endSize), beginChar, -1);
        beginSize = (-1 < beginSize) ? beginSize + 1 : 0;
        return cs.subSequence(beginSize, (cs.length() - endSize));
    }

}
