/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.text.DateFormatSymbols;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.text.NumberFormat;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import shohaku.core.lang.Eval;
import shohaku.core.lang.RangeInt;

/**
 * データの型変換を行うヘルパーメソッド群を提供します。
 */
public class HCnv {

    /*
     * Converts the character
     */

    /**
     * 先頭文字を大文字に変換します。
     * 
     * @param s
     *            変換する文字列
     * @return 変換後の文字列
     */
    public static String capitalize(String s) {
        if (s == null || s.length() == 0) {
            return s;
        }
        final char[] chars = s.toCharArray();
        chars[0] = Character.toUpperCase(chars[0]);
        return new String(chars);
    }

    /*
     * Number
     */

    /**
     * 文字列を指定された書式とデフォルトロケールで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, String pattern) {
        return toDecimal(num, Locale.getDefault(), pattern);
    }

    /**
     * 文字列を指定された書式と指定のロケールで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param locale
     *            ロケール
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, Locale locale, String pattern) {
        return toDecimal(num, new DecimalFormatSymbols(locale), pattern);
    }

    /**
     * 文字列を指定された書式と指定の数値変換の記号セットで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param symbols
     *            数値変換の記号セット
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, DecimalFormatSymbols symbols, String pattern) {
        return toDecimal(num, symbols, new ParsePosition(0), pattern);
    }

    /**
     * 文字列を指定された書式と指定の数値変換の記号セットで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param symbols
     *            数値変換の記号セット
     * @param pos
     *            解析中に現在の位置を追跡する ParsePosition オブジェクト
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, DecimalFormatSymbols symbols, ParsePosition pos, String pattern) {
        final DecimalFormat format = new DecimalFormat(pattern, symbols);
        return parseNumber(num, pos, format);
    }

    /**
     * 文字列を指定された複数の書式と指定の数値変換の記号セットで数値型に変換します。<br>
     * すべての変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param symbols
     *            数値変換の記号セットs
     * @param patterns
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, DecimalFormatSymbols symbols, Collection patterns) {
        return toDecimal(num, symbols, 0, patterns);
    }

    /**
     * 文字列を指定された複数の書式と指定の数値変換の記号セットで数値型に変換します。<br>
     * すべての変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param symbols
     *            数値変換の記号セット
     * @param posIndex
     *            解析の開始インデックス
     * @param patterns
     *            書式
     * @return 数値オブジェクト
     */
    public static Number toDecimal(CharSequence num, DecimalFormatSymbols symbols, int posIndex, Collection patterns) {
        final DecimalFormat format = new DecimalFormat();
        format.setDecimalFormatSymbols(symbols);

        for (Iterator i = patterns.iterator(); i.hasNext();) {
            final String pattern = (String) i.next();
            final ParsePosition pos = new ParsePosition(posIndex);
            format.applyPattern(pattern);
            final Number n = parseNumber(num, pos, format);
            if (n != null) {
                return n;
            }
        }
        return null;
    }

    /**
     * 文字列を指定された書式とデフォルトロケールで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param roundingMode
     *            BigDecimal で定義される小数部分の丸めモード
     * @param maxScale
     *            小数部分の最大桁数
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     * @throws ArithmeticException
     *             roundingMode==ROUND_UNNECESSARY であり、指定したスケール演算で丸めが必要な場合
     * @throws IllegalArgumentException
     *             roundingMode が有効な丸めモードを示さない場合
     * @see java.math.BigDecimal#ROUND_UP
     * @see java.math.BigDecimal#ROUND_DOWN
     * @see java.math.BigDecimal#ROUND_CEILING
     * @see java.math.BigDecimal#ROUND_FLOOR
     * @see java.math.BigDecimal#ROUND_HALF_UP
     * @see java.math.BigDecimal#ROUND_HALF_DOWN
     * @see java.math.BigDecimal#ROUND_HALF_EVEN
     * @see java.math.BigDecimal#ROUND_UNNECESSARY
     */
    public static Number toDecimal(CharSequence num, int roundingMode, int maxScale, String pattern) {
        return toDecimal(num, new DecimalFormatSymbols(), new ParsePosition(0), roundingMode, new RangeInt(0, maxScale), pattern);
    }

    /**
     * 文字列を指定された書式と指定の数値変換の記号セットで数値型に変換します。<br>
     * 変換に失敗した場合 null を返却します。<br>
     * 通常の値は BigDecimal オブジェクトとして返されます。<br>
     * 特殊な値 (正負の無限大や NaN など) は、対応する Double 定数値を含む Double インスタンスとして返されます。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param symbols
     *            数値変換の記号セット
     * @param pos
     *            解析中に現在の位置を追跡する ParsePosition オブジェクト
     * @param roundingMode
     *            BigDecimal で定義される小数部分の丸めモード
     * @param scale
     *            小数部分の最小桁数と最大桁数
     * @param pattern
     *            書式
     * @return 数値オブジェクト
     * @throws ArithmeticException
     *             roundingMode==ROUND_UNNECESSARY であり、指定したスケール演算で丸めが必要な場合
     * @throws IllegalArgumentException
     *             roundingMode が有効な丸めモードを示さない場合
     * @see java.math.BigDecimal#ROUND_UP
     * @see java.math.BigDecimal#ROUND_DOWN
     * @see java.math.BigDecimal#ROUND_CEILING
     * @see java.math.BigDecimal#ROUND_FLOOR
     * @see java.math.BigDecimal#ROUND_HALF_UP
     * @see java.math.BigDecimal#ROUND_HALF_DOWN
     * @see java.math.BigDecimal#ROUND_HALF_EVEN
     * @see java.math.BigDecimal#ROUND_UNNECESSARY
     */
    public static Number toDecimal(CharSequence num, DecimalFormatSymbols symbols, ParsePosition pos, int roundingMode, RangeInt scale, String pattern) {
        final DecimalFormat format = new DecimalFormat(pattern, symbols);
        format.setMinimumFractionDigits(scale.getMin());
        format.setMaximumFractionDigits(scale.getMax() + 1);
        final Number n = parseNumber(num, pos, format);
        if (n instanceof BigDecimal) {
            return ((BigDecimal) n).setScale(scale.getMax(), roundingMode);
        }
        return n;
    }

    /**
     * 数値文字シーケンスの解析処理を実行します。
     * 
     * @param num
     *            変換する文字シーケンス
     * @param pos
     *            解析中に現在の位置を追跡する ParsePosition オブジェクト
     * @param format
     *            数値フォーマッター
     * @return 数値オブジェクト
     */
    static Number parseNumber(CharSequence num, ParsePosition pos, NumberFormat format) {
        final Number n = format.parse(num.toString(), pos);
        if (pos.getErrorIndex() == -1) {
            return null;
        }
        return (Eval.isInfiniteOrNaN(n)) ? n : new BigDecimal(n.toString());
    }

    /*
     * 日付
     */

    /**
     * 引数を日付データを等価な日付オブジェクトで返却します。
     * 
     * @param src
     *            日付型またはカレンダー、数値型
     * @return 日付オブジェクト
     */
    public static Date asDataObject(Object src) {
        if (src instanceof Date) {
            return (Date) src;
        }
        if (src instanceof Calendar) {
            return ((Calendar) src).getTime();
        }
        if (src instanceof Number) {
            final Calendar calendar = Calendar.getInstance();
            calendar.setTimeInMillis(((Number) src).longValue());
            return calendar.getTime();
        }
        throw new IllegalArgumentException("is not Date Type. src=" + src);
    }

    /**
     * 文字列を指定の書式として解析を行い日付型に変換します。 <br>
     * 変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param pattern
     *            書式
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, String pattern) {
        return toDateTime(date, Locale.getDefault(), pattern);
    }

    /**
     * 文字列を指定の書式として解析を行い日付型に変換します。 <br>
     * 変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param locale
     *            ロケール
     * @param pattern
     *            書式
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, Locale locale, String pattern) {
        return toDateTime(date, new DateFormatSymbols(locale), pattern);
    }

    /**
     * 文字列を指定の書式として解析を行い日付型に変換します。 <br>
     * 変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, DateFormatSymbols symbols, String pattern) {
        return toDateTime(date, symbols, pattern, false);
    }

    /**
     * 文字列を日付に変換します。<br>
     * 変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param symbols
     *            日付の記号セット
     * @param pattern
     *            書式
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, DateFormatSymbols symbols, String pattern, boolean lenient) {
        return toDateTime(date, symbols, new ParsePosition(0), pattern, lenient);
    }

    /**
     * 文字列を日付に変換します。<br>
     * 変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param symbols
     *            日付の記号セット
     * @param pos
     *            解析中に現在の位置を追跡する ParsePosition オブジェクト
     * @param pattern
     *            書式
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, DateFormatSymbols symbols, ParsePosition pos, String pattern, boolean lenient) {
        final DateFormat format = new SimpleDateFormat(pattern, symbols);
        format.setLenient(lenient);
        final Date d = format.parse(date.toString(), pos);
        if (pos.getErrorIndex() != -1) {
            return null;
        }
        return d;
    }

    /**
     * 文字列を指定の書式の反復子順に解析を行い最初に成功した書式で返却します。 <br>
     * 全ての変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param patterns
     *            書式
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, Collection patterns) {
        return toDateTime(date, Locale.getDefault(), patterns);
    }

    /**
     * 文字列を書式の反復子順に解析を行い最初に成功した書式で日付に変換して返却します。 <br>
     * 全ての変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param locale
     *            ロケール
     * @param patterns
     *            書式
     * @return 一つ以上の書式で日付文字列として認識できる場合は true
     */
    public static Date toDateTime(CharSequence date, Locale locale, Collection patterns) {
        return toDateTime(date, new DateFormatSymbols(locale), patterns, false);
    }

    /**
     * 文字列を書式の反復子順に解析を行い最初に成功した書式で日付に変換して返却します。 <br>
     * 全ての変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param symbols
     *            日付の記号セット
     * @param patterns
     *            書式
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, DateFormatSymbols symbols, Collection patterns, boolean lenient) {
        return toDateTime(date, symbols, 0, patterns, lenient);
    }

    /**
     * 文字列を書式の反復子順に解析を行い最初に成功した書式で日付に変換して返却します。 <br>
     * 全ての変換に失敗した場合 null を返します。
     * 
     * @param date
     *            変換する文字シーケンス
     * @param symbols
     *            日付の記号セット
     * @param posIndex
     *            解析の開始インデックス
     * @param patterns
     *            書式
     * @param lenient
     *            日付/時刻解析を曖昧に行うか設定する、true=曖昧な解析
     * @return 日付オブジェクト
     */
    public static Date toDateTime(CharSequence date, DateFormatSymbols symbols, int posIndex, Collection patterns, boolean lenient) {

        final SimpleDateFormat format = new SimpleDateFormat();
        format.setDateFormatSymbols(symbols);
        format.setLenient(lenient);
        final String sdate = date.toString();

        for (Iterator i = patterns.iterator(); i.hasNext();) {
            final String pattern = (String) i.next();
            final ParsePosition pos = new ParsePosition(posIndex);
            format.applyPattern(pattern);

            final Date d = format.parse(sdate, pos);
            if (pos.getErrorIndex() == -1) {
                return d;
            }// else { // no op, try next }
        }
        return null;
    }

}
