/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import shohaku.core.lang.Boxing;
import shohaku.core.lang.Closure;
import shohaku.core.lang.Concat;
import shohaku.core.lang.Eval;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.feature.FeatureFactory;

/**
 * クラスに関するヘルパーメソッド群を提供します。
 */
public class HClass {

    /* オブジェクト型の配列シンボル。 */
    static final char OBJECT_SYMBOL = 'L';

    /* プリミティブ型のクラスと省略名のマッピング。 */
    static final Map PRIMITIVE_FOR_TYPE_MAP;
    static {
        HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE.getName(), Byte.TYPE);
        m.put(Short.TYPE.getName(), Short.TYPE);
        m.put(Integer.TYPE.getName(), Integer.TYPE);
        m.put(Long.TYPE.getName(), Long.TYPE);
        m.put(Float.TYPE.getName(), Float.TYPE);
        m.put(Double.TYPE.getName(), Double.TYPE);
        m.put(Boolean.TYPE.getName(), Boolean.TYPE);
        m.put(Character.TYPE.getName(), Character.TYPE);
        m.put(Void.TYPE.getName(), Void.TYPE);
        PRIMITIVE_FOR_TYPE_MAP = Collections.unmodifiableMap(m);
    }

    /* プリミティブ型のクラスとクラス名のマッピング。 */
    static final Map PRIMITIVE_CLASS_NAME_MAP;
    static {
        HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE, "B");
        m.put(Short.TYPE, "S");
        m.put(Integer.TYPE, "I");
        m.put(Long.TYPE, "J");
        m.put(Float.TYPE, "F");
        m.put(Double.TYPE, "D");
        m.put(Boolean.TYPE, "Z");
        m.put(Character.TYPE, "C");
        m.put(Void.TYPE, "V");
        PRIMITIVE_CLASS_NAME_MAP = Collections.unmodifiableMap(m);
    }

    /**
     * １次元または多次元配列の基になるクラスを返却します。<br>
     * また配列以外の場合はそのまま返却します。 <br>
     * つまり java.lang.String[][] または java.lang.String[]、java.lang.String に対して java.lang.String クラスが返されます。
     * 
     * @param clazz
     *            検証するクラス
     * @return １次元または多次元配列の基に為るクラス型
     */
    public static Class getSourceComponentType(Class clazz) {
        if (clazz.isArray()) {
            return getSourceComponentType(clazz.getComponentType());
        }
        return clazz;
    }

    /**
     * 配列の次元数を算出して返却します。
     * 
     * @param clazz
     *            検証するクラス
     * @return 配列の次元数
     */
    public static int getArraysDimensionSize(Class clazz) {
        int size = 0;
        Class srcClass = clazz;
        while (srcClass.isArray()) {
            size++;
            srcClass = srcClass.getComponentType();
        }
        return size;
    }

    /**
     * パッケージ名を含まないクラス名を返却します。
     * 
     * @param clazz
     *            クラス
     * @return パッケージ名を含まないクラス名
     */
    public static String getShortClassName(Class clazz) {
        return (String) HCut.cutLastIndexOf(clazz.getName(), '.', 0);
    }

    /**
     * ソースコードと類似の表記でクラス名を返却します。<br>
     * 配列を [] で表現します。
     * 
     * @param clazz
     *            検証するクラス
     * @return 配列を [] で示すクラス名
     */
    public static String getSourceClassName(Class clazz) {
        // 再起的に次元数分の配列シンボルを結合
        final StringBuffer arraySymbol = new StringBuffer();
        Class srcClass = clazz;
        while (srcClass.isArray()) {
            arraySymbol.append("[]");
            srcClass = srcClass.getComponentType();
        }
        // 根のクラスと配列シンボルの結合
        return Concat.get(srcClass.getName(), arraySymbol);
    }

    /**
     * 引数がプリミティブ型の場合は対応するラッパクラスを返却し、以外はそのまま返却します。
     * 
     * @param clazz
     *            クラス
     * @return 対応するオブジェクト型のクラス
     */
    public static Class box(Class clazz) {
        if (clazz.isPrimitive()) {
            return Boxing.boxClass(clazz);
        }
        return clazz;
    }

    /**
     * ラッパークラスの一致を検証します。
     * 
     * @param type
     *            比較元のクラス
     * @param type2
     *            比較先のクラス
     * @return ラッパークラスが同一クラスの場合は true
     */
    public static boolean boxEquals(Class type, Class type2) {
        Class t = type;
        if (t.isPrimitive()) {
            t = Boxing.boxClass(t);
        }
        Class t2 = type2;
        if (t2.isPrimitive()) {
            t2 = Boxing.boxClass(t2);
        }
        return t.equals(t2);
    }

    /**
     * Java構文（例：java.lang.Integer, int, Integer, Integer[]）からクラスをリードし返却します。
     * 
     * @param name
     *            クラスを示す文字列
     * @param loader
     *            クラスローダー、 null を指定すると標準のクラスローダを使用します
     * @param finder
     *            クラス名を引数としてクラスを検索し返すクロージャ、 null を指定すると何も行いません
     * @return ロードされたクラス
     * @throws NoSuchResourceException
     *             クラスのリードに失敗した場合
     */
    public static Class load(String name, ClassLoader loader, Closure finder) throws NoSuchResourceException {

        // 拡張の配列表記の検証 ”String[][] 等”
        final StringBuffer symbol = new StringBuffer(4);
        int off = name.length();
        while (off >= 2 && name.charAt(off - 1) == ']' && name.charAt(off - 2) == '[') {
            off -= 2;
            symbol.append('[');
        }
        final boolean isArray = (symbol.length() > 0);

        // 拡張表記でクラスを検索(クラス名の配列部を切る)
        final String rootName = ((isArray) ? name.substring(0, off) : name);
        final Class clazz = findByClassName(rootName, finder);
        if (isArray) {
            final StringBuffer className = new StringBuffer();
            // 配列の表記を付加する
            if (clazz != null) {
                final String primitiveName = (String) PRIMITIVE_CLASS_NAME_MAP.get(clazz);
                if (primitiveName == null) {
                    // オブジェクト型
                    className.append(symbol).append(OBJECT_SYMBOL).append(clazz.getName()).append(';');
                } else {
                    // プリミティブ型
                    className.append(symbol).append(primitiveName);
                }
            } else {
                // オブジェクト型
                className.append(symbol).append(OBJECT_SYMBOL).append(rootName).append(';');
            }
            // 通常のルールでクラスをロードする
            return FeatureFactory.getLoader().getClass(className.toString(), loader);
        } else {
            // 配列型以外は返却
            if (clazz != null) {
                return clazz;
            }
            // 通常のルールでクラスをロードする
            return FeatureFactory.getLoader().getClass(rootName, loader);
        }

    }

    static Class findByClassName(String className, Closure finder) {
        Class clazz = (Class) PRIMITIVE_FOR_TYPE_MAP.get(className);
        if (clazz == null) {
            if (!Eval.isContains(className, '.')) {
                try {
                    clazz = Class.forName(Concat.get("java.lang.", className));
                } catch (ClassNotFoundException e) {
                    // no op
                }
            }
        }
        if (clazz == null && finder != null) {
            clazz = (Class) finder.evaluate(className);
        }
        return clazz;
    }

}
