/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.WeakHashMap;

import shohaku.core.lang.Eval;
import shohaku.core.lang.IntrospectionBeansException;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.lang.feature.LogFeature;

/**
 * Java Bean を動的に内部参照を実行する為の定義情報を検索し保存する機能を提供します。
 */
class BeanIntrospectDescriptor {

    /* shohaku core logging. */
    private static final LogFeature log = FeatureFactory.getLog(BeanIntrospectDescriptor.class);

    /* 弱参照を用いてクラスをキーとしてBean記述子をキャッシュします。 */
    private static final Map classCache = new WeakHashMap();

    /**
     * 指定のクラスの情報を持つ BeanIntrospectDescriptor インスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @return BeanIntrospectDescriptor インスタンス
     * @throws IntrospectionBeansException
     *             生成に失敗した場合
     */
    static BeanIntrospectDescriptor forClass(Class clazz) throws IntrospectionBeansException {
        if (clazz == null) {
            throw new NullPointerException("argument is null.");
        }
        synchronized (classCache) {
            BeanIntrospectDescriptor results = (BeanIntrospectDescriptor) classCache.get(clazz);
            if (results == null) {
                // can throw IntrospectionBeansException
                results = new BeanIntrospectDescriptor(clazz);
                classCache.put(clazz, results);
            }
            return results;
        }
    }

    /* BeanInfo のキャッシュ */
    private final BeanInfo beanInfo;

    /* Constructor のコレクションです。 */
    private Collection constructors;

    /* PropertyDescriptor をプロパティ名でマッピングします。 */
    private Map propDesc;

    /* Method をメソッド名でグルーピングします。 */
    private Map methodGroup;

    /* 特権でアクセス可能な Method をメソッド名でグルーピングします。 */
    private Map accessibleMethodGroup;

    /* 静的な公開された定数フィールドをフィールド名でマッピングします。 */
    private Map constants;

    /* Field をフィールド名でグルーピングします。 */
    private Map fieldGroup;

    /* 特権でアクセス可能な Field をフィールド名でグルーピングします。 */
    private Map accessibleFieldGroup;

    /* 指定のクラスの BeanInfo をキャッシュして初期化する */
    private BeanIntrospectDescriptor(Class clazz) throws IntrospectionBeansException {

        try {

            if (log.isDebugEnabled()) {
                log.debug("get BeanInfo." + clazz.getName());
            }
            this.beanInfo = Introspector.getBeanInfo(clazz);

            // クラスと親クラスに関する Introspector の内部キャッシュ情報をフラッシュします。
            // このクラスは弱参照を用いるため GC の効率性に有効です。（Spring を参考にしました）
            Class classToFlush = clazz;
            do {
                Introspector.flushFromCaches(classToFlush);
                classToFlush = classToFlush.getSuperclass();
            } while (classToFlush != null);

        } catch (SecurityException e) {
            throw new IntrospectionBeansException("Security error. " + clazz.getName(), e);
        } catch (IntrospectionException e) {
            throw new IntrospectionBeansException("Cannot get BeanInfo for object. " + clazz.getName(), e);
        }
    }

    /* Constructor のコレクション */
    private Collection constructors() {
        final Collection conses = new LinkedList();
        try {
            final Constructor[] cs = getBeanClass().getConstructors();
            for (int i = 0; i < cs.length; i++) {
                conses.add(cs[i]);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        // unmodifiable
        return Collections.unmodifiableCollection(conses);
    }

    /* Method のグルーピング */
    private Map methodGroup() {
        if (!Modifier.isPublic(getBeanClass().getModifiers())) {
            return Collections.EMPTY_MAP;
        }
        final HashMap group = new HashMap();
        methodGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            final Map.Entry e = (Map.Entry) i.next();
            final Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* Method を再起的に検索して名前でグルーピングする */
    private void methodGroup0(Map group, Class clazz) {
        try {
            final Method[] ms = clazz.getDeclaredMethods();
            for (int i = 0; i < ms.length; i++) {
                final Method m = ms[i];
                if (!Modifier.isPublic(m.getModifiers())) {
                    continue;
                }
                if (!Modifier.isPublic(m.getDeclaringClass().getModifiers())) {
                    m.setAccessible(true);
                }
                Collection coll = (Collection) group.get(m.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(m.getName(), coll);
                }
                coll.add(m);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        final Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            final Class c = interfaces[i];
            methodGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            final Class c = clazz.getSuperclass();
            if (c != null) {
                methodGroup0(group, c);
            }
        }
    }

    /* 特権でアクセス可能な Method のグルーピング */
    private Map accessibleMethodGroup() {
        final HashMap group = new HashMap();
        accessibleMethodGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            final Map.Entry e = (Map.Entry) i.next();
            final Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* 特権でアクセス可能な Method を再起的に検索して名前でグルーピングする */
    private void accessibleMethodGroup0(Map group, Class clazz) {
        try {
            final Method[] ms = clazz.getDeclaredMethods();
            for (int i = 0; i < ms.length; i++) {
                final Method m = ms[i];
                if (!Modifier.isPublic(m.getModifiers()) || !Modifier.isPublic(m.getDeclaringClass().getModifiers())) {
                    m.setAccessible(true);
                }
                Collection coll = (Collection) group.get(m.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(m.getName(), coll);
                }
                coll.add(m);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        final Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            final Class c = interfaces[i];
            accessibleMethodGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            final Class c = clazz.getSuperclass();
            if (c != null) {
                accessibleMethodGroup0(group, c);
            }
        }
    }

    /* Field のグルーピング */
    private Map fieldGroup() {
        if (!Modifier.isPublic(getBeanClass().getModifiers())) {
            return Collections.EMPTY_MAP;
        }
        final HashMap group = new HashMap();
        fieldGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            final Map.Entry e = (Map.Entry) i.next();
            final Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* Field を再起的に検索して名前でグルーピングする */
    private void fieldGroup0(Map group, Class clazz) {
        try {
            final Field[] fs = clazz.getDeclaredFields();
            for (int i = 0; i < fs.length; i++) {
                final Field f = fs[i];
                if (!Modifier.isPublic(f.getModifiers())) {
                    continue;
                }
                if (!Modifier.isPublic(f.getDeclaringClass().getModifiers())) {
                    f.setAccessible(true);
                }
                Collection coll = (Collection) group.get(f.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(f.getName(), coll);
                }
                coll.add(f);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        final Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            final Class c = interfaces[i];
            fieldGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            final Class c = clazz.getSuperclass();
            if (c != null) {
                fieldGroup0(group, c);
            }
        }
    }

    /* 特権でアクセス可能な Field のグルーピング */
    private Map accessibleFieldGroup() {
        final HashMap group = new HashMap();
        accessibleFieldGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            final Map.Entry e = (Map.Entry) i.next();
            final Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* 特権でアクセス可能な Field を再起的に検索して名前でグルーピングする */
    private void accessibleFieldGroup0(Map group, Class clazz) {
        try {
            final Field[] fs = clazz.getDeclaredFields();
            for (int i = 0; i < fs.length; i++) {
                final Field f = fs[i];
                if (!Modifier.isPublic(f.getModifiers()) || !Modifier.isPublic(f.getDeclaringClass().getModifiers())) {
                    f.setAccessible(true);
                }
                Collection coll = (Collection) group.get(f.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(f.getName(), coll);
                }
                coll.add(f);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        final Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            final Class c = interfaces[i];
            accessibleFieldGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            final Class c = clazz.getSuperclass();
            if (c != null) {
                accessibleFieldGroup0(group, c);
            }
        }
    }

    /* PropertyDescriptor のマッピング */
    private Map propDesc() {
        if (!Modifier.isPublic(getBeanClass().getModifiers())) {
            return Collections.EMPTY_MAP;
        }
        final HashMap pdsMap = new HashMap();
        try {
            final PropertyDescriptor[] pds = this.beanInfo.getPropertyDescriptors();
            for (int i = 0; i < pds.length; i++) {
                final PropertyDescriptor pd = pds[i];
                final Method readMethod = pd.getReadMethod();
                final Method writeMethod = pd.getWriteMethod();
                if (readMethod != null && !Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                    readMethod.setAccessible(true);
                }
                if (writeMethod != null && !Modifier.isPublic(writeMethod.getDeclaringClass().getModifiers())) {
                    writeMethod.setAccessible(true);
                }
                pdsMap.put(pd.getName(), pd);
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        }
        return Collections.unmodifiableMap(pdsMap);
    }

    /* 定数フィールドのマッピング */
    private Map constants() {
        if (!Modifier.isPublic(getBeanClass().getModifiers())) {
            return Collections.EMPTY_MAP;
        }
        final HashMap fsMap = new HashMap();
        try {
            final Class clazz = getBeanClass();
            final Field[] fields = clazz.getFields();
            for (int i = 0; i < fields.length; i++) {
                final Field field = fields[i];
                // public and static and final
                final int modifiers = field.getModifiers();
                if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
                    fsMap.put(field.getName(), field.get(null));
                }
            }
        } catch (SecurityException e) {
            log.debug("Security error. " + getBeanClass().getName(), e);
        } catch (IllegalArgumentException e) {
            log.debug("field Introspection error. " + getBeanClass().getName(), e);
        } catch (IllegalAccessException e) {
            log.debug("field Introspection error. " + getBeanClass().getName(), e);
        }
        return Collections.unmodifiableMap(fsMap);
    }

    /*
     * BeanInfo
     */

    /**
     * このインスタンスの java.beans.BeanInfo を返却します。
     * 
     * @return java.beans.BeanInfo
     */
    BeanInfo getBeanInfo() {
        return this.beanInfo;
    }

    /**
     * このインスタンスのビーンクラスを返却します。
     * 
     * @return ビーンクラス
     */
    Class getBeanClass() {
        return this.beanInfo.getBeanDescriptor().getBeanClass();
    }

    /*
     * Constructors
     */

    /**
     * パブリック定数を不変マップに格納して返却します。
     * 
     * @return パブリック定数の名前と値の不変マップ
     */
    Collection getConstructors() {
        final Collection c = constructors;
        return (c != null) ? c : (constructors = constructors());
    }

    /**
     * 指定されたパラメータ型を持つ Constructor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param paramTypes
     *            パラメータ型
     * @return 指定されたパラメータ型を持つ Constructor
     */
    Constructor getConstructor(Class[] paramTypes) {
        if (paramTypes == null) {
            throw new NullPointerException("argument is null.");
        }
        final Collection conses = getConstructors();
        for (Iterator i = conses.iterator(); i.hasNext();) {
            final Constructor cons = (Constructor) i.next();
            if (Eval.isRefEquals(cons.getParameterTypes(), paramTypes)) {
                return cons;
            }
        }
        return null;
    }

    /**
     * 指定されたパラメータ型を持つ Constructor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param paramTypes
     *            パラメータ型
     * @return 指定されたパラメータ型を持つ Constructor
     */
    Constructor getAssignmentConstructor(Class[] paramTypes) {
        if (paramTypes == null) {
            throw new NullPointerException("argument is null.");
        }
        // equals parameter types
        Constructor cons = getConstructor(paramTypes);
        if (cons != null) {
            return cons;
        }
        // is Aassignment
        final Collection conses = getConstructors();
        for (Iterator i = conses.iterator(); i.hasNext();) {
            cons = (Constructor) i.next();
            if (HBeans.isAssignmentCompatible(cons.getParameterTypes(), paramTypes)) {
                return cons;
            }
        }
        return null;
    }

    /*
     * Constant Field
     */

    /**
     * パブリック定数を不変マップに格納して返却します。
     * 
     * @return パブリック定数の名前と値の不変マップ
     */
    Map getConstantFieldMap() {
        final Map m = constants;
        return (m != null) ? m : (constants = constants());
    }

    /**
     * 指定された名前のパブリック定数の値を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return パブリック定数の値
     */
    Object getConstantField(String fieldName) {
        return getConstantFieldMap().get(fieldName);
    }

    /*
     * PropertyDescriptor
     */

    /**
     * PropertyDescriptor を不変マップに格納して返却します。
     * 
     * @return PropertyDescriptor のプロパティ名と PropertyDescriptor の不変マップ
     */
    Map getPropertyDescriptorMap() {
        final Map m = propDesc;
        return (m != null) ? m : (propDesc = propDesc());
    }

    /**
     * PropertyDescriptor を不変コレクションに格納して返却します。
     * 
     * @return PropertyDescriptor のコレクション
     */
    Collection getPropertyDescriptors() {
        return getPropertyDescriptorMap().values();
    }

    /**
     * 指定された名前の PropertyDescriptor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param propertyName
     *            プロパティ名
     * @return PropertyDescriptor
     */
    PropertyDescriptor getPropertyDescriptor(String propertyName) {
        return (PropertyDescriptor) getPropertyDescriptorMap().get(propertyName);
    }

    /*
     * Field
     */

    /**
     * 同一のフィールド名の Field を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のフィールド名の Field を不変コレクションでマッピングした不変マップ
     */
    Map getFieldGroup() {
        final Map g = fieldGroup;
        return (g != null) ? g : (fieldGroup = fieldGroup());
    }

    /**
     * 指定された名前の Field を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field の不変コレクション
     */
    Collection getFields(String fieldName) {
        return (Collection) getFieldGroup().get(fieldName);
    }

    /**
     * 指定された名前の Field を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field
     */
    Field getField(String fieldName) {
        final Collection coll = (Collection) getFieldGroup().get(fieldName);
        if (coll != null) {
            return (Field) coll.iterator().next();
        }
        return null;
    }

    /*
     * Accessible Field
     */

    /**
     * 同一のフィールド名の、特権でアクセス可能な Field を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のフィールド名の Field を不変コレクションでマッピングした不変マップ
     */
    Map getAccessibleFieldGroup() {
        final Map g = accessibleFieldGroup;
        return (g != null) ? g : (accessibleFieldGroup = accessibleFieldGroup());
    }

    /**
     * 指定された名前の、特権でアクセス可能な Field を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field の不変コレクション
     */
    Collection getAccessibleFields(String fieldName) {
        return (Collection) getAccessibleFieldGroup().get(fieldName);
    }

    /**
     * 指定された名前の、特権でアクセス可能な Field を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field
     */
    Field getAccessibleField(String fieldName) {
        final Collection coll = (Collection) getAccessibleFieldGroup().get(fieldName);
        if (coll != null) {
            return (Field) coll.iterator().next();
        }
        return null;
    }

    /*
     * Method
     */

    /**
     * 同一のメソッド名の Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のメソッド名の Method を不変コレクションでマッピングした不変マップ
     */
    Map getMethodGroup() {
        final Map g = methodGroup;
        return (g != null) ? g : (methodGroup = methodGroup());
    }

    /**
     * 指定された名前の Method を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @return Method の不変コレクション
     */
    Collection getMethods(String methodName) {
        return (Collection) getMethodGroup().get(methodName);
    }

    /**
     * 指定された名前とパラメータ型を持つ Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getMethod(String methodName, Class[] parameterTypes) {
        final Collection coll = getMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                final Method m = (Method) i.next();
                if (Eval.isRefEquals(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

    /**
     * 指定された名前と、指定されたパラメータ型に割り当て可能なパラメータ型を持つ Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getAssignmentMethod(String methodName, Class[] parameterTypes) {
        final Collection coll = getMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                final Method m = (Method) i.next();
                if (HBeans.isAssignmentCompatible(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

    /*
     * Accessible Method
     */

    /**
     * 同一のメソッド名の、特権でアクセス可能な Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のメソッド名の Method を不変コレクションでマッピングした不変マップ
     */
    Map getAccessibleMethodGroup() {
        final Map g = accessibleMethodGroup;
        return (g != null) ? g : (accessibleMethodGroup = accessibleMethodGroup());
    }

    /**
     * 指定された名前の、特権でアクセス可能な Method を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @return Method の不変コレクション
     */
    Collection getAccessibleMethods(String methodName) {
        return (Collection) getAccessibleMethodGroup().get(methodName);
    }

    /**
     * 指定された名前とパラメータ型を持つ、特権でアクセス可能な Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getAccessibleMethod(String methodName, Class[] parameterTypes) {
        final Collection coll = getAccessibleMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                final Method m = (Method) i.next();
                if (Eval.isRefEquals(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

    /**
     * 指定された名前と、指定されたパラメータ型に割り当て可能なパラメータ型を持つ、特権でアクセス可能な Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getAssignmentAccessibleMethod(String methodName, Class[] parameterTypes) {
        final Collection coll = getAccessibleMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                final Method m = (Method) i.next();
                if (HBeans.isAssignmentCompatible(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

}
