/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.cache;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import shohaku.core.collections.Cache;
import shohaku.core.functor.FFactory;

/**
 * このクラスは、Cache インタフェースのスケルトン実装を提供し、このインタフェースを実装するのに必要な作業量を最小限に抑えます。
 */
public abstract class AbstractCache implements Cache {

    /* Object on which to synchronize */
    final Object mutex;

    /* 実装の基となる Map。 */
    final Map map;

    /* 最大キャッシュサイズ */
    int maximumSize;

    FFactory factory;

    /**
     * マップ、最大キャッシュサイズ、初期化データを登録して初期化します。
     * 
     * @param t
     *            基にする Map
     * @param maxSize
     *            最大キャッシュサイズ
     * @param initCache
     *            初期化データ、null の場合は無視される
     */
    protected AbstractCache(Map t, int maxSize, Cache initCache) {
        this.map = t;
        this.maximumSize = maxSize;
        this.mutex = this;
        if (initCache != null) {
            putAll(initCache);
        }
    }

    public Object getMutex() {
        return mutex;
    }

    public int size() {
        synchronized (mutex) {
            return map.size();
        }
    }

    public void clear() {
        synchronized (mutex) {
            map.clear();
        }
    }

    public int getMaxSize() {
        synchronized (mutex) {
            return maximumSize;
        }
    }

    public void setMaxSize(int maxSize) {
        synchronized (mutex) {
            this.maximumSize = maxSize;
            if (0 <= maxSize) {
                resize(maxSize);
            }
        }
    }

    public boolean isLimit() {
        synchronized (mutex) {
            return (maximumSize >= 0 && maximumSize <= size());
        }
    }

    public boolean resize(int newSize) {
        if (0 > newSize) {
            throw new IllegalArgumentException("size is a negative number. " + newSize);
        }
        synchronized (mutex) {
            int _size = size();
            if (newSize < _size) {
                final Iterator i = map.entrySet().iterator();
                while (i.hasNext()) {
                    i.next();
                    i.remove();
                    if (newSize == --_size) {
                        break;
                    }
                }
                return true;
            } else {
                return false;
            }
        }
    }

    public boolean isEmpty() {
        synchronized (mutex) {
            return map.isEmpty();
        }
    }

    public boolean containsKey(Object key) {
        synchronized (mutex) {
            return map.containsKey(key);
        }
    }

    public boolean containsValue(Object value) {
        synchronized (mutex) {
            return map.containsValue(value);
        }
    }

    public Object get(Object key) {
        synchronized (mutex) {
            if (factory == null) {
                return map.get(key);
            } else {
                if (!map.containsKey(key)) {
                    final Object o = factory.create(key);
                    put(key, o);
                    return o;
                }
                return map.get(key);
            }
        }
    }

    public FFactory getFactory() {
        synchronized (mutex) {
            return factory;
        }
    }

    public void setFactory(FFactory factory) {
        synchronized (mutex) {
            this.factory = factory;
        }
    }

    public Object remove(Object key) {
        synchronized (mutex) {
            return map.remove(key);
        }
    }

    public Object put(Object key, Object value) {
        synchronized (mutex) {
            if (!map.containsKey(key)) {
                if (isLimit()) {
                    resize(size() - 1);
                }
            }
            return map.put(key, value);
        }
    }

    public void putAll(Cache t) {
        synchronized (mutex) {
            for (Iterator i = t.entrySet().iterator(); i.hasNext();) {
                final Map.Entry e = (Map.Entry) i.next();
                put(e.getKey(), e.getValue());
            }
        }
    }

    public void putAll(Map t) {
        synchronized (mutex) {
            for (Iterator i = t.entrySet().iterator(); i.hasNext();) {
                final Map.Entry e = (Map.Entry) i.next();
                put(e.getKey(), e.getValue());
            }
        }
    }

    /*
     * Views
     */

    /* 値コレクションビューのキャッシュです。 */
    transient Collection values = null;

    /* キーセットビューのキャッシュです。 */
    transient Set keySet = null;

    /* エントリセットビューのキャッシュです。 */
    transient Set entrySet = null;

    public Collection values() {
        synchronized (mutex) {
            return ((values != null) ? values : (values = new Values()));
        }
    }

    public Set keySet() {
        synchronized (mutex) {
            return ((keySet != null) ? keySet : (keySet = new KeySet()));
        }
    }

    public Set entrySet() {
        synchronized (mutex) {
            return ((entrySet != null) ? entrySet : (entrySet = new EntrySet()));
        }
    }

    /**
     * 内部使用するマップの参照を返却します。
     * 
     * @return マップ
     */
    protected Map getSourceMap() {
        return map;
    }

    /*
     * classes
     */

    /* 値コレクションビューのシンプル実装を提供します。 */
    class Values extends AbstractView {
        Values() {
            super(map.values());
        }
    }

    /* キーセットビュー実装のシンプル実装を提供します。 */
    class KeySet extends AbstractView implements Set {
        KeySet() {
            super(map.keySet());
        }
    }

    /* エントリセットビュー実装のシンプル実装を提供します。 */
    class EntrySet extends AbstractView implements Set {
        EntrySet() {
            super(map.entrySet());
        }
    }

    /* ビューの基底実装を提供します。 */
    abstract class AbstractView implements Collection {

        private final Collection src;

        AbstractView(Collection c) {
            src = c;
        }

        public Iterator iterator() {
            synchronized (mutex) {
                // Must be manually synched by user!
                return src.iterator();
            }
        }

        public int size() {
            synchronized (mutex) {
                return src.size();
            }
        }

        public boolean isEmpty() {
            synchronized (mutex) {
                return src.isEmpty();
            }
        }

        public boolean contains(Object o) {
            synchronized (mutex) {
                return src.contains(o);
            }
        }

        public boolean containsAll(Collection c) {
            synchronized (mutex) {
                return src.containsAll(c);
            }
        }

        public boolean remove(Object o) {
            synchronized (mutex) {
                return src.remove(o);
            }
        }

        public boolean removeAll(Collection c) {
            synchronized (mutex) {
                return src.removeAll(c);
            }
        }

        public boolean retainAll(Collection c) {
            synchronized (mutex) {
                return src.retainAll(c);
            }
        }

        public void clear() {
            synchronized (mutex) {
                src.clear();
            }
        }

        public Object[] toArray() {
            synchronized (mutex) {
                return src.toArray();
            }
        }

        public Object[] toArray(Object a[]) {
            synchronized (mutex) {
                return src.toArray(a);
            }
        }

        public boolean add(Object o) {
            throw new UnsupportedOperationException();
        }

        public boolean addAll(Collection c) {
            throw new UnsupportedOperationException();
        }

    }

}
