/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import shohaku.core.collections.decorator.DecoratedMap;

/**
 * Map インタフェースのユーティリティメソッドを提供します。
 */
public class MapUtils {

    /**
     * 実装型に固有の機能へのアクセスを制限したマップを返却します。
     * 
     * @param map
     *            ラップするマップ
     * @return 実装を装飾したマップ
     */
    public static Map unextendsMap(Map map) {
        return new DecoratedMap(map);
    }

    /**
     * 第一引数のキーと接頭辞の一致するエントリを第三引数に追加して返却します。<br>
     * キーは文字列型である必要があります。
     * 
     * @param src
     *            取得基となるマップ
     * @param prefix
     *            検証するキーの接頭辞
     * @param dest
     *            追加先
     * @return 第三引数の参照
     */
    public static Map putPrefixKey(Map src, String prefix, Map dest) {
        if (prefix == null) {
            throw new NullPointerException();
        }
        final Set entrys = src.entrySet();
        final Iterator i = entrys.iterator();
        while (i.hasNext()) {
            Map.Entry e = (Map.Entry) i.next();
            String name = (String) e.getKey();
            if (name.startsWith(prefix)) {
                dest.put(name, e.getValue());
            }
        }
        return dest;
    }

    /**
     * 第一引数のキーと接頭辞の一致するエントリを削除し第三引数に追加して返却します。<br>
     * キーは文字列型である必要があります。
     * 
     * @param src
     *            削除基となるマップ
     * @param prefix
     *            検証するキーの接頭辞
     * @param dest
     *            削除エントリの追加先
     * @return 第三引数の参照
     */
    public static Map removePrefixKey(Map src, String prefix, Map dest) {
        if (prefix == null) {
            throw new NullPointerException();
        }
        final Object[] keys = src.keySet().toArray();
        for (int i = 0; i < keys.length; i++) {
            final String name = (String) keys[i];
            if (name.startsWith(prefix)) {
                dest.put(name, src.remove(name));
            }
        }
        return dest;
    }

}
