/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import shohaku.core.helpers.HHash;
import shohaku.core.lang.Eval;

/**
 * MapBox インタフェースのユーティリティメソッドを提供します。
 */
public class MapBoxUtils {

    /*
     * class
     */

    /* 単一のオブジェクトを格納するマップボックスを提供します。 */
    static class ObjectMapBox implements MapBox {
        private final Object key;

        private Object value;

        ObjectMapBox(Object k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            return Eval.isEquals(key, v);
        }
    }

    /* equals(Object) で同値性検証を行うオブジェクト配列を格納するマップボックスを提供します。 */
    static class ObjectArrayMapBox implements MapBox {
        private final Object[] key;

        private Object value;

        ObjectArrayMapBox(Object[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* 単一のオブジェクトを格納するマップボックスを提供します。 */
    static class IdentityMapBox implements MapBox {
        private final Object key;

        private Object value;

        IdentityMapBox(Object k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return System.identityHashCode(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            return (key == v);
        }
    }

    /* 参照一致で同値性検証を行うオブジェクト配列を格納するマップボックスを提供します。 */
    static class IdentityObjectArrayMapBox implements MapBox {
        private final Object[] key;

        private Object value;

        IdentityObjectArrayMapBox(Object[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.identityHash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            return Eval.isRefEquals(key, a);
        }
    }

    /* byte 配列を格納するマップボックスを提供します。 */
    static class ByteArrayMapBox implements MapBox {
        private final byte[] key;

        private Object value;

        ByteArrayMapBox(byte[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof byte[])) {
                return false;
            }
            byte[] a = (byte[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* short 配列を格納するマップボックスを提供します。 */
    static class ShortArrayMapBox implements MapBox {
        private final short[] key;

        private Object value;

        ShortArrayMapBox(short[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof short[])) {
                return false;
            }
            short[] a = (short[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* int 配列を格納するマップボックスを提供します。 */
    static class IntArrayMapBox implements MapBox {
        private final int[] key;

        private Object value;

        IntArrayMapBox(int[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof int[])) {
                return false;
            }
            int[] a = (int[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* long 配列を格納するマップボックスを提供します。 */
    static class LongArrayMapBox implements MapBox {
        private final long[] key;

        private Object value;

        LongArrayMapBox(long[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof long[])) {
                return false;
            }
            long[] a = (long[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* float 配列を格納するマップボックスを提供します。 */
    static class FloatArrayMapBox implements MapBox {
        private final float[] key;

        private Object value;

        FloatArrayMapBox(float[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof float[])) {
                return false;
            }
            float[] a = (float[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* double 配列を格納するマップボックスを提供します。 */
    static class DoubleArrayMapBox implements MapBox {
        private final double[] key;

        private Object value;

        DoubleArrayMapBox(double[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof double[])) {
                return false;
            }
            double[] a = (double[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /* boolean 配列を格納するマップボックスを提供します。 */
    static class BooleanArrayMapBox implements MapBox {
        private final boolean[] key;

        private Object value;

        BooleanArrayMapBox(boolean[] k, Object v) {
            this.key = k;
            this.value = v;
        }

        public Object getKey() {
            return key;
        }

        public Object setValue(Object v) {
            return this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(key);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof MapBox)) {
                return false;
            }
            MapBox e = (MapBox) o;
            Object v = e.getKey();
            if (key == v) {
                return true;
            }
            if (v == null || !(v instanceof boolean[])) {
                return false;
            }
            boolean[] a = (boolean[]) v;
            return Eval.isEquals(key, a);
        }
    }

    /*
     * factory
     */

    /**
     * 単一のオブジェクトのキーと値を格納するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() はキー k の (k == null) ? 0 : k.hashCode() となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、<br>
     * キー k が引数のキー k2 と (k == k2 || (k != null && k.equals(k2))) であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 単一のオブジェクトのマップボックス
     */
    public static MapBox mapBox(Object key, Object value) {
        return new ObjectMapBox(key, value);
    }

    /**
     * オブジェクト配列のキーと値を格納するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() はキーが null の場合は 0 、以外はキーの要素 e の (e == null) ? 0 : e.hashCode() の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、<br>
     * 引数のキーがオブジェクト配列であること、キーの要素 e と引数のキーの要素 e2 が全て (e == e2 || (e != null && e.equals(e2))) であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return オブジェクト配列のマップボックス
     */
    public static MapBox arrayMapBox(Object[] key, Object value) {
        return new ObjectArrayMapBox(key, value);
    }

    /**
     * 単一のオブジェクトのキーを参照同一性で比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() はキー k の (k == null) ? 0 : System.identityHashCode(k) となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キー k が引数のキー k2 と (k == k2) であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 単一のオブジェクトのマップボックス
     */
    public static MapBox identityMapBox(Object key, Object value) {
        return new IdentityMapBox(key, value);
    }

    /**
     * オブジェクト配列のキーの全要素を参照同一性で比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() はキーが null の場合は 0 、以外はキーの要素 e の (e == null) ? 0 : System.identityHashCode(e) の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、 <br>
     * 引数のキーがオブジェクト配列であること、キーの要素 e と引数のキーの要素 e2 が全て (e == e2) であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return オブジェクト配列のマップボックス
     */
    public static MapBox identityArrayMapBox(Object[] key, Object value) {
        return new IdentityObjectArrayMapBox(key, value);
    }

    /**
     * オブジェクト配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Byte.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが byte 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(byte[] key, Object value) {
        return new ByteArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Short.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが short 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(short[] key, Object value) {
        return new ShortArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Integer.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが int 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(int[] key, Object value) {
        return new IntArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Long.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが long 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(long[] key, Object value) {
        return new LongArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Float.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが float 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(float[] key, Object value) {
        return new FloatArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Double.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが double 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(double[] key, Object value) {
        return new DoubleArrayMapBox(key, value);
    }

    /**
     * 配列の全要素を比較するマップボックスを生成して返却します。 <br>
     * このマップボックスの hashCode() は Boolean.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このマップボックスの equals(Object) は引数が MapBox の実装クラスであり、キーが boolean 配列であること、全要素が同値であることを条件とします。
     * 
     * @param key
     *            キー
     * @param value
     *            格納する初期値
     * @return 配列のマップボックス
     */
    public static MapBox arrayMapBox(boolean[] key, Object value) {
        return new BooleanArrayMapBox(key, value);
    }

}
