/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import shohaku.core.helpers.HHash;
import shohaku.core.lang.Eval;

/**
 * Box インタフェースのユーティリティメソッドを提供します。
 */
public class BoxUtils {

    /*
     * class
     */

    /* 単一のオブジェクトを格納するボックスを提供します。 */
    static class ObjectBox implements Box {
        private final Object value;

        ObjectBox(Object v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            return Eval.isEquals(value, v);
        }
    }

    /* equals(Object) で同値性検証を行うオブジェクト配列を格納するボックスを提供します。 */
    static class ObjectArrayBox implements Box {
        private final Object[] value;

        ObjectArrayBox(Object[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* 単一のオブジェクトを格納するボックスを提供します。 */
    static class IdentityBox implements Box {
        private final Object value;

        IdentityBox(Object v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return System.identityHashCode(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            return (value == v);
        }
    }

    /* 参照一致で同値性検証を行うオブジェクト配列を格納するボックスを提供します。 */
    static class IdentityObjectArrayBox implements Box {
        private final Object[] value;

        IdentityObjectArrayBox(Object[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.identityHash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof Object[])) {
                return false;
            }
            Object[] a = (Object[]) v;
            return Eval.isRefEquals(value, a);
        }
    }

    /* byte 配列を格納するボックスを提供します。 */
    static class ByteArrayBox implements Box {
        private final byte[] value;

        ByteArrayBox(byte[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof byte[])) {
                return false;
            }
            byte[] a = (byte[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* short 配列を格納するボックスを提供します。 */
    static class ShortArrayBox implements Box {
        private final short[] value;

        ShortArrayBox(short[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof short[])) {
                return false;
            }
            short[] a = (short[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* int 配列を格納するボックスを提供します。 */
    static class IntArrayBox implements Box {
        private final int[] value;

        IntArrayBox(int[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof int[])) {
                return false;
            }
            int[] a = (int[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* long 配列を格納するボックスを提供します。 */
    static class LongArrayBox implements Box {
        private final long[] value;

        LongArrayBox(long[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof long[])) {
                return false;
            }
            long[] a = (long[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* float 配列を格納するボックスを提供します。 */
    static class FloatArrayBox implements Box {
        private final float[] value;

        FloatArrayBox(float[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof float[])) {
                return false;
            }
            float[] a = (float[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* double 配列を格納するボックスを提供します。 */
    static class DoubleArrayBox implements Box {
        private final double[] value;

        DoubleArrayBox(double[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof double[])) {
                return false;
            }
            double[] a = (double[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /* boolean 配列を格納するボックスを提供します。 */
    static class BooleanArrayBox implements Box {
        private final boolean[] value;

        BooleanArrayBox(boolean[] v) {
            this.value = v;
        }

        public Object getValue() {
            return value;
        }

        public int hashCode() {
            return HHash.hash(value);
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || !(o instanceof Box)) {
                return false;
            }
            Box e = (Box) o;
            Object v = e.getValue();
            if (value == v) {
                return true;
            }
            if (v == null || !(v instanceof boolean[])) {
                return false;
            }
            boolean[] a = (boolean[]) v;
            return Eval.isEquals(value, a);
        }
    }

    /*
     * factory
     */

    /**
     * 単一のオブジェクトを格納するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は値 v の (v == null) ? 0 : v.hashCode() となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、<br>
     * 値 v が引数 v2 と (v == v2 || (v != null && v.equals(v2))) であることを条件とします。
     * 
     * @param value
     *            登録するオブジェクト
     * @return 単一のオブジェクトのボックス
     */
    public static Box box(Object value) {
        return new ObjectBox(value);
    }

    /**
     * オブジェクト配列の値を格納するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は値が null の場合は 0 、以外は値の要素 e の (e == null) ? 0 : e.hashCode() の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、その値がオブジェクト配列であること、<br>
     * 値の要素 e と引数の値の要素 e2 が全て (e == e2 || (e != null && e.equals(e2))) であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return オブジェクト配列のボックス
     */
    public static Box arrayBox(Object[] value) {
        return new ObjectArrayBox(value);
    }

    /**
     * 単一のオブジェクトを参照同一性で比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は値 v の (v == null) ? 0 : System.identityHashCode(v) となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値 v が引数 v2 と (v == v2) であることを条件とします。
     * 
     * @param value
     *            登録するオブジェクト
     * @return 単一のオブジェクトのボックス
     */
    public static Box identityBox(Object value) {
        return new IdentityBox(value);
    }

    /**
     * オブジェクト配列の全要素を参照同一性で比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は値が null の場合は 0 、以外は値の要素 e の (e == null) ? 0 : System.identityHashCode(e) の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、その値がオブジェクト配列であること、<br>
     * 値の要素 e と引数の値の要素 e2 が全て (e == e2) であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return オブジェクト配列のボックス
     */
    public static Box identityArrayBox(Object[] value) {
        return new IdentityObjectArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Byte.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が byte 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(byte[] value) {
        return new ByteArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Short.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が short 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(short[] value) {
        return new ShortArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Integer.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が int 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(int[] value) {
        return new IntArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Long.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が long 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(long[] value) {
        return new LongArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Float.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が float 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(float[] value) {
        return new FloatArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Double.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が double 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(double[] value) {
        return new DoubleArrayBox(value);
    }

    /**
     * 配列の全要素を比較するボックスを生成して返却します。 <br>
     * このボックスの hashCode() は Boolean.hashCode() と同基準での全配列要素の総和となります。 <br>
     * このボックスの equals(Object) は引数が Box の実装クラスであり、値が boolean 配列であること、全要素が同値であることを条件とします。
     * 
     * @param value
     *            登録する配列
     * @return 配列のボックス
     */
    public static Box arrayBox(boolean[] value) {
        return new BooleanArrayBox(value);
    }

}
