/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;

import shohaku.core.functor.FTransformer;
import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HClass;
import shohaku.core.helpers.HLog;
import shohaku.core.lang.feature.FeatureFactory;
import shohaku.core.lang.feature.LogFeature;

/**
 * TypeTransformer のデフォルト実装を提供します。<br>
 * この実装では、識別子をオブジェクトクラスで統一的に扱う為、プリミティブ型は対応するラッパクラスに変換して登録、削除、検索されます。
 */
public class DefaultTypeTransformer implements TypeTransformer {

    /** shohaku core logging. */
    private final LogFeature log = FeatureFactory.getLog(TypeTransformer.class);

    /** エントリテーブル */
    private final Map table = Collections.synchronizedMap(new LinkedHashMap());

    /**
     * 空のエントリで初期化します。
     */
    public DefaultTypeTransformer() {
        init();
    }

    /**
     * クラスを識別子として変換ファンクタを追加します。<br>
     * 既にクラスが登録されている場合は変換ファンクタを上書きします。
     * 
     * @param clazz
     *            クラス
     * @param transformer
     *            変換ファンクタ
     * @return 上書きされた場合は既存の変換ファンクタ、以外は null
     */
    public FTransformer add(Class clazz, FTransformer transformer) {
        if (log.isTraceEnabled()) {
            log.trace(HLog.list("add transformer. ", clazz, transformer));
        }
        return (FTransformer) table.put(entryClass(clazz), transformer);
    }

    /**
     * 登録されているエントリをクリアします。
     */
    public void clear() {
        table.clear();
    }

    /**
     * クラスに対応するエントリを削除します。
     * 
     * @param clazz
     *            クラス
     * @return 実際に削除された場合は削除された変換ファンクタ、以外は null
     */
    public FTransformer remove(Class clazz) {
        if (log.isTraceEnabled()) {
            log.trace(HLog.list("remove transformer. ", clazz));
        }
        return (FTransformer) table.remove(entryClass(clazz));
    }

    /*
     * implements
     */

    public int size() {
        return table.size();
    }

    public Iterator getTypes() {
        // Must be manually synched by user!
        return table.keySet().iterator();
    }

    public boolean contains(Class clazz) {
        return table.containsKey(entryClass(clazz));
    }

    public FTransformer get(Class clazz) {
        return (FTransformer) table.get(entryClass(clazz));
    }

    public FTransformer find(Class clazz) {
        return findTransformer(clazz);
    }

    public Object transform(Class clazz, Object o) {
        return transform(clazz, o, Collections.EMPTY_MAP);
    }

    public Object transform(Class clazz, Object o, Map args) {

        if (log.isTraceEnabled()) {
            log.trace(HLog.list("transform. ", clazz, o, args));
        }

        FTransformer t = find(clazz);
        if (t == null) {
            throw new IllegalArgumentException("not find class. " + clazz);
        }
        return t.transform(o, args);
    }

    /*
     * protected
     */

    /**
     * コンストラクタから呼ばれる初期化のフックポイントです。
     */
    protected void init() {
        // np op
    }

    /**
     * 指定のクラスから推測される変換ファンクタを検索して返却します。<br>
     * 推測される変換ファンクタが発見出来ない場合は null が返ります。<br>
     * 継承クラスの拡張ポイントです。
     * 
     * @param clazz
     *            クラス
     * @return 指定のクラスから推測される変換ファンクタ
     */
    protected FTransformer findTransformer(Class clazz) {

        if (log.isTraceEnabled()) {
            log.trace(HLog.list("find transformer. ", clazz));
        }

        FTransformer t = (FTransformer) table.get(entryClass(clazz));
        if (t == null) {
            for (Iterator i = table.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                Class fromType = (Class) e.getKey();
                if (HBeans.isAssignmentCompatible(fromType, clazz)) {
                    t = (FTransformer) e.getValue();
                }
            }
        }
        return t;
    }

    /**
     * 指定のクラスから実際に登録するクラスを返却します。<br>
     * デフォルトでは指定のクラスがプリミティブ型の場合はラッパクラスを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 実際に登録するクラス
     */
    protected Class entryClass(Class clazz) {
        return HClass.box(clazz);
    }

    /**
     * TypeTransformer.class を識別名とするログを返却します。
     * 
     * @return ログ
     */
    protected LogFeature getLog() {
        return log;
    }

    /**
     * クラスをキーとして変換ファンクタを保管するマップを返却します。
     * 
     * @return マップ
     */
    protected Map getTable() {
        return this.table;
    }

}
