/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * 単項演算子の構文解析機能を提供します。
 */
class OgdlUnaryOperatorParser {

    /**
     * ビット否定演算を実行して返却します。
     * 
     * @param o
     *            演算元のオブジェクト
     * @return 演算結果のオブジェクト
     * @throws OgdlSyntaxException
     *             引数が演算可能な型以外の場合
     */
    static Object evaluateBitwiseComplement(OgdlEvent ev) throws OgdlSyntaxException {

        ev.off.increment();
        final Object o = OgdlRuntime.evaluate(ev);
        if (o instanceof Byte) {
            byte num = ((Byte) o).byteValue();
            return Boxing.box(~num);
        }
        if (o instanceof Short) {
            short num = ((Short) o).shortValue();
            return Boxing.box(~num);
        }
        if (o instanceof Integer) {
            int num = ((Integer) o).intValue();
            return Boxing.box(~num);
        }
        if (o instanceof Long) {
            long num = ((Long) o).longValue();
            return Boxing.box(~num);
        }
        if (o instanceof Float) {
            throw new OgdlSyntaxException(ev, HLog.log("value is not integer. ", o));
        }
        if (o instanceof Double) {
            throw new OgdlSyntaxException(ev, HLog.log("value is not integer. ", o));
        }
        if (o instanceof BigInteger) {
            BigInteger num = ((BigInteger) o);
            return num.not();
        }
        if (o instanceof BigDecimal) {
            throw new OgdlSyntaxException(ev, HLog.log("value is not integer. ", o));
        }
        throw new OgdlSyntaxException(ev, HLog.log("value is not Number. ", o));

    }

    /**
     * 論理否定演算を実行して返却します。
     * 
     * @param o
     *            演算元のオブジェクト
     * @return 演算結果のオブジェクト
     * @throws OgdlSyntaxException
     *             引数が演算可能な型以外の場合
     */
    static Object evaluateLogicalComplement(OgdlEvent ev) throws OgdlSyntaxException {

        ev.off.increment();
        final Object o = OgdlRuntime.evaluate(ev);
        if (!(o instanceof Boolean)) {
            throw new OgdlSyntaxException(ev, HLog.log("value is not Boolean. ", o));
        }
        return (Boolean.TRUE.equals(o)) ? Boolean.FALSE : Boolean.TRUE;

    }

    /**
     * 単項マイナス演算を実行して返却します。
     * 
     * @param o
     *            演算元のオブジェクト
     * @return 演算結果のオブジェクト
     * @throws OgdlSyntaxException
     *             引数が演算可能な型以外の場合
     */
    static Object evaluateUnaryMinus(OgdlEvent ev) throws OgdlSyntaxException {

        ev.off.increment();
        final Object o = OgdlRuntime.evaluate(ev);
        if (o instanceof Byte) {
            byte num = ((Byte) o).byteValue();
            return Boxing.box((byte) -num);
        }
        if (o instanceof Short) {
            short num = ((Short) o).shortValue();
            return Boxing.box((short) -num);
        }
        if (o instanceof Integer) {
            int num = ((Integer) o).intValue();
            return Boxing.box(-num);
        }
        if (o instanceof Long) {
            long num = ((Long) o).longValue();
            return Boxing.box(-num);
        }
        if (o instanceof Float) {
            float num = ((Float) o).floatValue();
            return Boxing.box(-num);
        }
        if (o instanceof Double) {
            double num = ((Double) o).doubleValue();
            return Boxing.box(-num);
        }
        if (o instanceof BigInteger) {
            BigInteger num = ((BigInteger) o);
            return num.negate();
        }
        if (o instanceof BigDecimal) {
            BigDecimal num = ((BigDecimal) o);
            return num.negate();
        }
        throw new OgdlSyntaxException(ev, HLog.log("value is not Number. ", o));

    }

}
