/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

package shohaku.ogdl;

/**
 * 構文解析の共通的な機能を提供します。
 */
class OgdlSyntax {

    static final int UPPER = 0x00000001;

    static final int LOWER = 0x00000002;

    static final int SPACE = 0x00000004;

    static final int PUNCT = 0x00000008;

    static final int CNTRL = 0x00000010;

    static final int DIGIT = 0x00000020;

    static final int HEX = 0x00000040;

    static final int OCTAL = 0x00000080;

    static final int FLOAT = 0x00000100;

    // static final int HEXFLOAT = 0x00000200; // java 1.5 [-+]? 0[xX][0-9a-fA-F]*\.[0-9a-fA-F]+ ([pP][-+]?[0-9]+)?

    static final int UNDER = 0x00100000;

    static final int DOT = 0x00200000;

    static final int DOLLAR = 0x00400000;

    static final int COLON = 0x00800000;

    static final int MINUS = 0x01000000;

    static final int ISOTIME = 0x02000000;

    static final int ALPHA = (UPPER | LOWER);

    static final int ALNUM = (UPPER | LOWER | DIGIT);

    static final int GRAPH = (PUNCT | UPPER | LOWER | DIGIT);

    static final int MEMBER = (UPPER | LOWER | DIGIT | UNDER);

    static final int MEMBER_BEGIN = (UPPER | LOWER | UNDER);

    static final int CLAZZ = (UPPER | LOWER | DIGIT | UNDER | DOT | DOLLAR);

    static final int NUMBER = (DIGIT | HEX | OCTAL | FLOAT);

    static final int BLANK = (CNTRL | SPACE);

    static final int FUNC = (UPPER | LOWER | DIGIT | UNDER | COLON);

    static final int OPERATOR = (PUNCT | LOWER);

    static final int DIFFERENCE = (UPPER | LOWER | UNDER | DIGIT | HEX | OCTAL | FLOAT | MINUS);

    static final int DATETIME = (DIGIT | COLON | DOT | MINUS | ISOTIME);

    private static final int[] GROUP = new int[0xFF];
    static {
        GROUP[0x00] = CNTRL; // (NUL)
        GROUP[0x01] = CNTRL; // (SOH)
        GROUP[0x02] = CNTRL; // (STX)
        GROUP[0x03] = CNTRL; // (ETX)
        GROUP[0x04] = CNTRL; // (EOT)
        GROUP[0x05] = CNTRL; // (ENQ)
        GROUP[0x06] = CNTRL; // (ACK)
        GROUP[0x07] = CNTRL; // (BEL)
        GROUP[0x08] = CNTRL; // (BS)
        GROUP[0x09] = SPACE + CNTRL; // (HT)
        GROUP[0x0A] = SPACE + CNTRL; // (LF)
        GROUP[0x0B] = SPACE + CNTRL; // (VT)
        GROUP[0x0C] = SPACE + CNTRL; // (FF)
        GROUP[0x0D] = SPACE + CNTRL; // (CR)
        GROUP[0x0E] = CNTRL; // (SI)
        GROUP[0x0F] = CNTRL; // (SO)
        GROUP[0x10] = CNTRL; // (DLE)
        GROUP[0x11] = CNTRL; // (DC1)
        GROUP[0x12] = CNTRL; // (DC2)
        GROUP[0x13] = CNTRL; // (DC3)
        GROUP[0x14] = CNTRL; // (DC4)
        GROUP[0x15] = CNTRL; // (NAK)
        GROUP[0x16] = CNTRL; // (SYN)
        GROUP[0x17] = CNTRL; // (ETB)
        GROUP[0x18] = CNTRL; // (CAN)
        GROUP[0x19] = CNTRL; // (EM)
        GROUP[0x1A] = CNTRL; // (SUB)
        GROUP[0x1B] = CNTRL; // (ESC)
        GROUP[0x1C] = CNTRL; // (FS)
        GROUP[0x1D] = CNTRL; // (GS)
        GROUP[0x1E] = CNTRL; // (RS)
        GROUP[0x1F] = CNTRL; // (US)
        GROUP[0x20] = SPACE; // SPACE
        GROUP[0x21] = PUNCT; // !
        GROUP[0x22] = PUNCT; // "
        GROUP[0x23] = PUNCT; // #
        GROUP[0x24] = PUNCT + DOLLAR; // $
        GROUP[0x25] = PUNCT; // %
        GROUP[0x26] = PUNCT; // &
        GROUP[0x27] = PUNCT; // '
        GROUP[0x28] = PUNCT; // (
        GROUP[0x29] = PUNCT; // )
        GROUP[0x2A] = PUNCT; // *
        GROUP[0x2B] = PUNCT + FLOAT; // +
        GROUP[0x2C] = PUNCT; // ,
        GROUP[0x2D] = PUNCT + FLOAT + MINUS; // -
        GROUP[0x2E] = PUNCT + FLOAT + DOT; // .
        GROUP[0x2F] = PUNCT; // /
        GROUP[0x30] = DIGIT + FLOAT + HEX + OCTAL; // 0
        GROUP[0x31] = DIGIT + FLOAT + HEX + OCTAL; // 1
        GROUP[0x32] = DIGIT + FLOAT + HEX + OCTAL; // 2
        GROUP[0x33] = DIGIT + FLOAT + HEX + OCTAL; // 3
        GROUP[0x34] = DIGIT + FLOAT + HEX + OCTAL; // 4
        GROUP[0x35] = DIGIT + FLOAT + HEX + OCTAL; // 5
        GROUP[0x36] = DIGIT + FLOAT + HEX + OCTAL; // 6
        GROUP[0x37] = DIGIT + FLOAT + HEX + OCTAL; // 7
        GROUP[0x38] = DIGIT + FLOAT + HEX; // 8
        GROUP[0x39] = DIGIT + FLOAT + HEX; // 9
        GROUP[0x3A] = PUNCT + COLON; // :
        GROUP[0x3B] = PUNCT; // ;
        GROUP[0x3C] = PUNCT; // <
        GROUP[0x3D] = PUNCT; // =
        GROUP[0x3E] = PUNCT; // >
        GROUP[0x3F] = PUNCT; // ?
        GROUP[0x40] = PUNCT; // @
        GROUP[0x41] = UPPER + HEX; // A
        GROUP[0x42] = UPPER + HEX; // B
        GROUP[0x43] = UPPER + HEX; // C
        GROUP[0x44] = UPPER + HEX; // D
        GROUP[0x45] = UPPER + HEX + FLOAT; // E
        GROUP[0x46] = UPPER + HEX; // F
        GROUP[0x47] = UPPER; // G
        GROUP[0x48] = UPPER; // H
        GROUP[0x49] = UPPER; // I
        GROUP[0x4A] = UPPER; // J
        GROUP[0x4B] = UPPER; // K
        GROUP[0x4C] = UPPER; // L
        GROUP[0x4D] = UPPER; // M
        GROUP[0x4E] = UPPER; // N
        GROUP[0x4F] = UPPER; // O
        GROUP[0x50] = UPPER; // P
        GROUP[0x51] = UPPER; // Q
        GROUP[0x52] = UPPER; // R
        GROUP[0x53] = UPPER; // S
        GROUP[0x54] = UPPER + ISOTIME; // T
        GROUP[0x55] = UPPER; // U
        GROUP[0x56] = UPPER; // V
        GROUP[0x57] = UPPER; // W
        GROUP[0x58] = UPPER + HEX; // X
        GROUP[0x59] = UPPER; // Y
        GROUP[0x5A] = UPPER; // Z
        GROUP[0x5B] = PUNCT; // [
        GROUP[0x5C] = PUNCT; // \
        GROUP[0x5D] = PUNCT; // ]
        GROUP[0x5E] = PUNCT; // ^
        GROUP[0x5F] = PUNCT + UNDER; // _
        GROUP[0x60] = PUNCT; // `
        GROUP[0x61] = LOWER + HEX; // a
        GROUP[0x62] = LOWER + HEX; // b
        GROUP[0x63] = LOWER + HEX; // c
        GROUP[0x64] = LOWER + HEX; // d
        GROUP[0x65] = LOWER + HEX + FLOAT; // e
        GROUP[0x66] = LOWER + HEX; // f
        GROUP[0x67] = LOWER; // g
        GROUP[0x68] = LOWER; // h
        GROUP[0x69] = LOWER; // i
        GROUP[0x6A] = LOWER; // j
        GROUP[0x6B] = LOWER; // k
        GROUP[0x6C] = LOWER; // l
        GROUP[0x6D] = LOWER; // m
        GROUP[0x6E] = LOWER; // n
        GROUP[0x6F] = LOWER; // o
        GROUP[0x70] = LOWER; // p
        GROUP[0x71] = LOWER; // q
        GROUP[0x72] = LOWER; // r
        GROUP[0x73] = LOWER; // s
        GROUP[0x74] = LOWER; // t
        GROUP[0x75] = LOWER; // u
        GROUP[0x76] = LOWER; // v
        GROUP[0x77] = LOWER; // w
        GROUP[0x78] = LOWER + HEX; // x
        GROUP[0x79] = LOWER; // y
        GROUP[0x7A] = LOWER; // z
        GROUP[0x7B] = PUNCT; // {
        GROUP[0x7C] = PUNCT; // |
        GROUP[0x7D] = PUNCT; // }
        GROUP[0x7E] = PUNCT; // ~
        GROUP[0x7F] = CNTRL; // (DEL)
    }

    static int getGroup(int ch) {
        return isAscii(ch) ? GROUP[ch] : 0;
    }

    static boolean isGroup(int ch, int type) {
        return (getGroup(ch) & type) != 0;
    }

    static boolean isAscii(int ch) {
        return ((ch & 0xFFFFFF80) == 0);
    }

    static boolean isGraph(int ch) {
        return isGroup(ch, GRAPH);
    }

    static boolean isPunct(int ch) {
        return isGroup(ch, PUNCT);
    }

    static boolean isSpace(int ch) {
        return isGroup(ch, SPACE);
    }

    static boolean isDigit(int ch) {
        return isGroup(ch, DIGIT);
    }

    static boolean isHexDigit(int ch) {
        return isGroup(ch, HEX);
    }

    static boolean isOctalDigits(int ch) {
        return isGroup(ch, OCTAL);
    }

    static boolean isFloatingPoint(int ch) {
        return isGroup(ch, FLOAT);
    }

    static boolean isCntrl(int ch) {
        return isGroup(ch, CNTRL);
    }

    static boolean isLower(int ch) {
        return isGroup(ch, LOWER);
    }

    static boolean isUpper(int ch) {
        return isGroup(ch, UPPER);
    }

    static boolean isMenber(int ch) {
        return isGroup(ch, MEMBER);
    }

    static boolean isClazz(int ch) {
        return isGroup(ch, CLAZZ);
    }

    static boolean isNumber(int ch) {
        return isGroup(ch, NUMBER);
    }

    static boolean isBlank(int ch) {
        return isGroup(ch, BLANK);
    }

    static int toLower(int ch) {
        return isUpper(ch) ? (ch + 0x20) : ch;
    }

    static int toUpper(int ch) {
        return isLower(ch) ? (ch - 0x20) : ch;
    }

    static int skip(CharSequence cs, int group, int index) {
        int i = index;
        while (i < cs.length()) {
            char ch = cs.charAt(i);
            if (!isGroup(ch, group)) {
                break;
            }
            i++;
        }
        return i;
    }

    /**
     * クラス名の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipExtendNameClass(CharSequence cs, int index) {
        int i = skip(cs, CLAZZ, index);
        return skip(cs, "[]", i);
    }

    /**
     * Javaのメンバ (フィールドやメソッド、コンストラクタ)の識別名の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipMemberName(CharSequence cs, int index) {
        return skip(cs, MEMBER, index);
    }

    /**
     * 演算子の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipOperator(CharSequence cs, int index) {
        int i = skip(cs, OPERATOR, index);
        if (cs.length() > i && isSpace(cs, i)) {
            return i;
        }
        return index;
    }

    /**
     * 組込み関数の完全名の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipFunctionName(CharSequence cs, int index) {
        return skip(cs, FUNC, index);
    }

    /**
     * 数値文字列の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipNumberString(CharSequence cs, int index) {
        int i = skip(cs, NUMBER, index);
        if (cs.length() > i && isNumberSuffix(cs, i)) {
            i++;
        }
        return i;
    }

    /**
     * １６進数値文字列の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipHexString(CharSequence cs, int index) {
        return skip(cs, HEX, index);
    }

    /**
     * １６進数値文字列の終了インデックスを解析して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終了インデックス
     */
    static int skipDateTimeString(CharSequence cs, int index) {
        return skip(cs, DATETIME, index);
    }

    /**
     * 開始文字に囲まれた文字列として終端インデックスを計算して返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 終端インデックス @ 構文例外
     */
    static int skipEncloseString(OgdlEvent ev, CharSequence cs, int index) {

        int off = index;
        validIndex(ev, cs, off);
        final char close = getEncloseCloseChar(cs, off);
        validIndex(ev, cs, ++off);

        final int length = cs.length();
        while (off < length) {
            off = indexOf(cs, off, close);
            if (-1 == off) {
                throw new OgdlSyntaxException(ev, HLog.log("not find close char.", cs, close));
            }
            if (isCharAt(cs, off + 1, close)) {
                off++;
            } else {
                break;
            }
            off++;
        }
        return ++off;// ++ (next index)

    }

    /**
     * 数値領域（等差数列や時間領域）式の文字列を返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 数値領域（等差数列や時間領域）式の文字列
     */
    static String cutDifferenceString(String s, OgdlParseIndex index) {
        int start = index.get();
        int end = skip(s, DIFFERENCE, start);
        index.set(end);
        return s.substring(start, end);
    }

    /**
     * isEncloseChar() が true を返す場合は開始文字に囲まれた文字列として、以外は Java のメンバーとして切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 切り出した文字列 @ 構文例外
     */
    static String cutEncloseOrNumberString(OgdlEvent ev, String s, OgdlParseIndex index) {
        if (isEncloseOpenChar(s, index.get())) {
            return cutEncloseString(ev, s, index);
        } else {
            return cutMemberName(s, index);
        }
    }

    /**
     * 開始文字に囲まれた文字列として文字列を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 切り出した文字列 @ 構文例外
     */
    static String cutEncloseString(OgdlEvent ev, String s, OgdlParseIndex index) {

        int offset = index.get();
        validIndex(ev, s, offset);
        final char close = getEncloseCloseChar(s, offset);
        validIndex(ev, s, ++offset);
        final StringBuffer buff = new StringBuffer(s.substring(offset));

        int off = 0;
        int delete = 0;
        while (off < buff.length()) {
            off = indexOf(buff, off, close);
            if (-1 == off) {
                throw new OgdlSyntaxException(ev, HLog.log("not find close char.", s, close));
            }
            if (isCharAt(buff, off + 1, close)) {
                delete++;
                buff.deleteCharAt(off);// delete '\\'
            } else {
                break;
            }
            off++;
        }
        String value = buff.substring(0, off);
        int nextIndex = index.increment() + (++off) + delete;// ++ (next index)
        index.set(nextIndex);
        return value;
    }

    /**
     * クラス名を解析して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return クラス名
     */
    static String cutExtendClassName(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipExtendNameClass(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * Javaのメンバ (フィールドやメソッド、コンストラクタ)の識別名を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return Javaのメンバの識別名
     */
    static String cutMemberName(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipMemberName(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * １６進数値文字列を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return １６進数値文字列
     */
    static String cutHexString(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipHexString(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * 日付時刻型の文字列を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 日付時刻型の文字列
     */
    static String cutDateTimeString(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipDateTimeString(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * 組込み関数の完全名を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 組込み関数の完全名
     */
    static String cutFunctionName(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipFunctionName(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * 数値文字列を切り出して返却します。
     * 
     * @param s
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 数値文字列
     */
    static String cutNumberString(String s, OgdlParseIndex index) {
        final int off = index.get();
        int nextIndex = skipNumberString(s, off);
        String value = s.substring(off, nextIndex);
        index.set(nextIndex);
        return value;
    }

    /**
     * クラス名として認識出来るか検証します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return クラス名として認識出来る場合は true
     */
    static boolean isBeginClassName(CharSequence cs, int index) {
        char ch = cs.charAt(index);
        return isGroup(ch, ALPHA);
    }

    /**
     * Javaのメンバ（フィールドやメソッド、コンストラクタ）として認識出来るか検証します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return Javaのメンバとして認識出来る場合は true
     */
    static boolean isBeginMember(CharSequence cs, int index) {
        char ch = cs.charAt(index);
        return isGroup(ch, MEMBER_BEGIN);
    }

    /**
     * 数値の接尾辞文字として有効な文字か検証します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 数値の接尾辞文字として有効な文字の場合は true
     */
    static boolean isNumberSuffix(CharSequence cs, int index) {
        char ch = cs.charAt(index);
        switch (ch) {
        case 'D':
        case 'F':
        case 'G':
        case 'H':
        case 'I':
        case 'L':
        case 'S':
        case 'U':
        case 'd':
        case 'f':
        case 'g':
        case 'h':
        case 'i':
        case 'l':
        case 's':
        case 'u':
            return true;
        default:
            return false;
        }
    }

    /**
     * 囲み文字列の開始文字として有効な文字か検証します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 囲み文字列の開始文字として有効な文字の場合は true
     */
    static boolean isEncloseOpenChar(CharSequence cs, int index) {
        char ch = cs.charAt(index);
        switch (ch) {
        case '!':
        case '"':
            // case '#':
            // case '$':
        case '%':
            // case '&':
        case '\'':
        case '(':
            // case '*':
            // case '+':
            // case ',':
            // case '-':
            // case '.':
        case '/':
            // case ':':
            // case ';':
        case '<':
            // case '=':
        case '?':
        case '@':
        case '[':
            // case '^':
            // case '_':
        case '`':
        case '{':
        case '|':
            // case '~':
            return true;
        default:
            return false;
        }
    }

    /**
     * 囲み文字列の終了文字を返却します。
     * 
     * @param cs
     *            解析する文字列
     * @param index
     *            解析開始位置
     * @return 囲み文字列の終了文字
     */
    static char getEncloseCloseChar(CharSequence cs, int index) {
        char ch = cs.charAt(index);
        switch (ch) {
        case '!':
            return ch;
        case '"':
            return ch;
            // case '#':
            // return ch;
            // case '$':
            // return ch;
        case '%':
            return ch;
            // case '&':
            // return ch;
        case '\'':
            return ch;
        case '(':
            return ')';
            // case '*':
            // return ch;
            // case '+':
            // return ch;
            // case ',':
            // return ch;
            // case '-':
            // return ch;
            // case '.':
            // return ch;
        case '/':
            return ch;
            // case ':':
            // return ch;
            // case ';':
            // return ch;
        case '<':
            return '>';
            // case '=':
            // return ch;
        case '?':
            return ch;
        case '@':
            return ch;
        case '[':
            return ']';
            // case '^':
            // return ch;
            // case '_':
            // return ch;
        case '`':
            return ch;
        case '{':
            return '}';
        case '|':
            return ch;
            // case '~':
            // return ch;
        default:
            return 0x00;
        }
    }

    /*
     * Space
     */

    /* 空白文字か検証します。 */
    static boolean isSpace(CharSequence cs, int index) {
        return (cs.charAt(index) <= ' ');
    }

    /* 次の空白文字のインデックスを検索し返却します。 */
    static int nextSpace(CharSequence cs, int index) {
        return skipNotSp(cs, index);
    }

    /* 空白文字（組込コメントを含む）以外のインデックスまで移動します。 */
    static int siftSpace(OgdlEvent ev) {
        ev.off.set(skipSpace(ev, ev.ptn, ev.off.get()));
        return ev.off.get();
    }

    /* 空白文字（組込コメントを含む）以外のインデックスを検索し返却します。 */
    private static int skipSpace(OgdlEvent ev, CharSequence cs, int index) {
        final int len = cs.length();
        int i = index;
        while (i < len) {
            i = skipSp(cs, i);
            if (isNotation(cs, i)) {
                i = skipEncloseString(ev, cs, ++i);
            } else {
                break;
            }
        }
        return i;
    }

    private static boolean isNotation(CharSequence cs, int i) {
        return isIndex(cs, (i + 1)) && 'N' == cs.charAt(i) && isEncloseOpenChar(cs, (i + 1));
    }

    /* 空白文字以外をスキップしたインデックスを返却します。 */
    private static int skipNotSp(CharSequence cs, int fromIndex) {
        int i = fromIndex;
        while (i < cs.length()) {
            if (!(cs.charAt(i) > ' ')) {
                break;
            }
            i++;
        }
        return i;
    }

    /* 空白文字をスキップしたインデックスを返却します。 */
    private static int skipSp(CharSequence cs, int fromIndex) {
        int i = fromIndex;
        int length = cs.length();
        while (i < length) {
            if (cs.charAt(i) > ' ') {
                break;
            }
            i++;
        }
        return i;
    }

    static boolean isArraylength(Class clazz, String memberName) {
        return clazz.isArray() && "length".equals(memberName);
    }

    /* 終端の文字を示すか検証する */
    static boolean isClose(CharSequence cs, OgdlParseIndex off, char close) {
        return isCharAt(cs, off.get(), close);
    }

    /* 終端の文字を示すか検証する（論理否定） */
    static boolean isNotClose(CharSequence cs, OgdlParseIndex off, char close) {
        return !isCharAt(cs, off.get(), close);
    }

    /* データ集合の区切り文字（論理否定） */
    static boolean isNotSeparator(CharSequence cs, int off) {
        return !isCharAt(cs, off, ',');
    }

    /* 写像を対応付ける区切り文字（論理否定） */
    static boolean isNotMapSeparator(CharSequence cs, OgdlParseIndex off) {
        return !isCharAt(cs, off.get(), '=');
    }

    static void validIndex(OgdlEvent ev, CharSequence cs, OgdlParseIndex off) {
        if (!isIndex(cs, off)) {
            throw new OgdlSyntaxException(ev, HLog.log("index err.", cs, off));
        }
    }

    static void validIndex(OgdlEvent ev, CharSequence cs, int index) {
        if (!isIndex(cs, index)) {
            throw new OgdlSyntaxException(ev, HLog.log("index err.", cs, index));
        }
    }

    static boolean isIndex(CharSequence cs, OgdlParseIndex off) {
        return isIndex(cs, off.get());
    }

    private static boolean isIndex(CharSequence cs, int index) {
        return (0 <= index && index < cs.length());
    }

    static boolean isCharAt(OgdlEvent ev, char c) {
        return isCharAt(ev.ptn, ev.off.get(), c);
    }

    static boolean isCharAt(CharSequence cs, int index, char c) {
        return (isIndex(cs, index) && cs.charAt(index) == c);
    }

    /* 引数に一致する文字列をスキップしたインデックスを返却します。 */
    private static int skip(CharSequence cs, CharSequence chars, int fromIndex) {
        final int charslen = chars.length();
        int i = fromIndex;
        while (i < cs.length()) {
            if (!HEval.isStartsWith(cs, chars, i)) {
                break;
            }
            i += charslen;
        }
        return i;
    }

    /* 指定の文字を指定の開始位置から検索してそのインデックスを返します。発見出来なかった場合は -1 を返します。 */
    private static int indexOf(CharSequence cs, int fromIndex, char searchChar) {
        for (int i = fromIndex; i < cs.length(); i++) {
            if (cs.charAt(i) == searchChar) {
                return i;
            }
        }
        return -1;
    }

    /*
     * static void isSwitch(int ch) { switch (ch) { case '!': case '"': case '#': case '$': case '%': case '&': case '\'': case '(': case ')': case '*': case
     * '+': case ',': case '-': case '.': case '/': case '0': case '1': case '2': case '3': case '4': case '5': case '6': case '7': case '8': case '9': case
     * ':': case ';': case '<': case '=': case '>': case '?': case '@': case 'A': case 'B': case 'C': case 'D': case 'E': case 'F': case 'G': case 'H': case
     * 'I': case 'J': case 'K': case 'L': case 'M': case 'N': case 'O': case 'P': case 'Q': case 'R': case 'S': case 'T': case 'U': case 'V': case 'W': case
     * 'X': case 'Y': case 'Z': case '[': case '\\': case ']': case '^': case '_': case '`': case 'a': case 'b': case 'c': case 'd': case 'e': case 'f': case
     * 'g': case 'h': case 'i': case 'j': case 'k': case 'l': case 'm': case 'n': case 'o': case 'p': case 'q': case 'r': case 's': case 't': case 'u': case
     * 'v': case 'w': case 'x': case 'y': case 'z': case '{': case '|': case '}': case '~': default: break; } }
     */
}
