/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;

/**
 * 基本的なデータ型の構文解析機能を提供します。
 */
class OgdlPrimitiveParser {

    /* 終了リテラル文字を検索して前置文字を削除した文字列を返却します。 */
    static String evaluateString(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            return evaluateEncloseString(ev);
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* リテラル文字で囲まれた文字列として解析して返却します。 */
    static String evaluateEncloseString(OgdlEvent ev) throws OgdlSyntaxException {
        return OgdlSyntax.cutEncloseString(ev, ev.ptn, ev.off);
    }

    /* リテラル文字で囲まれた文字オブジェクトとして解析して返却します。 */
    static Character evaluateCharacter(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            final String src = OgdlSyntax.cutEncloseString(ev, ev.ptn, ev.off);
            return Boxing.box(decodeCharacter(ev, src));
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* リテラル文字で囲まれた文字列を識別子とする、コンテキスト属性の参照を返却します。 */
    static Object evaluateReference(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            final String s = OgdlSyntax.cutEncloseString(ev, ev.ptn, ev.off);
            return ev.context.getAttribute(s);
        } else {
            final String s = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
            return ev.context.getAttribute(s);
        }
    }

    /* ISO書式の日付オブジェクトとして解析して返却します。 */
    static Date evaluateDateTime(OgdlEvent ev) throws OgdlSyntaxException {
        final String s;
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            s = OgdlSyntax.cutEncloseString(ev, ev.ptn, ev.off);
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
        final Date date = toDateTime(s);
        if (date == null) {
            throw new OgdlSyntaxException(ev, HLog.log("date time format err.", s));
        }
        return date;
    }

    /* リテラル文字で囲まれた正規表現パターンとして解析して返却します。 */
    static Pattern evaluateRegexPattern(OgdlEvent ev) throws OgdlSyntaxException {
        final String s;
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            s = OgdlSyntax.cutEncloseString(ev, ev.ptn, ev.off);
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
        try {
            return Pattern.compile(s);
        } catch (PatternSyntaxException e) {
            throw new OgdlSyntaxException(ev, HLog.log("regex pattern err.", s), e);
        }
    }

    /* 数値として解析して返却します。 */
    static Number evaluateNumber(OgdlEvent ev) {
        return toNumber(ev, OgdlSyntax.cutNumberString(ev.ptn, ev.off));
    }

    /* Integer を切り出して返却します。 */
    static Integer evaluateInteger(OgdlEvent ev) throws OgdlSyntaxException {
        return Boxing.box(toNumber(ev, OgdlSyntax.cutNumberString(ev.ptn, ev.off)).intValue());
    }

    /* 無限数として解析して返却します。 */
    static Number evaluateInfinity(OgdlEvent ev) {
        return toInfinity(ev.ptn);
    }

    /* 非数として解析して返却します。 */
    static Number evaluateNaN(OgdlEvent ev) {
        return toNaN(ev.ptn);
    }

    /* 指定されたクラス名をリードして返却します。 */
    static Class evaluateClassByExtendName(OgdlEvent ev) {
        return loadClassByExtendClassName(ev, OgdlSyntax.cutExtendClassName(ev.ptn, ev.off));
    }

    /*
     * private
     */

    /* 文字列を日付に変換して返却します。 */
    private static Date toDateTime(String s) {
        final SimpleDateFormat format = new SimpleDateFormat();
        format.setLenient(false);
        switch (s.length()) {
        case 2:
            format.applyPattern("HH");
            break;
        case 4:
            format.applyPattern("yyyy");
            break;
        case 5:
            format.applyPattern("HH:mm");
            break;
        case 7:
            format.applyPattern("yyyy-MM");
            break;
        case 8:
            format.applyPattern("HH:mm:ss");
            break;
        case 10:
            format.applyPattern("yyyy-MM-dd");
            break;
        case 12:
            format.applyPattern("HH:mm:ss.SSS");
            break;
        case 13:
            format.applyPattern("yyyy-MM-dd HH");
            break;
        case 16:
            format.applyPattern("yyyy-MM-dd HH:mm");
            break;
        case 19:
            format.applyPattern("yyyy-MM-dd HH:mm:ss");
            break;
        case 23:
            format.applyPattern("yyyy-MM-dd HH:mm:ss.SSS");
            break;
        default:
            return null;
        }
        return format.parse(s, new ParsePosition(0));
    }

    /* Unicode escapes の文字表現を文字型へ変換して返却します。 */
    private static char decodeCharacter(OgdlEvent ev, CharSequence src) {
        if (HEval.isEmpty(src)) {
            throw new OgdlSyntaxException(ev, HLog.log("Malformed character encoding.", src));
        }
        char c;
        c = src.charAt(0);
        if (c == '\\') {
            if (src.length() < 2) {
                throw new OgdlSyntaxException(ev, HLog.log("Malformed character encoding.", src));
            }
            c = src.charAt(1);
            if (c == 'u') {
                if (src.length() != 6) {
                    throw new OgdlSyntaxException(ev, HLog.log("Malformed \\uxxxx encoding.", src));
                }
                return decodeHexCharacter(ev, src, 2);
            } else {
                if (src.length() != 2) {
                    throw new OgdlSyntaxException(ev, HLog.log("Malformed \\x encoding.", src));
                }
                return decodeCntrlCharacter(ev, src, c);
            }
        } else {
            if (src.length() != 1) {
                throw new OgdlSyntaxException(ev, HLog.log("Malformed 'x' encoding.", src));
            }
            return c;
        }
    }

    private static char decodeCntrlCharacter(OgdlEvent ev, CharSequence src, char c) {
        switch (c) {
        case 'b':
            return '\b';
        case 't':
            return '\t';
        case 'n':
            return '\n';
        case 'f':
            return '\f';
        case 'r':
            return '\r';
        case '"':
            return '\"';
        case '\'':
            return '\'';
        case '\\':
            return '\\';
        case '0':
            return '\0';
        case '1':
            return '\1';
        case '2':
            return '\2';
        case '3':
            return '\3';
        case '4':
            return '\4';
        case '5':
            return '\5';
        case '6':
            return '\6';
        case '7':
            return '\7';
        default:
            throw new OgdlSyntaxException(ev, HLog.log("Malformed \\x encoding.", src));
        }
    }

    private static char decodeHexCharacter(OgdlEvent ev, CharSequence src, int index) {
        // Read the xxxx
        int value = 0;
        char c;
        for (int i = index; i < (index + 4); i++) {
            c = src.charAt(i);
            switch (c) {
            case '0':
            case '1':
            case '2':
            case '3':
            case '4':
            case '5':
            case '6':
            case '7':
            case '8':
            case '9':
                value = (value << 4) + c - '0';
                break;
            case 'a':
            case 'b':
            case 'c':
            case 'd':
            case 'e':
            case 'f':
                value = (value << 4) + 10 + c - 'a';
                break;
            case 'A':
            case 'B':
            case 'C':
            case 'D':
            case 'E':
            case 'F':
                value = (value << 4) + 10 + c - 'A';
                break;
            default:
                throw new OgdlSyntaxException(ev, HLog.log("Malformed \\uxxxx encoding.", src));
            }
        }
        return (char) value;
    }

    /* 拡張規則で文字列をJava数値型に変換します。 */
    private static Number toNumber(OgdlEvent ev, String s) throws OgdlSyntaxException {
        try {
            final int suffix = OgdlSyntax.toUpper(s.charAt(s.length() - 1));
            final String snum = s.substring(0, (s.length() - 1));
            Number num;
            switch (suffix) {
            case 'D':
                num = Double.valueOf(snum);
                break;
            case 'F':
                num = Float.valueOf(snum);
                break;
            case 'G':
                num = new BigDecimal(snum);
                break;
            case 'H':
                num = new BigInteger(snum);
                break;
            case 'I':
                num = Integer.decode(snum);
                break;
            case 'L':
                num = Long.decode(snum);
                break;
            case 'S':
                num = Short.decode(snum);
                break;
            case 'U':
                num = Byte.decode(snum);
                break;
            default:
                if (HEval.isContains(s, '.')) {
                    num = Double.valueOf(s);
                } else {
                    num = Integer.decode(s);
                }
            }
            return num;
        } catch (NumberFormatException e) {
            throw new OgdlSyntaxException(ev, HLog.log("value is not Number format. ", s), e);
        }
    }

    /* 正の無限大を返却します。 */
    private static Number toInfinity(String s) throws OgdlSyntaxException {
        final int suffix = OgdlSyntax.toUpper(s.charAt(s.length() - 1));
        switch (suffix) {
        case 'D':
            return Boxing.box(Double.POSITIVE_INFINITY);
        case 'F':
            return Boxing.box(Float.POSITIVE_INFINITY);
        default:
            return Boxing.box(Double.POSITIVE_INFINITY);
        }
    }

    /* 非数を返却します。 */
    private static Number toNaN(String s) throws OgdlSyntaxException {
        final int suffix = OgdlSyntax.toUpper(s.charAt(s.length() - 1));
        switch (suffix) {
        case 'D':
            return Boxing.box(Double.NaN);
        case 'F':
            return Boxing.box(Float.NaN);
        default:
            return Boxing.box(Double.NaN);
        }
    }

    /* Java構文（例：java.lang.Integer, int, Integer, Integer[]）からクラスをリードし返却します。 */
    private static Class loadClassByExtendClassName(OgdlEvent ev, String extendClassName) {

        // 拡張の配列表記の検証 ”String[][] 等”
        final StringBuffer symbol = new StringBuffer();
        int off = extendClassName.length();
        while (off >= 2 && extendClassName.charAt(off - 1) == ']' && extendClassName.charAt(off - 2) == '[') {
            off -= 2;
            symbol.append('[');
        }
        final boolean isArray = (symbol.length() > 0);

        // 拡張表記でクラスを検索(クラス名の配列部を切る)
        final String rootName = ((isArray) ? extendClassName.substring(0, off) : extendClassName);
        final Class clazz = findClass(rootName, ev);
        if (isArray) {
            final StringBuffer className = new StringBuffer();
            // 配列の表記を付加する
            if (clazz != null) {
                final String primitiveName = (String) HClass.PRIMITIVE_CLASS_NAME_MAP.get(clazz);
                if (primitiveName == null) {
                    // オブジェクト型
                    className.append(symbol).append(HClass.OBJECT_SYMBOL).append(clazz.getName()).append(';');
                } else {
                    // プリミティブ型
                    className.append(symbol).append(primitiveName);
                }
            } else {
                // オブジェクト型
                className.append(symbol).append(HClass.OBJECT_SYMBOL).append(rootName).append(';');
            }
            // 通常のルールでクラスをロードする
            final Class loadClass = HClass.loadClass(className.toString(), ev.loader);
            if (loadClass != null) {
                return loadClass;
            }
        } else {
            // 配列型以外は返却
            if (clazz != null) {
                return clazz;
            }
            // 通常のルールでクラスをロードする
            final Class loadClass = HClass.loadClass(rootName, ev.loader);
            if (loadClass != null) {
                return loadClass;
            }
        }
        throw new OgdlSyntaxException(ev, HLog.log("not find class. ", extendClassName));
    }

    private static Class findClass(String className, OgdlEvent ev) {
        Class clazz = (Class) HClass.PRIMITIVE_FOR_TYPE_MAP.get(className);
        if (clazz == null) {
            if (!HEval.isContains(className, '.')) {
                try {
                    clazz = Class.forName("java.lang." + className);
                } catch (ClassNotFoundException e) {
                    // no op
                }
            }
        }
        if (clazz == null) {
            clazz = ev.context.forImport(className);
        }
        return clazz;
    }

}
