/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * OGDLの構文解析機能を提供します。<br>
 * このパーザが個別機能を提供する、他のパーザの入口として機能します。
 */
class OgdlParser {

    /*
     * Prefix Expression
     */

    static Object evaluateExpression(OgdlEvent ev) {
        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char open = ptn.charAt(off.get());
        final int begin = off.get();

        switch (open) {
        case '!':
            // Logical Complement Operator
            return OgdlUnaryOperatorParser.evaluateLogicalComplement(ev);
            // case '"':
            // case '#':
            // case '$':
        case '%':
            // Regex
            return OgdlPrimitiveParser.evaluateRegexPattern(ev);
            // case '&':
        case '\'':
            // String
            return OgdlPrimitiveParser.evaluateEncloseString(ev);
        case '(':
            // Operator Function or Expressions Group
            return OgdlOperatorParser.evaluateEncloseOperator(ev);
            // case ')':
            // case '*':
            // case '+':
            // case ',':
        case '-':
            // Unary Minus Operator
            return OgdlUnaryOperatorParser.evaluateUnaryMinus(ev);
            // case '.':
        case '/':
            // Date Time
            return OgdlPrimitiveParser.evaluateDateTime(ev);
        case '0':
        case '1':
        case '2':
        case '3':
        case '4':
        case '5':
        case '6':
        case '7':
        case '8':
        case '9':
            // Number
            return OgdlPrimitiveParser.evaluateNumber(ev);
            // case ':':
            // case ';':
        case '<':
            // Set
            return OgdlCollectionParser.evaluateDefaultSet(ev);
            // case '=':
            // case '>':
            // case '?':
        case '@':
            // Reference or Class Operator
            if (isLiteral(ev, "@@", ptn, begin)) {
                // Class Operator
                return OgdlPrimitiveParser.evaluateClassByExtendName(ev);
            }
            // Reference Operator
            ev.off.increment();
            return OgdlPrimitiveParser.evaluateReference(ev);
            // case 'A':
            // case 'B':
            // case 'C':
        case 'D':
            // Date Time
            ev.off.increment();
            return OgdlPrimitiveParser.evaluateDateTime(ev);
            // case 'E':
        case 'F':
            // Operator Function or Expressions Group
            ev.off.increment();
            return OgdlOperatorParser.evaluateOperator(ev);
            // case 'G':
            // case 'H':
        case 'I':
            // Infinity
            if (isLiteral(ev, "Infinity", ptn, begin)) {
                return OgdlPrimitiveParser.evaluateInfinity(ev);
            }
            break;
        // case 'J':
        // case 'K':
        // case 'L':
        // case 'M':
        case 'N':
            // NaN
            if (isLiteral(ev, "NaN", ptn, begin)) {
                return OgdlPrimitiveParser.evaluateNaN(ev);
            }
            break;
        // case 'O':
        // case 'P':
        // case 'Q':
        case 'R':
            // Regex
            ev.off.increment();
            return OgdlPrimitiveParser.evaluateRegexPattern(ev);
        case 'S':
            // String
            ev.off.increment();
            return OgdlPrimitiveParser.evaluateString(ev);
            // case 'T':
        case 'U':
            // Character
            ev.off.increment();
            return OgdlPrimitiveParser.evaluateCharacter(ev);
            // case 'V':
            // case 'W':
            // case 'X':
            // case 'Y':
            // case 'Z':
        case '[':
            // List
            return OgdlCollectionParser.evaluateDefaultList(ev);
            // case '\\':
            // case ']':
            // case '^':
            // case '_':
        case '`':
            // Character
            return OgdlPrimitiveParser.evaluateCharacter(ev);
        case 'a':
            // array
            if (isPrefixName(ev, "array:", ptn, begin)) {
                return OgdlCollectionParser.evaluateArray(ev);
            }
            break;
        // case 'b':
        case 'c':
            // class
            if (isPrefixName(ev, "class:", ptn, begin)) {
                return OgdlPrimitiveParser.evaluateClassByExtendName(ev);
            }
            // coll
            if (isPrefixName(ev, "coll:", ptn, begin)) {
                return OgdlCollectionParser.evaluateCollection(ev);
            }
            // constructor
            if (isPrefixName(ev, "cons:", ptn, begin)) {
                return OgdlIntrospectParser.evaluateConstructor(ev);
            }
            break;
        // case 'd':
        // case 'e':
        case 'f':
            // Boolean.FALSE
            if (isLiteral(ev, "false", ptn, begin)) {
                return Boolean.FALSE;
            }
            // field
            if (isPrefixName(ev, "field:", ptn, begin)) {
                return OgdlIntrospectParser.evaluateField(ev);
            }
            // function
            if (isPrefixName(ev, "f:", ptn, begin)) {
                return OgdlFunctionParser.evaluateFunction(ev);
            }
            break;
        // case 'g':
        // case 'h':
        // case 'i':
        // case 'j':
        // case 'k':
        case 'l':
            // list
            if (isPrefixName(ev, "list:", ptn, begin)) {
                return OgdlCollectionParser.evaluateList(ev);
            }
            break;
        case 'm':
            // map
            if (isPrefixName(ev, "map:", ptn, begin)) {
                return OgdlCollectionParser.evaluateMap(ev);
            }
            // method
            if (isPrefixName(ev, "method:", ptn, begin)) {
                return OgdlIntrospectParser.evaluateMethod(ev);
            }
            break;
        case 'n':
            // Null
            if (isLiteral(ev, "null", ptn, begin)) {
                return null;
            }
            // new
            if (isPrefixName(ev, "new:", ptn, begin)) {
                return OgdlIntrospectParser.evaluateNewInstance(ev);
            }
            break;
        // case 'o':
        // case 'p':
        // case 'q':
        case 'r':
            // Reference Operator
            if (isPrefixName(ev, "ref:", ptn, begin)) {
                return OgdlPrimitiveParser.evaluateReference(ev);
            }
            break;
        case 's':
            // set
            if (isPrefixName(ev, "set:", ptn, begin)) {
                return OgdlCollectionParser.evaluateSet(ev);
            }
            // sortedSet
            if (isPrefixName(ev, "sortedSet:", ptn, begin)) {
                return OgdlCollectionParser.evaluateSortedSet(ev);
            }
            // sortedMap
            if (isPrefixName(ev, "sortedMap:", ptn, begin)) {
                return OgdlCollectionParser.evaluateSortedMap(ev);
            }
            break;
        case 't':
            // Boolean.TRUE
            if (isLiteral(ev, "true", ptn, begin)) {
                return Boolean.TRUE;
            }
            break;
        // case 'u':
        // case 'v':
        // case 'w':
        // case 'x':
        // case 'y':
        // case 'z':
        case '{':// Map
            return OgdlCollectionParser.evaluateDefaultMap(ev);
            // case '|':
            // case '}':
        case '~':// Bitwise Complement Operator
            return OgdlUnaryOperatorParser.evaluateBitwiseComplement(ev);
        default:
            break;
        }
        throw new OgdlSyntaxException(ev, "not found expression.");
    }

    /*
     * Collection
     */

    /* コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateEncloseCollection(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateEncloseCollection(ev);
    }

    /* 開始文字を検証せずに、コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateCloseCollection(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateCloseCollection(ev);
    }

    /* マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateEncloseMap(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateEncloseMap(ev);
    }

    /* 開始文字を検証せずに、マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateCloseMap(OgdlEvent ev) {
        return OgdlCollectionParser.evaluateCloseMap(ev);
    }

    /* 数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として引数のリストに格納します。 */
    static List evaluateEncloseProgressionList(OgdlEvent ev) {
        return (List) OgdlCollectionParser.evaluateEncloseProgressionCollection(ev);
    }

    /* 開始文字を検証せずに、数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として解析して引数のリストに格納して返却します。 */
    static List evaluateCloseProgressionList(OgdlEvent ev) {
        return (List) OgdlCollectionParser.evaluateCloseProgressionCollection(ev);
    }

    /*
     * Postfix Expression
     */

    /* 後置式を実行します。 */
    static Object evaluatePostfixExpression(OgdlEvent ev, Object returnValue) {
        final char open = ev.ptn.charAt(ev.off.get());
        switch (open) {
        case '[':
        case '.':
        case '#':
            ev.target = returnValue;
            return OgdlIntrospectParser.evaluateObjectNavigate(ev);
        default:
            break;
        }
        return returnValue;
    }

    /*
     * private
     */

    private static boolean isLiteral(OgdlEvent ev, String literal, String ptn, int begin) {
        if (ptn.startsWith(literal, begin)) {
            ev.off.add(literal.length());
            return true;
        }
        return false;
    }

    /*   */
    private static boolean isPrefixName(OgdlEvent ev, String id, String ptn, int begin) {
        if (ptn.startsWith(id, begin)) {
            ev.off.add(id.length());
            OgdlSyntax.siftSpace(ev);
            if (ev.off.get() >= ev.ptn.length()) {
                throw new OgdlSyntaxException(ev, "offset err.");
            }
            return true;
        }
        return false;
    }

}
