/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * 演算子の構文解析機能を提供します。
 */
class OgdlOperatorParser {

    /* 演算式を実行します。 */
    static Object evaluateOperator(OgdlEvent ev) {
        if (OgdlSyntax.isEncloseOpenChar(ev.ptn, ev.off.get())) {
            return evaluateEncloseOperator(ev);
        } else {
            throw new OgdlSyntaxException(ev, "is not enclose literal.");
        }
    }

    /* 演算式を実行します。 */
    static Object evaluateEncloseOperator(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char close = OgdlSyntax.getEncloseCloseChar(ptn, off.get());
        off.increment();

        int begin = off.get();
        int next = OgdlSyntax.skipOperator(ptn, begin);

        // 演算子が検出出来ない場合、式の列挙として解析
        if (next == begin || (next - begin) > 4) {
            return evaluateExpressionGroup(ev, close);
        }

        // 演算子の種別を検出する
        final String name = ptn.substring(begin, next);
        final int size = next - begin;
        final int type = asOperatorType(name, size);

        // 演算子が検出出来ない場合、式の列挙として解析
        if (-1 == type) {
            return evaluateExpressionGroup(ev, close);
        }

        // 演算処理を実行する
        off.set(next);
        final List args = getArguments(ev, close);
        try {

            if (args.isEmpty()) {
                throw new OgdlFunctionException(ev, "empty arguments. ");
            }
            if (0 <= type && type <= 7) {
                return evaluateArithmeticOperator(ev, type, args);
            }
            if (8 <= type && type <= 10) {
                return evaluateShiftOperator(ev, type, args);
            }
            if (11 <= type && type <= 16) {
                return evaluateRelationalOperator(ev, type, args);
            }
            if (20 <= type) {
                return evaluateObjectOperator(ev, type, args);
            }
            throw new OgdlFunctionException(ev, "illegal operator type. ");

        } catch (OgdlFunctionException e) {
            throw e.throwFor(ev, name, args);
        } catch (RuntimeException e) {
            final OgdlFunctionException fe = new OgdlFunctionException("java runtime err.", e);
            throw fe.throwFor(ev, name, args);
        }

    }

    /*
     * 算術演算子
     */

    /* 算術演算子を実行して返却します。 */
    private static Object evaluateArithmeticOperator(OgdlEvent ev, int type, List args) {
        if (2 > args.size()) {
            throw new OgdlFunctionException(ev, "illegal arguments size. 2 > size. ");
        }
        final Object f = args.get(0);
        final Object t = args.get(1);
        if (isNumberOperand(f, t)) {
            return evaluateArithmeticNumberOperator(ev, type, args);
        }
        if (isBooleanOperand(f, t)) {
            return evaluateArithmeticBooleanOperator(ev, type, args);
        }
        if (isStringOperand(f, t)) {
            return evaluateArithmeticStringOperator(ev, type, args);
        }
        throw new OgdlFunctionException(ev, "illegal operand type. ");
    }

    /*
     * シフト演算子
     */

    /* シフト演算子を実行して返却します。 */
    private static Object evaluateShiftOperator(OgdlEvent ev, int type, List args) {
        if (2 > args.size()) {
            throw new OgdlFunctionException(ev, "illegal arguments size. 2 > size. ");
        }
        final Object f = args.get(0);
        final Object t = args.get(1);
        if (isNumberOperand(f, t)) {
            return evaluateShiftNumberOperator(ev, type, args);
        }
        throw new OgdlFunctionException(ev, "illegal operand type. ");
    }

    /*
     * 比較演算子
     */

    /* 比較演算子を実行して返却します。 */
    private static Object evaluateRelationalOperator(OgdlEvent ev, int type, List args) {
        if (2 > args.size()) {
            throw new OgdlFunctionException(ev, "illegal arguments size. 2 > size. ");
        }
        final Object f = args.get(0);
        final Object t = args.get(1);
        if (isNumberOperand(f, t)) {
            return evaluateRelationalNumberOperator(ev, type, args);
        }
        if (isComparableOperand(f, t)) {
            return evaluateRelationalComparableOperator(ev, type, args);
        }
        throw new OgdlFunctionException(ev, "illegal operand type. ");
    }

    /*
     * オブジェクト型共通の演算子
     */

    /* Object 型の演算を実行して返却します。 */
    private static Object evaluateObjectOperator(OgdlEvent ev, int type, List args) {
        switch (type) {
        case 20: // iof (instanceof)
            return evaluateInstanceOfOperator(ev, args);
        case 21: // eqs
            return evaluateEqualsOperator(args);
        case 22: // neqs
            return evaluateNotEqualsOperator(args);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /*
     * 算術演算子を実行する
     */

    /* 数値型の算術演算子を実行して返却します。 */
    private static Object evaluateArithmeticNumberOperator(OgdlEvent ev, int type, List args) {
        try {
            // 1. 片方が BigInteger であり、片方が (Byte|Short|Integer|Long) の場合は、BigInteger に変換
            // 2. 片方が BigInteger であり、片方が (Float|Double) の場合は、BigDecimal に変換
            // 3. 片方が BigDecimal であり、片方が (Byte|Short|Integer|Long|Float|Double|BigInteger) の場合は、BigDecimal に変換
            // 5. 双方が BigInteger の場合は、変換なし
            // 6. 双方が BigDecimal の場合は、変換なし
            // 7. 以外は java と同等
            final Iterator i = args.iterator();
            Number f = toNumber(i.next());
            Number t = null;
            while (i.hasNext()) {
                t = toNumber(i.next());
                // BigDecimal
                if (isBigDecimalOperand(f, t)) {
                    BigDecimal bf = toBigDecimal(f);
                    BigDecimal bt = toBigDecimal(t);
                    f = evaluateArithmeticBigDecimalOperator(ev, type, bf, bt);
                } else // BigInteger
                if (f instanceof BigInteger || t instanceof BigInteger) {
                    BigInteger bf = toBigInteger(f);
                    BigInteger bt = toBigInteger(t);
                    f = evaluateArithmeticBigIntegerOperator(ev, type, bf, bt);
                } else // Double
                if (f instanceof Double || t instanceof Double) {
                    f = evaluateArithmeticDoubleOperator(ev, type, f.doubleValue(), t.doubleValue());
                } else // Float
                if (f instanceof Float || t instanceof Float) {
                    f = evaluateArithmeticFloatOperator(ev, type, f.floatValue(), t.floatValue());
                } else // Long
                if (f instanceof Long || t instanceof Long) {
                    f = evaluateArithmeticLongOperator(ev, type, f.longValue(), t.longValue());
                } else // Integer or Short or Byte
                if (isIntegerOperand(f, t)) {
                    f = evaluateArithmeticIntegerOperator(ev, type, f.intValue(), t.intValue());
                } else {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                }
            }
            return f;

        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args is not Number. ", e);
        } catch (NumberFormatException e) {
            throw new OgdlFunctionException(ev, "args is infinite or NaN. ", e);
        } catch (ArithmeticException e) {
            throw new OgdlFunctionException(ev, "arithmetic err. ", e);
        }
    }

    private static boolean isBigDecimalOperand(Number f, Number t) {
        // BigDecimal
        if (f instanceof BigDecimal || t instanceof BigDecimal) {
            return true;
        } else // BigInteger or (Double | Float)
        if (f instanceof BigInteger && (t instanceof Double || t instanceof Float)) {
            return true;
        } else // BigInteger or (Double | Float)
        if (t instanceof BigInteger && (f instanceof Double || f instanceof Float)) {
            return true;
        }
        return false;
    }

    /* 論理型の算術演算子を実行して返却します。 */
    private static Object evaluateArithmeticBooleanOperator(OgdlEvent ev, int type, List args) {
        try {
            final Iterator i = args.iterator();
            Boolean f = (Boolean) i.next();
            Boolean t = null;
            for (Iterator j = i; j.hasNext();) {
                t = (Boolean) j.next();
                f = evaluateArithmeticBooleanOperator(ev, type, f.booleanValue(), t.booleanValue());
            }
            return f;
        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args is not Boolean. ", e);
        }
    }

    /* 論理型の算術演算子を実行して返却します。 */
    private static Object evaluateArithmeticStringOperator(OgdlEvent ev, int type, List args) {
        switch (type) {
        case 0: // '+':
            return evaluateStringConcatenationOperator(args);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* int 型の算術演算子を実行して返却します。 */
    private static Integer evaluateArithmeticIntegerOperator(OgdlEvent ev, int type, int from, int to) {
        switch (type) {
        case 0: // '+':
            return Boxing.box(from + to);
        case 1: // '-':
            return Boxing.box(from - to);
        case 2: // '*':
            return Boxing.box(from * to);
        case 3: // '/':
            return Boxing.box(from / to);
        case 4: // '%':
            return Boxing.box(from % to);
        case 5: // '&':
            return Boxing.box(from & to);
        case 6: // '^':
            return Boxing.box(from ^ to);
        case 7: // '|':
            return Boxing.box(from | to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* long 型の算術演算子を実行して返却します。 */
    private static Long evaluateArithmeticLongOperator(OgdlEvent ev, int type, long from, long to) {
        switch (type) {
        case 0: // '+':
            return Boxing.box(from + to);
        case 1: // '-':
            return Boxing.box(from - to);
        case 2: // '*':
            return Boxing.box(from * to);
        case 3: // '/':
            return Boxing.box(from / to);
        case 4: // '%':
            return Boxing.box(from % to);
        case 5: // '&':
            return Boxing.box(from & to);
        case 6: // '^':
            return Boxing.box(from ^ to);
        case 7: // '|':
            return Boxing.box(from | to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* float 型の算術演算子を実行して返却します。 */
    private static Float evaluateArithmeticFloatOperator(OgdlEvent ev, int type, float from, float to) {
        switch (type) {
        case 0: // '+':
            return Boxing.box(from + to);
        case 1: // '-':
            return Boxing.box(from - to);
        case 2: // '*':
            return Boxing.box(from * to);
        case 3: // '/':
            return Boxing.box(from / to);
        case 4: // '%':
            return Boxing.box(from % to);
        default:// '&' '^' '|'
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* double 型の算術演算子を実行して返却します。 */
    private static Double evaluateArithmeticDoubleOperator(OgdlEvent ev, int type, double from, double to) {
        switch (type) {
        case 0: // '+':
            return Boxing.box(from + to);
        case 1: // '-':
            return Boxing.box(from - to);
        case 2: // '*':
            return Boxing.box(from * to);
        case 3: // '/':
            return Boxing.box(from / to);
        case 4: // '%':
            return Boxing.box(from % to);
        default:// '&' '^' '|'
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* BigInteger 型の算術演算子を実行して返却します。 */
    private static BigInteger evaluateArithmeticBigIntegerOperator(OgdlEvent ev, int type, BigInteger from, BigInteger to) {
        switch (type) {
        case 0: // '+':
            return (from.add(to));
        case 1: // '-':
            return (from.subtract(to));
        case 2: // '*':
            return (from.multiply(to));
        case 3: // '/':
            return (from.divide(to));
        case 4: // '%':
            return (from.remainder(to));
        case 5: // '&':
            return (from.and(to));
        case 6: // '^':
            return (from.xor(to));
        case 7: // '|':
            return (from.or(to));
        default:// 
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* BigDecimal 型の算術演算子を実行して返却します。 */
    private static BigDecimal evaluateArithmeticBigDecimalOperator(OgdlEvent ev, int type, BigDecimal from, BigDecimal to) {
        switch (type) {
        case 0: // '+':
            return (from.add(to));
        case 1: // '-':
            return (from.subtract(to));
        case 2: // '*':
            return (from.multiply(to));
        case 3: // '/':
            return (from.divide(to, BigDecimal.ROUND_HALF_EVEN));
        case 4: // '%':
            return from.subtract(new BigDecimal(from.divide(to, BigDecimal.ROUND_DOWN).toBigInteger()).multiply(to));
        default:// '&' '^' '|'
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* boolean 型の算術演算子を実行して返却します。 */
    private static Boolean evaluateArithmeticBooleanOperator(OgdlEvent ev, int type, boolean from, boolean to) {
        switch (type) {
        case 5: // '&'
            return Boxing.box(from & to);
        case 6: // '^':
            return Boxing.box(from ^ to);
        case 7: // '|':
            return Boxing.box(from | to);
        default:// '-' '*' '/' '%'
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /*
     * シフト演算子を実行する
     */

    /* 数値型のシフト演算子を実行して返却します。 */
    private static Object evaluateShiftNumberOperator(OgdlEvent ev, int type, List args) {
        try {
            // 1. java と同等、プリミティブの整数型のみ有効
            final Iterator i = args.iterator();
            Number f = toNumber(i.next());
            Number t = null;
            while (i.hasNext()) {
                t = toNumber(i.next());
                // BigDecimal
                if (f instanceof BigDecimal || t instanceof BigDecimal) {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                } else // Double
                if (f instanceof Double || t instanceof Double) {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                } else // Float
                if (f instanceof Float || t instanceof Float) {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                }
                // BigInteger
                if (f instanceof BigInteger && isIntegerOperand(t)) {
                    f = evaluateShiftBigIntegerIntOperator(ev, type, (BigInteger) f, t.intValue());
                } else // Long
                if (f instanceof Long && t instanceof Long) {
                    f = evaluateShiftLongLongOperator(ev, type, f.longValue(), t.longValue());
                } else // Long and (Integer or Short or Byte)
                if (isIntegerOperand(f) && t instanceof Long) {
                    f = evaluateShiftIntegerLongOperator(ev, type, f.intValue(), t.longValue());
                } else // Long and (Integer or Short or Byte)
                if (f instanceof Long && isIntegerOperand(t)) {
                    f = evaluateShiftLongIntegerOperator(ev, type, f.longValue(), t.intValue());
                } else // Integer or Short or Byte
                if (isIntegerOperand(f, t)) {
                    f = evaluateShiftIntegerIntegerOperator(ev, type, f.intValue(), t.intValue());
                } else {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                }
            }
            return f;

        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args is not Number. ", e);
        }
    }

    /* int 型とint 型のシフト演算子を実行して返却します。 */
    private static Integer evaluateShiftIntegerIntegerOperator(OgdlEvent ev, int type, int from, int to) {
        switch (type) {
        case 8: // '<<':
            return Boxing.box(from << to);
        case 9: // '>>':
            return Boxing.box(from >> to);
        case 10: // '>>>':
            return Boxing.box(from >>> to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* int 型とlong 型のシフト演算子を実行して返却します。 */
    private static Integer evaluateShiftIntegerLongOperator(OgdlEvent ev, int type, int from, long to) {
        switch (type) {
        case 8: // '<<':
            return Boxing.box(from << to);
        case 9: // '>>':
            return Boxing.box(from >> to);
        case 10: // '>>>':
            return Boxing.box(from >>> to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* long 型とint 型のシフト演算子を実行して返却します。 */
    private static Long evaluateShiftLongIntegerOperator(OgdlEvent ev, int type, long from, int to) {
        switch (type) {
        case 8: // '<<':
            return Boxing.box(from << to);
        case 9: // '>>':
            return Boxing.box(from >> to);
        case 10: // '>>>':
            return Boxing.box(from >>> to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* long 型とlong 型のシフト演算子を実行して返却します。 */
    private static Long evaluateShiftLongLongOperator(OgdlEvent ev, int type, long from, long to) {
        switch (type) {
        case 8: // '<<':
            return Boxing.box(from << to);
        case 9: // '>>':
            return Boxing.box(from >> to);
        case 10: // '>>>':
            return Boxing.box(from >>> to);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /* BigInteger 型と int 型のシフト演算子を実行して返却します。 */
    private static BigInteger evaluateShiftBigIntegerIntOperator(OgdlEvent ev, int type, BigInteger from, int to) {
        switch (type) {
        case 8: // '<<':
            return from.shiftLeft(to);
        case 9: // '>>':
            return from.shiftRight(to);
        default:// '>>>'
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /*
     * 比較演算子を実行する
     */

    /* 数値型の比較演算子を実行して返却します。 */
    private static Boolean evaluateRelationalNumberOperator(OgdlEvent ev, int type, List args) {
        try {
            // 1. 片方が BigInteger であり、片方が (Byte|Short|Integer|Long) の場合は、BigInteger に変換
            // 2. 片方が BigInteger であり、片方が (Float|Double) の場合は、BigDecimal に変換
            // 3. 片方が BigDecimal であり、片方が (Byte|Short|Integer|Long|Float|Double|BigInteger) の場合は、BigDecimal に変換
            // 5. 双方が BigInteger の場合は、変換なし
            // 6. 双方が BigDecimal の場合は、変換なし
            // 7. 以外は java と同等
            final Iterator i = args.iterator();
            final Number f = toNumber(i.next());
            Number t = null;
            for (Iterator j = i; j.hasNext();) {
                t = toNumber(j.next());
                // BigDecimal
                if (isBigDecimalOperand(f, t)) {
                    BigDecimal bf = toBigDecimal(f);
                    BigDecimal bt = toBigDecimal(t);
                    if (!evaluateRelationalComparableOperator(ev, type, bf, bt)) {
                        return Boolean.FALSE;
                    }
                } else // BigInteger
                if (f instanceof BigInteger || t instanceof BigInteger) {
                    BigInteger bf = toBigInteger(f);
                    BigInteger bt = toBigInteger(t);
                    if (!evaluateRelationalComparableOperator(ev, type, bf, bt)) {
                        return Boolean.FALSE;
                    }
                } else // Double
                if (f instanceof Double || t instanceof Double) {
                    if (!evaluateRelationalComparableOperator(ev, type, Boxing.box(f.doubleValue()), Boxing.box(t.doubleValue()))) {
                        return Boolean.FALSE;
                    }
                } else // Float
                if (f instanceof Float || t instanceof Float) {
                    if (!evaluateRelationalComparableOperator(ev, type, Boxing.box(f.floatValue()), Boxing.box(t.floatValue()))) {
                        return Boolean.FALSE;
                    }
                } else // Long
                if (f instanceof Long || t instanceof Long) {
                    if (!evaluateRelationalComparableOperator(ev, type, Boxing.box(f.longValue()), Boxing.box(t.longValue()))) {
                        return Boolean.FALSE;
                    }
                } else // Integer
                if (isIntegerOperand(f, t)) {
                    if (evaluateRelationalComparableOperator(ev, type, Boxing.box(f.intValue()), Boxing.box(t.intValue()))) {
                        return Boolean.FALSE;
                    }
                } else {
                    throw new OgdlFunctionException(ev, "illegal operand type. ");
                }
            }
            return Boolean.TRUE;

        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args is not Number. ", e);
        } catch (NumberFormatException e) {
            throw new OgdlFunctionException(ev, "args is infinite or NaN. ", e);
        }
    }

    /* 論理型の比較演算子を実行して返却します。 */
    private static Boolean evaluateRelationalComparableOperator(OgdlEvent ev, int type, List args) {
        try {
            final Iterator i = args.iterator();
            Comparable f = (Comparable) i.next();
            Comparable t = null;
            for (Iterator j = i; j.hasNext();) {
                t = (Comparable) j.next();
                if (!evaluateRelationalComparableOperator(ev, type, f, t)) {
                    return Boolean.FALSE;
                }
            }
            return Boolean.TRUE;
        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args is not Comparable. ", e);
        }
    }

    /* BigInteger 型と BigDecimal 型の比較演算子を実行して返却します。 */
    private static boolean evaluateRelationalComparableOperator(OgdlEvent ev, int type, Comparable from, Comparable to) {
        switch (type) {
        case 11: // ==
            return (from.compareTo(to) == 0);
        case 12: // !=
            return (from.compareTo(to) != 0);
        case 13: // <
            return (from.compareTo(to) < 0);
        case 14: // <=
            return (from.compareTo(to) <= 0);
        case 15: // >
            return (from.compareTo(to) > 0);
        case 16: // >=
            return (from.compareTo(to) >= 0);
        default:
            throw new OgdlFunctionException(ev, "illegal operator type. ");
        }
    }

    /*
     * オブジェクト型共通の演算子を実行する
     */

    /*
     * instanceOf
     */

    private static Boolean evaluateInstanceOfOperator(OgdlEvent ev, List args) {
        try {
            final Iterator i = args.iterator();
            final Object e = i.next();
            while (i.hasNext()) {
                if (!((Class) i.next()).isInstance(e)) {
                    return Boolean.FALSE;
                }
            }
            return Boolean.TRUE;
        } catch (ClassCastException e) {
            throw new OgdlFunctionException(ev, "args[1 < n] is not Class. ", e);
        }
    }

    /*
     * Equals
     */

    private static Boolean evaluateNotEqualsOperator(List args) {
        final Iterator i = args.iterator();
        final Object e = i.next();
        while (i.hasNext()) {
            if (eqs(e, i.next())) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }

    private static Boolean evaluateEqualsOperator(List args) {
        final Iterator i = args.iterator();
        final Object e = i.next();
        while (i.hasNext()) {
            if (!eqs(e, i.next())) {
                return Boolean.FALSE;
            }
        }
        return Boolean.TRUE;
    }

    private static boolean eqs(final Object e, Object e2) {
        return e == e2 || (e != null && e.equals(e2));
    }

    /*
     * String Concatenation
     */

    private static String evaluateStringConcatenationOperator(List args) {
        final StringBuffer buff = new StringBuffer();
        final Iterator i = args.iterator();
        while (i.hasNext()) {
            buff.append(i.next());
        }
        return buff.toString();
    }

    /*
     * Helper
     */

    /* オペレータのタイプを示す数値を返却します。 */
    private static int asOperatorType(String name, int size) {
        int type = -1;
        switch (size) {
        case 1:
            char c = name.charAt(0);
            switch (c) {
            case '%':// Remainder
                type = 4;
                break;
            case '&':// Bitwise and operator
                type = 5;
                break;
            case '*':// Multiplication
                type = 2;
                break;
            case '+':// Addition or string concatenation
                type = 0;
                break;
            case '-':// Subtraction
                type = 1;
                break;
            case '/':// Division
                type = 3;
                break;
            case '<':// Less than comparison
                type = 13;
                break;
            case '>':// Greater than comparison
                type = 15;
                break;
            case '^':// Bitwise exclusive-or operator
                type = 6;
                break;
            case '|':// Bitwise or operator
                type = 7;
                break;
            default:
                break;
            }
            break;
        case 2:
            // Equality
            if (name.equals("eq")) {
                type = 11;
            } else // Inequality
            if (name.equals("ne")) {
                type = 12;
            } else // Equality
            if (name.equals("==")) {
                type = 11;
            } else // Inequality
            if (name.equals("!=")) {
                type = 12;
            } else // Less than comparison
            if (name.equals("lt")) {
                type = 13;
            } else // Greater than comparison
            if (name.equals("gt")) {
                type = 15;
            } else // Less than or equals comparison
            if (name.equals("le")) {
                type = 14;
            } else // Greater than or equals comparison
            if (name.equals("ge")) {
                type = 16;
            } else // Less than or equals comparison
            if (name.equals("<=")) {
                type = 14;
            } else // Greater than or equals comparison
            if (name.equals(">=")) {
                type = 16;
            } else // Bit shift left
            if (name.equals("<<")) {
                type = 8;
            } else // Bit shift right
            if (name.equals(">>")) {
                type = 9;
            }
            break;
        case 3:
            // Equality's
            if (name.equals("eqs")) {
                type = 21;
            } else // Inequality's
            if (name.equals("nes")) {
                type = 22;
            } else // Type Comparison
            if (name.equals("iof")) {
                type = 20;
            } else // Multiplication
            if (name.equals("mul")) {
                type = 2;
            } else // Addition or string concatenation
            if (name.equals("add")) {
                type = 0;
            } else // Subtraction
            if (name.equals("sub")) {
                type = 1;
            } else // Division
            if (name.equals("div")) {
                type = 3;
            } else // Remainder
            if (name.equals("mod")) {
                type = 4;
            } else // Bitwise exclusive-or operator
            if (name.equals("xor")) {
                type = 6;
            } else // Bitwise or operator
            if (name.equals("bor")) {
                type = 7;
            } else // Bit shift left
            if (name.equals("shl")) {
                type = 8;
            } else // Bit shift right
            if (name.equals("shr")) {
                type = 9;
            } else // Logical shift right
            if (name.equals(">>>")) {
                type = 10;
            }
            break;
        case 4:
            // Logical shift right
            if (name.equals("ushr")) {
                type = 10;
            } else // Bitwise and operator
            if (name.equals("band")) {
                type = 5;
            }
            break;
        }
        return type;
    }

    /* コンマ区切りの式のグループとして解析して末尾の結果を返却します。 */
    static Object evaluateExpressionGroup(OgdlEvent ev, char close) {
        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        Object output = null;
        if (OgdlSyntax.isNotClose(ptn, off, close)) {
            while (off.get() < ptn.length()) {
                OgdlSyntax.siftSpace(ev);
                output = OgdlRuntime.evaluate(ev);// 末尾のデータを返却する
                OgdlSyntax.validIndex(ev, ptn, off);
                OgdlSyntax.siftSpace(ev);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                off.increment();// + ','
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        off.increment();// next index
        return output;
    }

    /* (method: arg, arg, ...) の引数を解析して返却します。 */
    private static List getArguments(OgdlEvent ev, char close) {
        return (List) OgdlParser.evaluateCloseCollection(ev.get('\u0000', close, new LinkedList()));
    }

    /* 数値型に変換して返却します */
    private static Number toNumber(Object o) {
        return (o instanceof Character) ? Boxing.box((int) ((Character) o).charValue()) : (Number) o;
    }

    /* BigInteger 型に変換して返却します */
    private static BigInteger toBigInteger(Number n) {
        return ((n instanceof BigInteger) ? (BigInteger) n : new BigInteger(String.valueOf(n)));
    }

    /* BigDecimal 型に変換して返却します */
    private static BigDecimal toBigDecimal(Number n) {
        return ((n instanceof BigDecimal) ? (BigDecimal) n : (n instanceof BigInteger) ? new BigDecimal((BigInteger) n) : new BigDecimal(String.valueOf(n)));
    }

    private static boolean isStringOperand(Object f, Object t) {
        return (f instanceof String) || (t instanceof String);
    }

    private static boolean isBooleanOperand(Object f, Object t) {
        return (f instanceof Boolean) && (t instanceof Boolean);
    }

    private static boolean isComparableOperand(Object f, Object t) {
        return (f instanceof Comparable) && (t instanceof Comparable);
    }

    private static boolean isNumberOperand(Object f, Object t) {
        return (f instanceof Number || f instanceof Character) && (t instanceof Number || t instanceof Character);
    }

    private static boolean isIntegerOperand(Number f, Number t) {
        return isIntegerOperand(f) && isIntegerOperand(t);
    }

    private static boolean isIntegerOperand(Number n) {
        return (n instanceof Integer || n instanceof Short || n instanceof Byte);
    }

}
