/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.beans.IndexedPropertyDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

/**
 * イントロスペクションの構文解析機能を提供します。
 */
class OgdlIntrospectParser {

    /* データ要素へアクセスする式を実行して返却します。 */
    static Object evaluateObjectNavigate(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final Object target = ev.target;

        // index, Field, Method
        try {
            Object o = target;
            while ((off.get() + 1) < ptn.length()) {

                final char open = ptn.charAt(off.get());
                switch (open) {
                case '[':
                    Object key = readIndexingKeys(ev, ptn, off);
                    o = readIndexingValue(ev, key, o);
                    break;
                case '.':
                    off.increment();// add separator
                    final String memberName = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
                    final Class clazz = o.getClass();
                    // method or property or field
                    if (OgdlSyntax.isCharAt(ptn, off.get(), '(')) {// method 'method(arg..)'
                        o = invokeMethod(ev, memberName, o, clazz);
                    } else if (OgdlSyntax.isCharAt(ptn, off.get(), '[')) {// index property or field or property
                        o = readIndexPropertyOrFieldOrProperty(ev, memberName, o, clazz);
                    } else {// field or property or Array.length
                        o = readFieldOrProperty(ev, memberName, o, clazz);
                    }
                    break;
                case '#':
                    off.increment();// add separator
                    final String staticMemberName = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
                    // static method or static field
                    if (OgdlSyntax.isCharAt(ptn, off.get(), '(')) {// static method 'method(arg..)'
                        o = invokeStaticMethod(ev, staticMemberName, o);
                    } else {// static field
                        o = readStaticFieldValue(ev, staticMemberName, o);
                    }
                    break;
                default:
                    return o;
                }
            }
            return o;

        } catch (IndexOutOfBoundsException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        } catch (NumberFormatException e) {
            throw new OgdlSyntaxException(ev, "syntax err.", e);
        }

    }

    /* インスタンス生成式を実行して値を返却します。 */
    static Object evaluateNewInstance(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        // get class
        Class clazz = OgdlPrimitiveParser.evaluateClassByExtendName(ev);

        // get value
        Object obj = null;

        char offLiteral = ptn.charAt(off.get());
        // switch create type
        if (offLiteral == '#') {

            // Factory Method

            // get member name
            off.increment();// add offLiteral length
            String memberName = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
            if (OgdlSyntax.isCharAt(ptn, off.get(), '(')) {

                Object[] args = evaluateEncloseArguments(ev.get('(', ')'));
                try {
                    obj = BeanIntrospectHelper.invokeMethod(clazz, null, memberName, args);
                } catch (OgdlSyntaxException e) {
                    throw e.throwFor(ev);
                }
                int next_off = OgdlSyntax.siftSpace(ev);

                // set properties
                if (next_off < ptn.length() && OgdlSyntax.isCharAt(ptn, next_off, '{')) {
                    off.set(next_off);
                    writeInitProperties(ev.get('{', '}'), obj);
                }

            } else {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }

        } else if (offLiteral == '(') {

            // Constructor
            Object[] args = evaluateEncloseArguments(ev.get('(', ')'));
            try {
                obj = BeanIntrospectHelper.newInstance(clazz, args);
            } catch (OgdlSyntaxException e) {
                throw e.throwFor(ev);
            }
            int next_off = OgdlSyntax.siftSpace(ev);

            // set properties
            if (OgdlSyntax.isCharAt(ptn, next_off, '{')) {
                off.set(next_off);
                writeInitProperties(ev.get('{', '}'), obj);
            }

        } else if (offLiteral == '[') {

            // Array "class[n][n]"
            Integer[] dimensions = evaluateDimensionArraySizes(ev);
            try {
                obj = Array.newInstance(clazz, Boxing.unbox(dimensions));
            } catch (IllegalArgumentException e) {
                throw new OgdlSyntaxException(ev, "arrays newInstance err.", e);
            } catch (NegativeArraySizeException e) {
                throw new OgdlSyntaxException(ev, "arrays newInstance err.", e);
            }

        } else {

            // Array "class[][]"
            if (!clazz.isArray()) {
                throw new OgdlSyntaxException(ev, "no array type. " + clazz);
            }

            int next_off = OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isCharAt(ptn, next_off, '{')) {

                off.set(next_off);

                // 数値領域（等差数列や時間領域）リストで生成する。
                List list = (List) OgdlCollectionParser.evaluateEncloseProgressionCollection(ev.get('{', '}', new LinkedList()));

                // 配列への格納
                try {
                    obj = HSet.toArray(list, clazz.getComponentType());
                } catch (IllegalArgumentException e) {
                    throw new OgdlSyntaxException(ev, "array item type mismatch.", e);
                }

            } else {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }

        return obj;
    }

    static Object evaluateConstructor(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        final Class[] argsType = evaluateEncloseTypeArray(ev.get('(', ')'));
        try {
            return BeanIntrospectHelper.getConstructor(type, argsType);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    static Object evaluateMethod(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        if (!OgdlSyntax.isCharAt(ev, '#')) {
            throw new OgdlSyntaxException(ev, "method separator err. ");
        }
        ev.off.increment();
        final String methodName = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
        final Class[] argsType = evaluateEncloseTypeArray(ev.get('(', ')'));
        try {
            return BeanIntrospectHelper.getMethod(type, methodName, argsType);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    static Object evaluateField(OgdlEvent ev) {
        final Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        if (!OgdlSyntax.isCharAt(ev, '#')) {
            throw new OgdlSyntaxException(ev, "field separator err. ");
        }
        ev.off.increment();
        final String fieldName = OgdlSyntax.cutMemberName(ev.ptn, ev.off);
        try {
            return BeanIntrospectHelper.getField(type, fieldName);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    /*
     * private
     */

    /* コンマ区切りのクラス名のリスト書式のクラス配列として返却します。 */
    private static Class[] evaluateEncloseTypeArray(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char open = ev.open;
        final char close = ev.close;

        if (!OgdlSyntax.isCharAt(ptn, off.get(), open)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        off.increment();

        ArrayList list = new ArrayList();
        OgdlSyntax.siftSpace(ev);
        if (OgdlSyntax.isNotClose(ptn, off, close)) {

            while (off.get() < ptn.length()) {

                list.add(OgdlPrimitiveParser.evaluateClassByExtendName(ev));

                OgdlSyntax.siftSpace(ev);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not comma.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        Class[] types = (Class[]) list.toArray(new Class[0]);
        off.increment();// next index
        return types;

    }

    /* 配列生成式の配列数の初期化部分（例：[2][4]）を解析して返却します。 */
    private static Integer[] evaluateDimensionArraySizes(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        final Collection output = new LinkedList();
        while (OgdlSyntax.isCharAt(ptn, off.get(), '[')) {
            off.increment();// + '['
            OgdlSyntax.siftSpace(ev);
            output.add(OgdlPrimitiveParser.evaluateInteger(ev));
            OgdlSyntax.validIndex(ev, ptn, off);
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, ']')) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
            off.increment();// + ']'
        }
        return (Integer[]) output.toArray(new Integer[output.size()]);
    }

    /* 前後のスペースをスキップして。式を実行し式の結果の値を返却します。 */
    private static Object evaluateSkipSpace(OgdlEvent ev) {
        OgdlSyntax.siftSpace(ev);
        Object o = OgdlRuntime.evaluate(ev);
        OgdlSyntax.siftSpace(ev);
        return o;
    }

    private static Object readFieldOrProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {
        try {
            PropertyDescriptor pd = BeanIntrospectHelper.getPropertyDescriptor(clazz, memberName);
            Object r = null;
            if (pd == null) {
                // is Array.length
                if (OgdlSyntax.isArraylength(clazz, memberName)) {
                    r = Boxing.box(Array.getLength(o));
                } else {
                    r = BeanIntrospectHelper.getFieldValue(clazz, o, memberName);
                }
            } else {
                r = BeanIntrospectHelper.getProperty(o, pd.getReadMethod());
            }
            return r;

        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    private static Object readStaticFieldValue(OgdlEvent ev, String memberName, Object o) {
        try {
            if (o instanceof Class) {
                // static field
                return BeanIntrospectHelper.getFieldValue((Class) o, null, memberName);
            } else {
                throw new OgdlSyntaxException(ev, "is not Class class.");
            }
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    private static Object readIndexedValue(OgdlEvent ev, String ptn, OgdlParseIndex off) {
        off.increment();
        Object inx = evaluateSkipSpace(ev);
        if (!OgdlSyntax.isCharAt(ptn, off.get(), ']')) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        if (!(inx instanceof Integer)) {
            throw new OgdlSyntaxException(ev, HLog.clazz("not indexing type. ", inx));
        }
        off.increment();
        return inx;
    }

    private static Object readIndexingKeys(OgdlEvent ev, String ptn, OgdlParseIndex off) {
        off.increment();// add indexer
        Object key = evaluateSkipSpace(ev);
        if (!OgdlSyntax.isCharAt(ptn, off.get(), ']')) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        off.increment();// add indexer
        return key;
    }

    private static int toKeyAsIndex(OgdlEvent ev, Object key) {
        if (!(key instanceof Integer)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        return ((Integer) key).intValue();
    }

    private static Object readIndexingValue(OgdlEvent ev, Object key, Object o) {
        // index or map
        Object r = null;
        if (o instanceof Map) {
            r = ((Map) o).get(key);
        } else if (o instanceof List) {
            int index = toKeyAsIndex(ev, key);
            r = ((List) o).get(index);
        } else if (HEval.isArray(o)) {
            int index = toKeyAsIndex(ev, key);
            r = Array.get(o, index);
        } else if (o instanceof CharSequence) {
            int index = toKeyAsIndex(ev, key);
            r = Boxing.box(((CharSequence) o).charAt(index));
        } else {
            throw new OgdlSyntaxException(ev, HLog.clazz("not indexing type. ", o));
        }
        return r;
    }

    private static Object readIndexPropertyOrFieldOrProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {
        try {
            final String ptn = ev.ptn;
            final OgdlParseIndex off = ev.off;
            IndexedPropertyDescriptor ipd = BeanIntrospectHelper.getIndexedPropertyDescriptor(clazz, memberName);
            Object r = null;
            if (ipd != null) {
                // get index
                Object inx = readIndexedValue(ev, ptn, off);
                r = BeanIntrospectHelper.getIndexedProperty(o, ipd.getIndexedReadMethod(), (Integer) inx);
            } else {
                r = readFieldOrProperty(ev, memberName, o, clazz);
            }
            return r;
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    private static Object invokeMethod(OgdlEvent ev, String memberName, Object o, Class clazz) {
        try {
            Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
            return BeanIntrospectHelper.invokeMethod(clazz, o, memberName, paramValues);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }

    }

    private static Object invokeStaticMethod(OgdlEvent ev, String memberName, Object o) {
        try {
            Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
            if (o instanceof Class) {
                // static method
                return BeanIntrospectHelper.invokeMethod((Class) o, null, memberName, paramValues);
            } else {
                throw new OgdlSyntaxException(ev, "is not Class class.");
            }
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    /* メソッド引数の書式として解析してオブジェクト型配列として返却します。 イベントに格納されるリテラル文字で区切られる前提で解析されます。 */
    private static Object[] evaluateEncloseArguments(OgdlEvent ev) {
        Object val = OgdlParser.evaluateEncloseCollection(ev.get(new LinkedList()));
        return ((List) val).toArray();
    }

    /* プロパティ設定式として解析して、そのプロパティを格納します。 */
    private static void writeInitProperties(OgdlEvent ev, Object obj) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char open = ev.open;
        final char close = ev.close;
        final Class clazz = obj.getClass();

        if (!OgdlSyntax.isCharAt(ptn, off.get(), open)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        off.increment();

        if (OgdlSyntax.isNotClose(ptn, off, close)) {
            while (off.get() < ptn.length()) {

                OgdlSyntax.siftSpace(ev);

                // get member name
                String memberName = OgdlSyntax.cutMemberName(ptn, off);

                // method or property or field
                if (OgdlSyntax.isCharAt(ptn, off.get(), '(')) {// method 'method(arg..)'
                    invokeInitMethod(ev, memberName, obj, clazz);
                } else if (OgdlSyntax.isCharAt(ptn, off.get(), '[')) {// index property
                    invokeInitIndexedProperty(ev, memberName, obj, clazz);
                } else {// property
                    invokeInitProperty(ev, memberName, obj, clazz);
                }

                OgdlSyntax.siftSpace(ev);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }

                // map entry separator
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not comma.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        off.increment();// next index
        OgdlSyntax.siftSpace(ev);

    }

    private static void invokeInitIndexedProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        try {
            IndexedPropertyDescriptor ipd = BeanIntrospectHelper.getIndexedPropertyDescriptor(clazz, memberName);
            if (ipd != null) {
                // get index
                Object inx = readIndexedValue(ev, ptn, off);

                OgdlSyntax.siftSpace(ev);
                // name value separator
                if (OgdlSyntax.isNotMapSeparator(ptn, off)) {
                    throw new OgdlSyntaxException(ev, "is not equal.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);

                BeanIntrospectHelper.setIndexedProperty(o, ipd.getIndexedWriteMethod(), (Integer) inx, OgdlRuntime.evaluate(ev));

            } else {
                throw new OgdlSyntaxException(ev, "is not Indexed property.");
            }
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

    private static void invokeInitMethod(OgdlEvent ev, String memberName, Object o, Class clazz) {
        try {
            Object[] paramValues = evaluateEncloseArguments(ev.get('(', ')'));
            BeanIntrospectHelper.invokeMethod(clazz, o, memberName, paramValues);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }

    }

    private static void invokeInitProperty(OgdlEvent ev, String memberName, Object o, Class clazz) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        try {
            PropertyDescriptor pd = BeanIntrospectHelper.getPropertyDescriptor(clazz, memberName);
            if (pd != null) {

                OgdlSyntax.siftSpace(ev);
                // name value separator
                if (OgdlSyntax.isNotMapSeparator(ptn, off)) {
                    throw new OgdlSyntaxException(ev, "is not equal.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);

                // set property
                BeanIntrospectHelper.setProperty(o, pd.getWriteMethod(), OgdlRuntime.evaluate(ev));

            } else {
                throw new OgdlSyntaxException(ev, "is not property.");
            }
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }
}
