/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Map;

/**
 * OGDLの利用を補助するヘルパーメソッド群を提供します。
 */
public class OgdlHelper {

    /**
     * OgdlContext を生成して返却します。
     * 
     * @return OgdlContext
     */
    public static OgdlContext getOgdlContext() {
        return new SimpleOgdlContext();
    }

    /**
     * 値を格納するマップを格納して OgdlContext を生成して返却します。
     * 
     * @param values
     *            値を格納するマップ
     * @return OgdlContext
     */
    public static OgdlContext getOgdlContext(Map values) {
        return new SimpleOgdlContext(values);
    }

    /**
     * 値を格納するマップとクラス参照を格納するマップを格納して OgdlContext を生成して返却します。
     * 
     * @param values
     *            値を格納するマップ
     * @param classes
     *            クラス参照を格納するマップ
     * @return OgdlContext
     */
    public static OgdlContext getOgdlContext(Map values, Map classes) {
        return new SimpleOgdlContext(values, classes);
    }

    /**
     * 値を格納するマップとクラス参照を格納するマップ、組込み関数を格納するマップを格納して OgdlContext を生成して返却します。
     * 
     * @param values
     *            値を格納するマップ
     * @param classes
     *            クラス参照を格納するマップ
     * @param functions
     *            組込み関数を格納するマップ
     * @return OgdlContext
     */
    public static OgdlContext getOgdlContext(Map values, Map classes, Map functions) {
        return new SimpleOgdlContext(values, classes, functions);
    }

    /**
     * クラスに定義されている public かつ static なメソッドを、メソッド名でグルーピングしたマップを返却します。<br>
     * マップのキーはメソッド名で、値は java.lang.reflect.Method 型の要素を持つコレクションです。
     * 
     * @param clazz
     *            組込み関数を定義sるクラス
     * @return 組込み関数を格納するマップ
     */
    public static Map getFunctionsMap(Class clazz) {
        return BeanIntrospectHelper.getStaticMethodGroup(clazz);
    }

}
