/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

/**
 * OGDLの実行コンテキストを提供します。
 */
public interface OgdlContext {

    /*
     * Import
     */

    /**
     * クラスをインポートします。<br>
     * 式からパッケージ名を除いたクラス名で参照することが出来る様になります。
     * 
     * @param clazz
     *            インポートするクラス
     * @return インポートが上書きされた場合は true
     */
    boolean addImport(Class clazz);

    /**
     * クラス名に対応するクラスを返却します。
     * 
     * @param shortName
     *            式からパッケージ名を除いたクラス名
     * @return クラス、該当するクラスが無い場合は null
     */
    Class forImport(String shortName);

    /*
     * Function
     */

    /**
     * 組込み関数を一括で登録します。<br>
     * 組込み関数は public かつ static な java.lang.reflect.Method 型で指定します。<br>
     * マップの値は Method 型の要素を持つコレクションの必要があります。<br>
     * 同一の関数名が既にある場合は上書きされます。
     * 
     * @param prefix
     *            関数名の接頭辞
     * @param functionsMap
     *            名前をキーとして、組込み関数を値とするマップ
     */
    void addFunctionsMap(String prefix, Map functionsMap);

    /**
     * 接頭辞と関数名を含む完全名に対応する組込み関数を返却します。<br>
     * コレクションの要素は public かつ static な java.lang.reflect.Method 型です。
     * 
     * @param name
     *            接頭辞と関数名を含む完全名
     * @return 組込み関数、該当する関数が無い場合は null
     */
    Collection forFunctions(String name);

    /*
     * Attribute
     */

    /**
     * 全ての属性名の反復子を返却します。
     * 
     * @return 属性名の反復子
     */
    Iterator attributeNames();

    /**
     * 引数の名前を持つ属性値を返却します。 <br>
     * 指定された属性が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    Object getAttribute(String name);

    /**
     * 引数の名前で属性を格納します。
     * 
     * @param name
     *            属性名
     * @param value
     *            属性値
     */
    void setAttribute(String name, Object value);

    /**
     * 引数の名前を持つ属性値を削除して既存の値を返却します。 <br>
     * 既存の属性が存在しない場合は null が返されます。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    Object removeAttribute(String name);

    /**
     * 引数の名前が存在する場合は true を返却します。
     * 
     * @param name
     *            属性名
     * @return 引数の名前が存在する場合は true
     */
    boolean containsAttribute(String name);

}
