/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.SortedSet;

/**
 * コレクションの構文解析機能を提供します。
 */
class OgdlCollectionParser {

    static Object evaluateCollection(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedCollection(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableCollection(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedCollection(ev);
        }
        return evaluateUnmodifiableCollection(ev);
    }

    static Object evaluateSortedMap(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedSortedMap(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableSortedMap(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedSortedMap(ev);
        }
        return evaluateUnmodifiableSortedMap(ev);
    }

    static Object evaluateMap(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedMap(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableMap(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedMap(ev);
        }
        return evaluateUnmodifiableMap(ev);
    }

    static Object evaluateSortedSet(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedSortedSet(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableSortedSet(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedSortedSet(ev);
        }
        return evaluateUnmodifiableSortedSet(ev);
    }

    static Object evaluateSet(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedSet(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableSet(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedSet(ev);
        }
        return evaluateUnmodifiableSet(ev);
    }

    static Object evaluateList(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final int begin = ev.off.get();

        if (isSynchronizedView(off, ptn, begin)) {
            return evaluateSynchronizedList(ev);
        }
        if (isUnmodifiableView(off, ptn, begin)) {
            return evaluateUnmodifiableList(ev);
        }
        if (isUndecoratedView(off, ptn, begin)) {
            return evaluateUndecoratedList(ev);
        }
        return evaluateUnmodifiableList(ev);
    }

    /* 同期コレクションの指定の場合は true を返却します。 */
    private static boolean isSynchronizedView(OgdlParseIndex off, String ptn, int begin) {
        if (ptn.startsWith("syn:", begin)) {
            off.add(4);
            return true;
        }
        return false;
    }

    /* 不変コレクションの指定の場合は true を返却します。 */
    private static boolean isUnmodifiableView(OgdlParseIndex off, String ptn, int begin) {
        if (ptn.startsWith("unm:", begin)) {
            off.add(4);
            return true;
        }
        return false;
    }

    /* 非装飾コレクションの指定の場合は true を返却します。 */
    private static boolean isUndecoratedView(OgdlParseIndex off, String ptn, int begin) {
        if (ptn.startsWith("und:", begin)) {
            off.add(4);
            return true;
        }
        return false;
    }

    /* コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateEncloseCollection(OgdlEvent ev) {
        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char open = ev.open;

        if (!OgdlSyntax.isCharAt(ptn, off.get(), open)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        off.increment();
        return evaluateCloseCollection(ev);
    }

    /* 開始文字を検証せずに、コンマ区切りのコレクション書式として解析して返却します。 */
    static Collection evaluateCloseCollection(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char close = ev.close;

        final Collection output = (Collection) ev.target;

        if (OgdlSyntax.isNotClose(ptn, off, close)) {
            while (off.get() < ptn.length()) {
                OgdlSyntax.siftSpace(ev);
                output.add(OgdlRuntime.evaluate(ev));
                OgdlSyntax.validIndex(ev, ptn, off);
                OgdlSyntax.siftSpace(ev);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                off.increment();// + ','
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        off.increment();// next index
        return output;
    }

    /* マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateEncloseMap(OgdlEvent ev) {
        final OgdlParseIndex off = ev.off;
        final char open = ev.open;

        if (!OgdlSyntax.isCharAt(ev.ptn, off.get(), open)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }
        off.increment();
        return evaluateCloseMap(ev);
    }

    /* 開始文字を検証せずに、マップ書式として解析して指定のマップに追加して返却します。 */
    static Map evaluateCloseMap(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char close = ev.close;
        final Map output = (Map) ev.target;

        if (OgdlSyntax.isNotClose(ptn, off, close)) {
            while (off.get() < ev.len) {
                OgdlSyntax.siftSpace(ev);
                // get key
                Object rKey = OgdlRuntime.evaluate(ev);
                OgdlSyntax.siftSpace(ev);
                OgdlSyntax.validIndex(ev, ptn, off);

                // key value separator
                if (OgdlSyntax.isNotMapSeparator(ptn, off)) {
                    throw new OgdlSyntaxException(ev, "syntax err.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);
                // get value
                Object val = OgdlRuntime.evaluate(ev);
                OgdlSyntax.siftSpace(ev);
                // add Entry
                output.put(rKey, val);
                if (OgdlSyntax.isIndex(ptn, off) && OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }
                // map entry separator
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                off.increment();
                OgdlSyntax.siftSpace(ev);
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        off.increment();// next index
        return output;

    }

    /* 数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として引数のリストに格納します。 */
    static Collection evaluateEncloseProgressionCollection(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char open = ev.open;

        if (!OgdlSyntax.isCharAt(ptn, off.get(), open)) {
            throw new OgdlSyntaxException(ev, "syntax err.");
        }

        off.increment();
        return evaluateCloseProgressionCollection(ev);
    }

    /* 開始文字を検証せずに、数値領域（等差数列や時間領域）式を有効とするコンマ区切りのリスト書式として解析して引数のリストに格納して返却します。 */
    static Collection evaluateCloseProgressionCollection(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;
        final char close = ev.close;
        final Collection output = (Collection) ev.target;

        OgdlSyntax.siftSpace(ev);
        if (OgdlSyntax.isNotClose(ptn, off, close)) {
            while (off.get() < ptn.length()) {
                OgdlSyntax.siftSpace(ev);
                final Object e = OgdlRuntime.evaluate(ev);

                OgdlSyntax.validIndex(ev, ptn, off);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    output.add(e);
                    break;
                }

                // is range 'from:to'
                if (OgdlSyntax.isCharAt(ptn, off.get(), ':')) {
                    off.increment();
                    final Object to = OgdlRuntime.evaluate(ev);
                    addListProgressionValue(ev, output, e, to);
                } else {
                    output.add(e);
                }
                OgdlSyntax.validIndex(ev, ptn, off);

                OgdlSyntax.siftSpace(ev);
                if (OgdlSyntax.isClose(ptn, off, close)) {
                    break;
                }
                if (OgdlSyntax.isNotSeparator(ptn, off.get())) {
                    throw new OgdlSyntaxException(ev, "is not separator.");
                }
                off.increment();// + ','
            }
            OgdlSyntax.siftSpace(ev);
            if (OgdlSyntax.isNotClose(ptn, off, close)) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
        }
        off.increment();// next index
        return output;

    }

    /* 数値領域（等差数列や時間領域）式を解析して引数のリストに格納し次の式の開始位置を返却します。 */
    private static void addListProgressionValue(OgdlEvent ev, Collection output, Object from, Object to) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        // get difference "from:to(difference)"
        String difference = null;
        OgdlSyntax.siftSpace(ev);
        if (OgdlSyntax.isCharAt(ptn, off.get(), '(')) {
            off.increment();
            OgdlSyntax.siftSpace(ev);
            difference = OgdlSyntax.cutDifferenceString(ptn, off);
            // is delimiter
            OgdlSyntax.siftSpace(ev);
            if (!OgdlSyntax.isCharAt(ptn, off.get(), ')')) {
                throw new OgdlSyntaxException(ev, "syntax err.");
            }
            off.increment();
            addNumberDomain(ev, output, from, to, difference);
        } else {
            addNumberDomain(ev, output, from, to);
        }
    }

    /*
     * evaluate collections
     */

    /* マップ生成式を実行して値を返却します。 */
    static Object evaluateDefaultMap(OgdlEvent ev) {
        final Map output = new LinkedHashMap();
        return Collections.unmodifiableMap(evaluateEncloseMap(ev.get('{', '}', output)));
    }

    /* リスト生成式を実行して値を返却します。 */
    static Object evaluateDefaultList(OgdlEvent ev) {
        final List output = new ArrayList();
        return Collections.unmodifiableList((List) evaluateEncloseProgressionCollection(ev.get('[', ']', output)));
    }

    /* セット生成式を実行して値を返却します。 */
    static Object evaluateDefaultSet(OgdlEvent ev) {
        final Set output = new LinkedHashSet();
        return Collections.unmodifiableSet((Set) evaluateEncloseProgressionCollection(ev.get('<', '>', output)));
    }

    /* 配列生成式を実行して値を返却します。 */
    static Object evaluateArray(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        Class clazz = null;
        if (OgdlSyntax.isBeginMember(ptn, off.get())) {
            clazz = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        } else {
            clazz = Object.class;
        }
        OgdlSyntax.siftSpace(ev);

        Object output = null;

        // 範囲書式のリストで生成する。
        List list = (List) evaluateEncloseProgressionCollection(ev.get('{', '}', new LinkedList()));

        // 配列への格納
        try {
            output = HSet.toArray(list, clazz);
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(ev, "array item type mismatch.", e);
        }

        return output;
    }

    /*
     * List
     */

    /* リスト生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedList(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        List output = null;
        if (OgdlSyntax.isBeginMember(ptn, off.get())) {

            Object val = newCollectionInstance(ev);
            try {
                output = (List) val;
            } catch (ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not list type. " + val.getClass(), e);
            }

        } else {
            output = new ArrayList();
        }

        OgdlSyntax.siftSpace(ev);

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期リストの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedList(OgdlEvent ev) {
        Object val = evaluateUndecoratedList(ev);
        return Collections.synchronizedList((List) val);
    }

    /* 変更不可のリスト生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableList(OgdlEvent ev) {
        Object val = evaluateUndecoratedList(ev);
        return Collections.unmodifiableList((List) val);
    }

    /*
     * Set
     */

    /* セット生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSet(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        Set output = null;
        if (OgdlSyntax.isBeginMember(ptn, off.get())) {

            Object val = newCollectionInstance(ev);
            try {
                output = (Set) val;
            } catch (ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not set type. " + val.getClass(), e);
            }

        } else {
            output = new LinkedHashSet();
        }

        OgdlSyntax.siftSpace(ev);

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期セットの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSet(OgdlEvent ev) {
        Object val = evaluateUndecoratedSet(ev);
        Set set = Collections.synchronizedSet((Set) val);
        return set;
    }

    /* 変更不可のセット生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSet(OgdlEvent ev) {
        Object val = evaluateUndecoratedSet(ev);
        Set set = Collections.unmodifiableSet((Set) val);
        return set;
    }

    /*
     * SortedSet
     */

    /* ソートセットの生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSortedSet(OgdlEvent ev) {
        Object val = evaluateUndecoratedSet(ev);
        try {
            SortedSet sortedSet = Collections.synchronizedSortedSet((SortedSet) val);
            return sortedSet;
        } catch (ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not SortedSet type. " + val.getClass(), e);
        }

    }

    /* 同期ソートセットの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSortedSet(OgdlEvent ev) {
        Object val = evaluateUndecoratedSortedSet(ev);
        return Collections.synchronizedSortedSet((SortedSet) val);
    }

    /* 変更不可のソートセット生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSortedSet(OgdlEvent ev) {
        Object val = evaluateUndecoratedSortedSet(ev);
        return Collections.unmodifiableSortedSet((SortedSet) val);
    }

    /*
     * Map
     */

    /* マップ生成式を実行して値を返却します。 */
    private static Map evaluateUndecoratedMap(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        Map output = null;
        if (OgdlSyntax.isBeginMember(ptn, off.get())) {

            Object val = newCollectionInstance(ev);
            try {
                output = (Map) val;
            } catch (ClassCastException e) {
                throw new OgdlSyntaxException(ev, "not map type. " + val.getClass(), e);
            }

        } else {
            output = new LinkedHashMap();
        }

        OgdlSyntax.siftSpace(ev);

        return evaluateEncloseMap(ev.get('{', '}', output));

    }

    /* 同期マップの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedMap(OgdlEvent ev) {
        return Collections.synchronizedMap(evaluateUndecoratedMap(ev));

    }

    /* 変更不可のマップ生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableMap(OgdlEvent ev) {
        return Collections.unmodifiableMap(evaluateUndecoratedMap(ev));

    }

    /*
     * SortedMap
     */

    /* ソートマップの生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedSortedMap(OgdlEvent ev) {
        Object val = evaluateUndecoratedMap(ev);
        try {
            SortedMap sortedMap = Collections.synchronizedSortedMap((SortedMap) val);
            return sortedMap;
        } catch (ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not SortedMap type. " + val.getClass(), e);
        }

    }

    /* 同期ソートマップの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedSortedMap(OgdlEvent ev) {
        Object val = evaluateUndecoratedSortedMap(ev);
        return Collections.synchronizedSortedMap((SortedMap) val);
    }

    /* 変更不可のソートマップ生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableSortedMap(OgdlEvent ev) {
        Object val = evaluateUndecoratedSortedMap(ev);
        SortedMap sortedMap = Collections.unmodifiableSortedMap((SortedMap) val);
        return sortedMap;
    }

    /*
     * Collection
     */

    /* コレクション生成式を実行して値を返却します。 */
    private static Object evaluateUndecoratedCollection(OgdlEvent ev) {

        final String ptn = ev.ptn;
        final OgdlParseIndex off = ev.off;

        Collection output = null;
        if (!OgdlSyntax.isBeginMember(ptn, off.get())) {
            throw new OgdlSyntaxException(ev, "collection class is empty." + ptn);
        }

        Object val = newCollectionInstance(ev);
        try {
            output = (Collection) val;
        } catch (ClassCastException e) {
            throw new OgdlSyntaxException(ev, "not collection type. " + val.getClass(), e);
        }

        OgdlSyntax.siftSpace(ev);

        return evaluateEncloseProgressionCollection(ev.get('{', '}', output));
    }

    /* 同期コレクションの生成式を実行して値を返却します。 */
    private static Object evaluateSynchronizedCollection(OgdlEvent ev) {
        Object val = evaluateUndecoratedCollection(ev);
        Collection coll = Collections.synchronizedCollection((Collection) val);
        return coll;
    }

    /* 変更不可のコレクション生成式を実行して値を返却します。 */
    private static Object evaluateUnmodifiableCollection(OgdlEvent ev) {
        Object val = evaluateUndecoratedCollection(ev);
        Collection coll = Collections.unmodifiableCollection((Collection) val);
        return coll;
    }

    /*
     * data domain
     */

    /* 1.0 の BigDecimal */
    private static final BigDecimal BIGDECIMAL_ONE = new BigDecimal(BigInteger.ONE, 0);

    /* 数値領域（等差数列や時間領域）をデフォルトの公差で追加して返却します。 */
    private static void addNumberDomain(OgdlEvent ev, Collection coll, Object from, Object to) {
        if (from instanceof Number && to instanceof Number) {
            if (from instanceof Byte && to instanceof Byte) {
                HSet.arithmeticProgression(coll, (Byte) from, (Byte) to, (byte) 1);
            } else if (from instanceof Short && to instanceof Short) {
                HSet.arithmeticProgression(coll, (Short) from, (Short) to, (short) 1);
            } else if (from instanceof Integer && to instanceof Integer) {
                HSet.arithmeticProgression(coll, (Integer) from, (Integer) to, 1);
            } else if (from instanceof Long && to instanceof Long) {
                HSet.arithmeticProgression(coll, (Long) from, (Long) to, 1L);
            } else if (from instanceof Float && to instanceof Float) {
                HSet.arithmeticProgression(coll, (Float) from, (Float) to, 1);
            } else if (from instanceof Double && to instanceof Double) {
                HSet.arithmeticProgression(coll, (Double) from, (Double) to, 1L);
            } else if (from instanceof BigInteger && to instanceof BigInteger) {
                HSet.arithmeticProgression(coll, (BigInteger) from, (BigInteger) to, BigInteger.ONE);
            } else if (from instanceof BigDecimal && to instanceof BigDecimal) {
                HSet.arithmeticProgression(coll, (BigDecimal) from, (BigDecimal) to, BIGDECIMAL_ONE);
            } else {
                throw new OgdlSyntaxException(ev, "no domain type. ");
            }
        } else if (from instanceof Character && to instanceof Character) {
            HSet.arithmeticProgression(coll, (Character) from, (Character) to, 1);
        } else if (from instanceof Date && to instanceof Date) {
            HSet.dateTimeDomain(coll, (Date) from, (Date) to, Calendar.DATE, 1);
        } else {
            throw new OgdlSyntaxException(ev, "no domain type. ");
        }
    }

    /* 数値領域（等差数列や時間領域）を指定の公差で追加して返却します。 */
    private static void addNumberDomain(OgdlEvent ev, Collection coll, Object from, Object to, String difference) {
        try {
            if (from instanceof Number && to instanceof Number) {
                if (from instanceof Byte && to instanceof Byte) {
                    HSet.arithmeticProgression(coll, (Byte) from, (Byte) to, Byte.parseByte(difference));
                } else if (from instanceof Short && to instanceof Short) {
                    HSet.arithmeticProgression(coll, (Short) from, (Short) to, Short.parseShort(difference));
                } else if (from instanceof Integer && to instanceof Integer) {
                    HSet.arithmeticProgression(coll, (Integer) from, (Integer) to, Integer.parseInt(difference));
                } else if (from instanceof Long && to instanceof Long) {
                    HSet.arithmeticProgression(coll, (Long) from, (Long) to, Long.parseLong(difference));
                } else if (from instanceof Float && to instanceof Float) {
                    HSet.arithmeticProgression(coll, (Float) from, (Float) to, Integer.parseInt(difference));
                } else if (from instanceof Double && to instanceof Double) {
                    HSet.arithmeticProgression(coll, (Double) from, (Double) to, Long.parseLong(difference));
                } else if (from instanceof BigInteger && to instanceof BigInteger) {
                    HSet.arithmeticProgression(coll, (BigInteger) from, (BigInteger) to, new BigInteger(difference));
                } else if (from instanceof BigDecimal && to instanceof BigDecimal) {
                    HSet.arithmeticProgression(coll, (BigDecimal) from, (BigDecimal) to, new BigDecimal(difference));
                } else {
                    throw new OgdlSyntaxException(ev, "no domain type. ");
                }
            } else if (from instanceof Character && to instanceof Character) {
                HSet.arithmeticProgression(coll, (Character) from, (Character) to, Integer.parseInt(difference));
            } else if (from instanceof Date && to instanceof Date) {
                int[] differences = toDateDifferences(ev, difference);
                HSet.dateTimeDomain(coll, (Date) from, (Date) to, differences[0], differences[1]);
            } else {
                throw new OgdlSyntaxException(ev, "no domain type. ");
            }
        } catch (NumberFormatException e) {
            throw new OgdlSyntaxException(ev, "number syntax err.", e);
        }
    }

    private static int[] toDateDifferences(OgdlEvent ev, String difference) {

        String[] splits = difference.split("\\.");
        if (splits.length != 2) {
            throw new OgdlSyntaxException(ev, HLog.log("range difference syntax err.", difference));
        }
        int[] differences = new int[] { Calendar.DATE, 1 };
        differences[1] = Integer.parseInt(splits[1]);
        Integer ofield = (Integer) getCalendarConstants(ev).get(splits[0]);
        if (ofield != null) {
            differences[0] = ofield.intValue();
        } else {
            throw new OgdlSyntaxException(ev, HLog.log("not is calendar field.", splits[0]));
        }
        return differences;
    }

    private static Object newCollectionInstance(OgdlEvent ev) {

        Class type = OgdlPrimitiveParser.evaluateClassByExtendName(ev);
        Object output = null;
        try {
            output = BeanIntrospectHelper.newInstance(type);
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
        return output;

    }

    private static Map calendarConstants;

    private static Map getCalendarConstants(OgdlEvent ev) {
        final Map consts = calendarConstants;
        try {
            return (consts != null) ? consts : (calendarConstants = BeanIntrospectHelper.getConstantFieldMap(Calendar.class));
        } catch (OgdlSyntaxException e) {
            throw e.throwFor(ev);
        }
    }

}
