/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.lang.reflect.Array;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

/**
 * 煩雑なデータ集合の構築を簡易化する為のヘルパーメソッド群を提供します。
 */
class HSet {

    /**
     * コレクションを指定されたクラスの配列に変換します、プリミティブ型の生成を可能とします。
     * 
     * @param coll
     *            生成基のコレクション
     * @param type
     *            配列の基クラス
     * @return コレクションと同じ要素を持つ配列
     */
    static Object toArray(Collection coll, Class type) {
        final Object a = Array.newInstance(type, coll.size());
        if (type.isPrimitive()) {
            int i = 0;
            Iterator iter = coll.iterator();
            while (iter.hasNext()) {
                Array.set(a, i++, iter.next());
            }
        } else {
            coll.toArray((Object[]) a);
        }
        return a;
    }

    /**
     * コレクションに初項から末項以下のフィールドの間隔を持つ時刻領域を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行いフィールドと数値分の生成した日付をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param field
     *            Calendar.add(int, int) の第一引数に渡す Calendar クラスの定数
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection dateTimeDomain(Collection coll, Date from, Date to, int field, int difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        final Calendar fc = Calendar.getInstance();
        final Calendar tc = Calendar.getInstance();
        fc.setTimeInMillis(from.getTime());
        tc.setTimeInMillis(to.getTime());
        if (HEval.isLess(from, to)) {
            // ascending order
            final long maxTime = tc.getTimeInMillis();
            while (fc.getTimeInMillis() <= maxTime) {
                coll.add(fc.getTime());
                fc.add(field, difference);// +difference
            }
        } else {
            // descending order
            final long minTime = tc.getTimeInMillis();
            while (fc.getTimeInMillis() >= minTime) {
                coll.add(fc.getTime());
                fc.add(field, -difference);// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した文字をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Character from, Character to, int difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            char e = from.charValue();
            final char maxValue = to.charValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            char e = from.charValue();
            final char minValue = to.charValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Byte from, Byte to, byte difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            byte e = from.byteValue();
            final byte maxValue = to.byteValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            byte e = from.byteValue();
            final byte minValue = to.byteValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Short from, Short to, short difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            short e = from.shortValue();
            final short maxValue = to.shortValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            short e = from.shortValue();
            final short minValue = to.shortValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Integer from, Integer to, int difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            int e = from.intValue();
            final int maxValue = to.intValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            int e = from.intValue();
            final int minValue = to.intValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Long from, Long to, long difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            long e = from.longValue();
            final long maxValue = to.longValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            long e = from.longValue();
            final long minValue = to.longValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Float from, Float to, int difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isInfiniteOrNaN(from) || HEval.isInfiniteOrNaN(to)) {
            throw new IllegalArgumentException("Infinite or NaN");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            float e = from.floatValue();
            final float maxValue = to.floatValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            float e = from.floatValue();
            final float minValue = to.floatValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, Double from, Double to, long difference) {
        if (difference <= 0) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isInfiniteOrNaN(from) || HEval.isInfiniteOrNaN(to)) {
            throw new IllegalArgumentException("Infinite or NaN");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        if (HEval.isLess(from, to)) {
            // ascending order
            double e = from.doubleValue();
            final double maxValue = to.doubleValue();

            while (e <= maxValue) {
                coll.add(Boxing.box(e));
                e += difference;// +difference
            }
        } else {
            // descending order
            double e = from.doubleValue();
            final double minValue = to.doubleValue();

            while (e >= minValue) {
                coll.add(Boxing.box(e));
                e -= difference;// -difference
            }
        }
        return coll;
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, BigInteger from, BigInteger to, BigInteger difference) {
        if (HEval.isLess(difference, BigInteger.ZERO)) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        BigInteger e = from;
        if (HEval.isLess(from, to)) {
            // ascending order
            while (0 > e.compareTo(to)) {
                coll.add(e);
                e = e.add(difference);// +difference
            }
        } else {
            // descending order
            while (0 < e.compareTo(to)) {
                coll.add(e);
                e = e.subtract(difference);// -difference
            }
        }
        return coll;
    }

    /* 0.0 の BigDecimal */
    private static final BigDecimal BIGDECIMAL_ZERO = new BigDecimal(BigInteger.ZERO, 0);

    /**
     * コレクションに初項から末項以下の等差数列を追加します。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, BigDecimal from, BigDecimal to, BigInteger difference) {
        return arithmeticProgression(coll, from, to, new BigDecimal(difference));
    }

    /**
     * コレクションに初項から末項以下の等差数列を追加します、任意精度の少数演算をサポートします。<br>
     * (from > to) の場合は加算し以外は減算を行い生成した数値をコレクションに格納します。
     * 
     * @param coll
     *            データを格納するコレクション
     * @param from
     *            初項
     * @param to
     *            末項
     * @param difference
     *            公差
     * @return 引数のコレクションの参照
     */
    static Collection arithmeticProgression(Collection coll, BigDecimal from, BigDecimal to, BigDecimal difference) {
        if (HEval.isLess(difference, BIGDECIMAL_ZERO)) {
            throw new IllegalArgumentException("difference <= 0");
        }
        if (HEval.isEqual(from, to)) {
            return coll;
        }
        BigDecimal e = from;
        if (HEval.isLess(from, to)) {
            // ascending order
            while (0 > e.compareTo(to)) {
                coll.add(e);
                e = e.add(difference);// +difference
            }
        } else {
            // descending order
            while (0 < e.compareTo(to)) {
                coll.add(e);
                e = e.subtract(difference);// -difference
            }
        }
        return coll;
    }

}
