/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

/**
 * 値を評価するヘルパーメソッド群を提供します。
 */
class HEval {

    /**
     * 引数が null または空の文字シーケンスであるかを評価します。
     * 
     * @param cs
     *            評価する文字シーケンス
     * @return 引数が null または空の文字シーケンスの場合のみ true
     */
    static boolean isEmpty(CharSequence cs) {
        return (cs == null || cs.length() == 0);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    static boolean isOrNull(Object o, Object o2) {
        return (o == null || o2 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    static boolean isOrNull(Object o, Object o2, Object o3) {
        return (o == null || o2 == null || o3 == null);
    }

    /**
     * 引数に null 値が含まれるか評価します。
     * 
     * @param o
     *            評価する一番目の値
     * @param o2
     *            評価する二番目の値
     * @param o3
     *            評価する三番目の値
     * @param o4
     *            評価する四番目の値
     * @return 引数に null 値が含まれる場合のみ true
     */
    static boolean isOrNull(Object o, Object o2, Object o3, Object o4) {
        return (o == null || o2 == null || o3 == null || o4 == null);
    }

    /**
     * 指定された値の比較結果が同値であるか評価します。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が同値の場合のみ true
     */
    static boolean isEqual(Comparable from, Comparable to) {
        return (0 == from.compareTo(to));
    }

    /**
     * 指定された値の比較結果が第一引数の方が小さい事を示すか評価します。
     * 
     * @param from
     *            比較基の値
     * @param to
     *            比較先の値
     * @return 値の比較結果が第一引数の方が小さい事を示す場合のみ true
     */
    static boolean isLess(Comparable from, Comparable to) {
        return (0 > from.compareTo(to));
    }

    /**
     * 指定の文字シーケンスの位置から前方一致するか評価します。
     * 
     * @param cs
     *            評価基の文字シーケンス
     * @param prefix
     *            評価先の接頭辞
     * @param offset
     *            評価を開始する相対インデックス
     * @return 指定の文字シーケンスの位置から前方一致する場合のみ true
     */
    static boolean isStartsWith(CharSequence cs, CharSequence prefix, int offset) {

        int plen = prefix.length();
        if ((offset < 0) || (offset > cs.length() - plen)) {
            return false;
        }
        int to = offset;
        int po = 0;
        while (--plen >= 0) {
            if (cs.charAt(to++) != prefix.charAt(po++)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 配列型であるかを評価します。
     * 
     * @param o
     *            評価するオブジェクト
     * @return 配列型の場合のみ true
     */
    static boolean isArray(Object o) {
        return (o != null && o.getClass().isArray());
    }

    /**
     * 指定の文字列に指定の文字が含まれているか評価します。
     * 
     * @param source
     *            評価する文字列
     * @param c
     *            検索する文字
     * @return 文字列に指定の文字が含まれている場合のみ true
     */
    static boolean isContains(String source, char c) {
        if (source == null) {
            return false;
        }
        return (0 <= source.indexOf(c));
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    static boolean isInfiniteOrNaN(Float val) {
        return (val.isInfinite() || val.isNaN());
    }

    /**
     * 数値の絶対値が無限量か、または非数 (NaN) であるか評価します。
     * 
     * @param val
     *            評価する数値
     * @return 数値の絶対値が無限量か、または非数 (NaN) の場合のみ true
     */
    static boolean isInfiniteOrNaN(Double val) {
        return (val.isInfinite() || val.isNaN());
    }

}
