/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * クラスを制御するユーティリティを提供します。
 */
class HClass {

    /* オブジェクト型の配列シンボル。 */
    static final char OBJECT_SYMBOL = 'L';

    /* プリミティブ型のクラスと省略名のマッピング。 */
    static final Map PRIMITIVE_FOR_TYPE_MAP;
    static {
        HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE.getName(), Byte.TYPE);
        m.put(Short.TYPE.getName(), Short.TYPE);
        m.put(Integer.TYPE.getName(), Integer.TYPE);
        m.put(Long.TYPE.getName(), Long.TYPE);
        m.put(Float.TYPE.getName(), Float.TYPE);
        m.put(Double.TYPE.getName(), Double.TYPE);
        m.put(Boolean.TYPE.getName(), Boolean.TYPE);
        m.put(Character.TYPE.getName(), Character.TYPE);
        m.put(Void.TYPE.getName(), Void.TYPE);
        PRIMITIVE_FOR_TYPE_MAP = Collections.unmodifiableMap(m);
    }

    /* プリミティブ型のクラスとクラス名のマッピング。 */
    static final Map PRIMITIVE_CLASS_NAME_MAP;
    static {
        HashMap m = new HashMap();
        // primitive
        m.put(Byte.TYPE, "B");
        m.put(Short.TYPE, "S");
        m.put(Integer.TYPE, "I");
        m.put(Long.TYPE, "J");
        m.put(Float.TYPE, "F");
        m.put(Double.TYPE, "D");
        m.put(Boolean.TYPE, "Z");
        m.put(Character.TYPE, "C");
        m.put(Void.TYPE, "V");
        PRIMITIVE_CLASS_NAME_MAP = Collections.unmodifiableMap(m);
    }

    /**
     * パッケージ名を含まないクラス名を返却します。
     * 
     * @param clazz
     *            クラス
     * @return パッケージ名を含まないクラス名
     */
    static String getShortClassName(Class clazz) {
        final String s = clazz.getName();
        final int off = s.lastIndexOf('.');
        final int beginSize = (-1 < off) ? off + 1 : 0;
        return s.substring(beginSize, s.length());
    }

    /**
     * 指定のクラス名からクラスを検索し、クラス参照を返却します。 <br>
     * 取得に失敗した場合は null を返却します。
     * 
     * @param className
     *            クラス名
     * @param loader
     *            クラスローダ
     * @return クラス参照
     */
    static Class loadClass(String className, ClassLoader loader) {
        ClassLoader classLoader = loader;
        // Class.forName：指定クラスの ClassLoader
        try {
            if (classLoader != null) {
                return Class.forName(className, true, loader);
            } else {
                return Class.forName(className);
            }
        } catch (Exception e) {
            // no op
        }
        // Current Thread の ClassLoader から取得
        try {
            classLoader = getContextClassLoader();
            if (classLoader != null) {
                return classLoader.loadClass(className);
            }
        } catch (Exception e) {
            // no op
        }
        // システムクラスローダを使って検索します。
        try {
            classLoader = ClassLoader.getSystemClassLoader();
            return classLoader.loadClass(className);
        } catch (Exception e) {
            // no op
        }
        return null;
    }

    /* 現在の Thread のコンテキスト ClassLoader を返却します。 */
    private static ClassLoader getContextClassLoader() {
        return Thread.currentThread().getContextClassLoader();
    }

}
