/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.beans.IndexedPropertyDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Map;

/**
 * JavaBean を制御するユーティリティを提供します。
 */
class BeanIntrospectHelper {

    /* 空の引数型の配列 */
    static final Class[] EMPTY_ARG_TYPES = new Class[0];

    /* 空の引数値の配列 */
    static final Object[] EMPTY_ARG_VALUES = new Object[0];

    /*
     * PropertyDescriptor
     */

    /**
     * プロパティ名のプロパティ定義を返却します。<br>
     * 指定のプロパティが存在しない場合は null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            プロパティ名
     * @return プロパティ定義
     * @throws OgdlSyntaxException
     *             ビーン記述子へのアクセスに失敗した場合
     */
    static PropertyDescriptor getPropertyDescriptor(Class clazz, String propName) {
        return getBeanDescriptor(clazz).getPropertyDescriptor(propName);
    }

    /**
     * プロパティ名のインデックス付きプロパティ定義を返却します。<br>
     * 指定のインデックス付きプロパティが存在しない場合は null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            プロパティ名
     * @return インデックス付きプロパティ定義
     * @throws OgdlSyntaxException
     *             ビーン記述子へのアクセスに失敗した場合
     */
    static IndexedPropertyDescriptor getIndexedPropertyDescriptor(Class clazz, String propName) {
        PropertyDescriptor pd = getPropertyDescriptor(clazz, propName);
        if (pd instanceof IndexedPropertyDescriptor) {
            return (IndexedPropertyDescriptor) pd;
        }
        return null;
    }

    /*
     * Property
     */

    /**
     * ビーンからプロパティ定義の示すプロパティを取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param pd
     *            プロパティ定義
     * @return プロパティ
     * @throws OgdlSyntaxException
     *             プロパティのアクセスに失敗した場合
     */
    static Object getProperty(Object bean, Method readPropertyMethod) {

        if (HEval.isOrNull(bean, readPropertyMethod)) {
            throw new NullPointerException("argument is null.");
        }
        try {
            return readPropertyMethod.invoke(bean, (Object[]) null);
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("get property err. ", bean.getClass(), readPropertyMethod), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("get property err. ", bean.getClass(), readPropertyMethod), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("get property err. ", bean.getClass(), readPropertyMethod), e);
        }

    }

    /**
     * ビーンからインデックス付きプロパティ定義の示すプロパティを取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param ipd
     *            java.beans.IndexedPropertyDescriptor
     * @param index
     *            インデックス
     * @return プロパティ
     * @throws OgdlSyntaxException
     *             プロパティのアクセスに失敗した場合
     */
    static Object getIndexedProperty(Object bean, Method readIndexedPropertyMethod, Integer index) {

        if (HEval.isOrNull(bean, readIndexedPropertyMethod)) {
            throw new NullPointerException("argument is null.");
        }
        try {
            return readIndexedPropertyMethod.invoke(bean, new Object[] { index });
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("get indexed property err. ", bean.getClass(), readIndexedPropertyMethod, index), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("get indexed property err. ", bean.getClass(), readIndexedPropertyMethod, index), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("get indexed property err. ", bean.getClass(), readIndexedPropertyMethod, index), e);
        }

    }

    /**
     * ビーンのプロパティ定義の示すプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param pd
     *            プロパティ定義
     * @param newValue
     *            格納する値
     * @throws OgdlSyntaxException
     *             プロパティのアクセスに失敗した場合
     */
    static void setProperty(Object bean, Method writePropertyMethod, Object newValue) {

        if (HEval.isOrNull(bean, writePropertyMethod)) {
            throw new NullPointerException("argument is null.");
        }
        try {
            writePropertyMethod.invoke(bean, new Object[] { newValue });
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writePropertyMethod, newValue), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writePropertyMethod, newValue), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writePropertyMethod, newValue), e);
        }

    }

    /**
     * ビーンのプロパティ定義の示すインデックス付きプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param ipd
     *            プロパティ定義
     * @param index
     *            インデックス
     * @param newValue
     *            格納する値
     * @throws OgdlSyntaxException
     *             プロパティのアクセスに失敗した場合
     */
    static void setIndexedProperty(Object bean, Method writeIndexedPropertyMethod, Integer index, Object newValue) {

        if (HEval.isOrNull(bean, writeIndexedPropertyMethod)) {
            throw new NullPointerException("argument is null.");
        }
        try {
            writeIndexedPropertyMethod.invoke(bean, new Object[] { index, newValue });
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writeIndexedPropertyMethod, index, newValue), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writeIndexedPropertyMethod, index, newValue), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("set property err. ", bean.getClass(), writeIndexedPropertyMethod, index, newValue), e);
        }

    }

    /*
     * Method
     */

    /**
     * 指定されたクラスから処理の対象となるメソッドを返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @param paramTypes
     *            パラメータ型
     * @return 指定のメソッド
     * @throws OgdlSyntaxException
     *             指定のメソッドが発見出来ない場合
     */
    static Method getMethod(Class clazz, String methodName, Class[] paramTypes) {
        return getBeanDescriptor(clazz).getAssignmentMethod(methodName, paramTypes);
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。
     * 
     * @param clazz
     *            実行するクラス
     * @param obj
     *            実行するインスタンス
     * @param methodName
     *            メソッド名
     * @param paramValues
     *            パラメータ値
     * @return メソッドの返却値
     * @throws OgdlSyntaxException
     *             メソッドの呼出に失敗した場合
     */
    static Object invokeMethod(Class clazz, Object obj, String methodName, Object[] paramValues) {

        if (HEval.isOrNull(clazz, methodName, paramValues)) {
            throw new NullPointerException("argument is null.");
        }
        final Class[] paramTypes = toClassArray(paramValues);
        final Method method = getMethod(clazz, methodName, paramTypes);
        if (method == null) {
            throw new OgdlSyntaxException(HLog.log("not matching method, ", clazz, methodName, paramTypes, paramValues));
        }
        try {

            if (Modifier.isStatic(method.getModifiers())) {
                return method.invoke(null, paramValues);
            }
            if (obj == null) {
                throw new NullPointerException("argument obj is null.");
            }
            return method.invoke(obj, paramValues);

        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("method invocation err. ", clazz, methodName, paramTypes, paramValues), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("method invocation err. ", clazz, methodName, paramTypes, paramValues), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("method invocation err. ", clazz, methodName, paramTypes, paramValues), e);
        }

    }

    /*
     * Static Method
     */

    /**
     * 指定されたクラスから処理の対象となるメソッドを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 指定のメソッド
     * @throws OgdlSyntaxException
     *             指定のメソッドが発見出来ない場合
     */
    static Map getStaticMethodGroup(Class clazz) {
        return getBeanDescriptor(clazz).getStaticMethodGroup();
    }

    /*
     * Field
     */

    /**
     * アクセス可能なフィールドを検索して返却します。
     * 
     * @param clazz
     *            検索するクラス
     * @param fieldName
     *            フィールド名
     * @return アクセス可能なフィールド、発見できない場合 null
     * @throws OgdlSyntaxException
     *             フィールドのアクセスに失敗した場合
     */
    static Field getField(Class clazz, String fieldName) {
        return getBeanDescriptor(clazz).getField(fieldName);
    }

    /**
     * オブジェクトからフィールド名の示すフィールド値を取得して返却します。
     * 
     * @param clazz
     *            フィールドを保有するクラス
     * @param obj
     *            フィールドを保有するオブジェクト
     * @param fieldName
     *            フィールド名
     * @return フィールドの値
     * @throws OgdlSyntaxException
     *             フィールドの呼出に失敗した場合
     */
    static Object getFieldValue(Class clazz, Object obj, String fieldName) {
        try {
            return getField(clazz, fieldName).get(obj);
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("get field err. ", clazz, fieldName), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("get field err. ", clazz, fieldName), e);
        }
    }

    /**
     * ビーンの static final なフィールドを不変マップに格納して返却します。
     * 
     * @param clazz
     *            クラス
     * @return 全ての static final なフィールドを格納する不変マップ
     * @throws OgdlSyntaxException
     *             フィールドの取得に失敗した場合
     */
    static Map getConstantFieldMap(Class clazz) {
        return getBeanDescriptor(clazz).getConstantFieldMap();
    }

    /*
     * Constructor
     */

    /* コンストラクタを検索して返却します。 */
    static Constructor getConstructor(Class clazz, Class[] paramTypes) {
        return getBeanDescriptor(clazz).getAssignmentConstructor(paramTypes);
    }

    /*
     * new instance
     */

    /**
     * パブリックな空コンストラクタからインスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @return 生成されたインスタンス
     * @throws OgdlSyntaxException
     *             インスタンスの生成に失敗した場合
     */
    static Object newInstance(Class clazz) {
        try {
            return clazz.newInstance();
        } catch (SecurityException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz), e);
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz), e);
        } catch (InstantiationException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz), e);
        }
    }

    /**
     * パラメータを指定したコンストラクタからインスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @param paramValues
     *            パラメータ値
     * @return 生成されたインスタンス
     * @throws OgdlSyntaxException
     *             インスタンスの生成に失敗した場合
     */
    static Object newInstance(Class clazz, Object[] paramValues) {
        try {
            if (paramValues.length == 0) {
                return getConstructor(clazz, EMPTY_ARG_TYPES).newInstance(paramValues);
            } else {
                final Class[] paramTypes = toClassArray(paramValues);
                return getConstructor(clazz, paramTypes).newInstance(paramValues);
            }
        } catch (SecurityException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz, paramValues), e);
        } catch (IllegalArgumentException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz, paramValues), e);
        } catch (InstantiationException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz, paramValues), e);
        } catch (IllegalAccessException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz, paramValues), e);
        } catch (InvocationTargetException e) {
            throw new OgdlSyntaxException(HLog.log("new instance err. ", clazz, paramValues), e);
        }
    }

    /*
     * Utils
     */

    /* オブジェクトの配列を基にその要素のクラスの配列を生成して返却する。 */
    static Class[] toClassArray(Object[] os) {
        final Class[] cs = new Class[os.length];
        for (int i = 0; i < os.length; i++) {
            cs[i] = os[i].getClass();
        }
        return cs;
    }

    /* 第一引数に対して第二引数が全て割り当て可能か検証します。 */
    static boolean isAssignmentCompatible(Class[] ts, Class[] ts2) {
        return BeanIntrospectDescriptor.isAssignmentCompatible(ts, ts2);
    }

    private static BeanIntrospectDescriptor getBeanDescriptor(Class clazz) {
        return BeanIntrospectDescriptor.forClass(clazz);
    }

}
