/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ogdl;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.WeakHashMap;

/**
 * Java Bean を動的に内部参照を実行する為の定義情報を保有する機能を提供します。
 */
class BeanIntrospectDescriptor {

    /* 弱参照を用いてクラスをキーとしてBean記述子をキャッシュします。 */
    private static final Map classCache = new WeakHashMap();

    /**
     * 指定のクラスの情報を持つ BeanIntrospectDescriptor インスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @return BeanIntrospectDescriptor インスタンス
     * @throws OgdlSyntaxException
     *             生成に失敗した場合
     */
    static BeanIntrospectDescriptor forClass(Class clazz) throws OgdlSyntaxException {
        if (clazz == null) {
            throw new NullPointerException("argument is null.");
        }
        synchronized (classCache) {
            BeanIntrospectDescriptor results = (BeanIntrospectDescriptor) classCache.get(clazz);
            if (results == null) {
                // can throw OgdlSyntaxException
                results = new BeanIntrospectDescriptor(clazz);
                classCache.put(clazz, results);
            }
            return results;
        }
    }

    /* BeanInfo のキャッシュ */
    private final BeanInfo beanInfo;

    /* Constructor のコレクションです。 */
    private Collection constructors;

    /* PropertyDescriptor をプロパティ名でマッピングします。 */
    private Map propDesc;

    /* Method をメソッド名でグルーピングします。 */
    private Map methodGroup;

    /* Static Method をメソッド名でグルーピングします。 */
    private Map staticMethodGroup;

    /* 静的な公開された定数フィールドをフィールド名でマッピングします。 */
    private Map constants;

    /* Field をフィールド名でグルーピングします。 */
    private Map fieldGroup;

    /* 指定のクラスの BeanInfo をキャッシュして初期化する */
    private BeanIntrospectDescriptor(Class clazz) throws OgdlSyntaxException {

        try {
            this.beanInfo = Introspector.getBeanInfo(clazz);

            // クラスと親クラスに関する Introspector の内部キャッシュ情報をフラッシュします。
            // このクラスは弱参照を用いるため GC の効率性に有効です。（Spring を参考にしました）
            Class classToFlush = clazz;
            do {
                Introspector.flushFromCaches(classToFlush);
                classToFlush = classToFlush.getSuperclass();
            } while (classToFlush != null);

        } catch (SecurityException e) {
            throw new OgdlSyntaxException("Security error. " + clazz.getName(), e);
        } catch (IntrospectionException e) {
            throw new OgdlSyntaxException("Cannot get BeanInfo for object. " + clazz.getName(), e);
        }
    }

    /* Constructor のコレクション */
    private Collection constructors() {
        final Collection conses = new LinkedList();
        try {
            final Constructor[] cs = getBeanClass().getConstructors();
            for (int i = 0; i < cs.length; i++) {
                conses.add(cs[i]);
            }
        } catch (SecurityException e) {
            // no op
        }
        // unmodifiable
        return Collections.unmodifiableCollection(conses);
    }

    /* Method のグルーピング */
    private Map methodGroup() {
        final HashMap group = new HashMap(8);
        methodGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* Method を再起的に検索して名前でグルーピングする */
    private void methodGroup0(Map group, Class clazz) {
        try {
            final Method[] ms = clazz.getDeclaredMethods();
            for (int i = 0; i < ms.length; i++) {
                Method m = ms[i];
                if (!Modifier.isPublic(m.getModifiers())) {
                    continue;
                }
                if (!Modifier.isPublic(m.getDeclaringClass().getModifiers())) {
                    m.setAccessible(true);
                }
                Collection coll = (Collection) group.get(m.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(m.getName(), coll);
                }
                coll.add(m);
            }
        } catch (SecurityException e) {
            // no op
        }
        Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            Class c = interfaces[i];
            methodGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            Class c = clazz.getSuperclass();
            if (c != null) {
                methodGroup0(group, c);
            }
        }
    }

    /* Static Method のグルーピング */
    private Map staticMethodGroup() {
        final HashMap group = new HashMap(8);
        staticMethodGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* Static Method を再起的に検索して名前でグルーピングする */
    private void staticMethodGroup0(Map group, Class clazz) {
        try {
            final Method[] ms = clazz.getMethods();
            for (int i = 0; i < ms.length; i++) {
                Method m = ms[i];
                // public and static and final
                int modifiers = m.getModifiers();
                if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers)) {
                    Collection coll = (Collection) group.get(m.getName());
                    if (coll == null) {
                        coll = new LinkedList();
                        group.put(m.getName(), coll);
                    }
                    coll.add(m);
                }
            }
        } catch (SecurityException e) {
            // no op
        }
    }

    /* Field のグルーピング */
    private Map fieldGroup() {
        final HashMap group = new HashMap(8);
        fieldGroup0(group, getBeanClass());
        // unmodifiable
        for (Iterator i = group.entrySet().iterator(); i.hasNext();) {
            Map.Entry e = (Map.Entry) i.next();
            Collection coll = (Collection) e.getValue();
            e.setValue(Collections.unmodifiableCollection(coll));
        }
        return Collections.unmodifiableMap(group);
    }

    /* Field を再起的に検索して名前でグルーピングする */
    private void fieldGroup0(Map group, Class clazz) {
        try {
            final Field[] fs = clazz.getDeclaredFields();
            for (int i = 0; i < fs.length; i++) {
                Field f = fs[i];
                if (!Modifier.isPublic(f.getModifiers())) {
                    continue;
                }
                if (!Modifier.isPublic(f.getDeclaringClass().getModifiers())) {
                    f.setAccessible(true);
                }
                Collection coll = (Collection) group.get(f.getName());
                if (coll == null) {
                    coll = new LinkedList();
                    group.put(f.getName(), coll);
                }
                coll.add(f);
            }
        } catch (SecurityException e) {
            // no op
        }
        Class[] interfaces = clazz.getInterfaces();
        for (int i = 0; i < interfaces.length; i++) {
            Class c = interfaces[i];
            fieldGroup0(group, c);
        }
        if (!clazz.isInterface()) {
            Class c = clazz.getSuperclass();
            if (c != null) {
                fieldGroup0(group, c);
            }
        }
    }

    /* PropertyDescriptor のマッピング */
    private Map propDesc() {
        final HashMap pdsMap = new HashMap(8);
        try {
            final PropertyDescriptor[] pds = this.beanInfo.getPropertyDescriptors();
            for (int i = 0; i < pds.length; i++) {
                PropertyDescriptor pd = pds[i];
                Method readMethod = pd.getReadMethod();
                Method writeMethod = pd.getWriteMethod();
                if (readMethod != null && !Modifier.isPublic(readMethod.getDeclaringClass().getModifiers())) {
                    readMethod.setAccessible(true);
                }
                if (writeMethod != null && !Modifier.isPublic(writeMethod.getDeclaringClass().getModifiers())) {
                    writeMethod.setAccessible(true);
                }
                pdsMap.put(pd.getName(), pd);
            }
        } catch (SecurityException e) {
            // no op
        }
        return Collections.unmodifiableMap(pdsMap);
    }

    /* 定数フィールドのマッピング */
    private Map constants() {
        HashMap fsMap = new HashMap();
        try {
            Class clazz = getBeanClass();
            Field[] fields = clazz.getFields();
            for (int i = 0; i < fields.length; i++) {
                Field field = fields[i];
                // public and static and final
                int modifiers = field.getModifiers();
                if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
                    fsMap.put(field.getName(), field.get(null));
                }
            }
        } catch (SecurityException e) {
            // no op
        } catch (IllegalArgumentException e) {
            // no op
        } catch (IllegalAccessException e) {
            // no op
        }
        return Collections.unmodifiableMap(fsMap);
    }

    /*
     * BeanInfo
     */

    /**
     * このインスタンスの java.beans.BeanInfo を返却します。
     * 
     * @return java.beans.BeanInfo
     */
    BeanInfo getBeanInfo() {
        return this.beanInfo;
    }

    /**
     * このインスタンスのビーンクラスを返却します。
     * 
     * @return ビーンクラス
     */
    Class getBeanClass() {
        return this.beanInfo.getBeanDescriptor().getBeanClass();
    }

    /*
     * Constructors
     */

    /**
     * パブリック定数を不変マップに格納して返却します。
     * 
     * @return パブリック定数の名前と値の不変マップ
     */
    Collection getConstructors() {
        Collection c = constructors;
        return (c != null) ? c : (constructors = constructors());
    }

    /**
     * 指定されたパラメータ型を持つ Constructor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param parameterTypes
     *            パラメータ型
     * @return 指定されたパラメータ型を持つ Constructor
     */
    Constructor getConstructor(Class[] paramTypes) {
        if (paramTypes == null) {
            throw new NullPointerException("argument is null.");
        }
        final Collection conses = getConstructors();
        for (Iterator i = conses.iterator(); i.hasNext();) {
            Constructor cons = (Constructor) i.next();
            if (isEquals(cons.getParameterTypes(), paramTypes)) {
                return cons;
            }
        }
        return null;
    }

    /**
     * 指定されたパラメータ型を持つ Constructor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param parameterTypes
     *            パラメータ型
     * @return 指定されたパラメータ型を持つ Constructor
     */
    Constructor getAssignmentConstructor(Class[] paramTypes) {
        if (paramTypes == null) {
            throw new NullPointerException("argument is null.");
        }
        // equals parameter types
        Constructor cons = getConstructor(paramTypes);
        if (cons != null) {
            return cons;
        }
        // is Aassignment
        final Collection conses = getConstructors();
        for (Iterator i = conses.iterator(); i.hasNext();) {
            cons = (Constructor) i.next();
            if (isAssignmentCompatible(cons.getParameterTypes(), paramTypes)) {
                return cons;
            }
        }
        return null;
    }

    /*
     * Constant Field
     */

    /**
     * パブリック定数を不変マップに格納して返却します。
     * 
     * @return パブリック定数の名前と値の不変マップ
     */
    Map getConstantFieldMap() {
        Map m = constants;
        return (m != null) ? m : (constants = constants());
    }

    /**
     * 指定された名前のパブリック定数の値を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return パブリック定数の値
     */
    Object getConstantField(String fieldName) {
        return getConstantFieldMap().get(fieldName);
    }

    /*
     * PropertyDescriptor
     */

    /**
     * PropertyDescriptor を不変マップに格納して返却します。
     * 
     * @return PropertyDescriptor のプロパティ名と PropertyDescriptor の不変マップ
     */
    Map getPropertyDescriptorMap() {
        Map m = propDesc;
        return (m != null) ? m : (propDesc = propDesc());
    }

    /**
     * PropertyDescriptor を不変コレクションに格納して返却します。
     * 
     * @return PropertyDescriptor のコレクション
     */
    Collection getPropertyDescriptors() {
        return getPropertyDescriptorMap().values();
    }

    /**
     * 指定された名前の PropertyDescriptor を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param propertyName
     *            プロパティ名
     * @return PropertyDescriptor
     */
    PropertyDescriptor getPropertyDescriptor(String propertyName) {
        return (PropertyDescriptor) getPropertyDescriptorMap().get(propertyName);
    }

    /*
     * Field
     */

    /**
     * 同一のフィールド名の Field を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のフィールド名の Field を不変コレクションでマッピングした不変マップ
     */
    Map getFieldGroup() {
        Map g = fieldGroup;
        return (g != null) ? g : (fieldGroup = fieldGroup());
    }

    /**
     * 指定された名前の Field を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field の不変コレクション
     */
    Collection getFields(String fieldName) {
        return (Collection) getFieldGroup().get(fieldName);
    }

    /**
     * 指定された名前の Field を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param fieldName
     *            フィールド名
     * @return Field
     */
    Field getField(String fieldName) {
        Collection coll = (Collection) getFieldGroup().get(fieldName);
        if (coll != null) {
            return (Field) coll.iterator().next();
        }
        return null;
    }

    /*
     * Method
     */

    /**
     * 同一のメソッド名の Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のメソッド名の Method を不変コレクションでマッピングした不変マップ
     */
    Map getMethodGroup() {
        Map g = methodGroup;
        return (g != null) ? g : (methodGroup = methodGroup());
    }

    /**
     * 指定された名前の Method を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @return Method の不変コレクション
     */
    Collection getMethods(String methodName) {
        return (Collection) getMethodGroup().get(methodName);
    }

    /**
     * 指定された名前と、指定されたパラメータ型に割り当て可能なパラメータ型を持つ Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getAssignmentMethod(String methodName, Class[] parameterTypes) {
        Collection coll = getMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                Method m = (Method) i.next();
                if (isAssignmentCompatible(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

    /*
     * Static Method
     */

    /**
     * 同一のメソッド名の Static Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @return 同一のメソッド名の Method を不変コレクションでマッピングした不変マップ
     */
    Map getStaticMethodGroup() {
        Map g = staticMethodGroup;
        return (g != null) ? g : (staticMethodGroup = staticMethodGroup());
    }

    /**
     * 指定された名前の Static Method を不変コレクションに格納して返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @return Method の不変コレクション
     */
    Collection getStaticMethods(String methodName) {
        return (Collection) getStaticMethodGroup().get(methodName);
    }

    /**
     * 指定された名前と、指定されたパラメータ型に割り当て可能なパラメータ型を持つ Static Method を返却します。<br>
     * 発見出来ない場合は null を返却します。
     * 
     * @param methodName
     *            メソッド名
     * @param parameterTypes
     *            パラメータ型
     * @return 指定された名前とパラメータ型を持つ Method
     */
    Method getAssignmentStaticMethod(String methodName, Class[] parameterTypes) {
        Collection coll = getStaticMethods(methodName);
        if (coll != null) {
            for (Iterator i = coll.iterator(); i.hasNext();) {
                Method m = (Method) i.next();
                if (isAssignmentCompatible(m.getParameterTypes(), parameterTypes)) {
                    return m;
                }
            }
        }
        return null;
    }

    /* 第一引数に対して第二引数が全て割り当て可能か検証します。 */
    static boolean isAssignmentCompatible(Class[] ts, Class[] ts2) {
        if (ts.length != ts2.length) {
            return false;
        }
        for (int i = 0; i < ts.length; i++) {
            if (!isAssignmentCompatible(ts[i], ts2[i])) {
                return false;
            }
        }
        return true;
    }

    /* 第一引数に対して第二引数が割り当て可能か検証します。 */
    static boolean isAssignmentCompatible(Class t, Class t2) {
        // 単純に割り当て可能か検証する
        if (t.isAssignableFrom(t2)) {
            return true;
        }
        // ラッパー型が一致する場合許可
        return ((t.isPrimitive()) ? Boxing.boxClass(t) : t).equals(((t2.isPrimitive()) ? Boxing.boxClass(t2) : t2));
    }

    /* 配列の全要素が一致するか評価します。 */
    static boolean isEquals(Class[] ts, Class[] ts2) {
        if (ts == ts2) {
            return true;
        }
        if (ts == null || ts2 == null) {
            return false;
        }
        if (ts.length != ts2.length) {
            return false;
        }
        for (int i = 0; i < ts.length; i++) {
            if (!isEquals(ts[i], ts2[i])) {
                return false;
            }
        }
        return true;
    }

    /* 指定の値が同値であるかを評価します。 */
    static boolean isEquals(Class t, Class t2) {
        return (t == t2) || (t != null && t.equals(t2));
    }

}
