/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.util.debug;

import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Array;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Map;
import java.util.regex.Pattern;

import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HFmt;
import shohaku.core.lang.VariableInteger;

/**
 * デバック文を出力する拡張プリントストリームを提供します。 <br>
 * <br>
 * 配列・多次元配列の要素の出力や日付のフォーマット出力、 java bean のプロパティ出力等の機能が提供されています。<br>
 * 以外にも様々なフォーマットで出力するユーティリティメソッドが多数提供されています。 <br>
 */
public class DumpStream extends PrintStream {

    /** 反復処理を行う要素数のデフォルトの上限値。 */
    public static final int DEFAULT_REPETITION_LIMIT = 1024;

    /* 反復処理を行う要素数の上限値。 */
    private int repetitionLimit = DEFAULT_REPETITION_LIMIT;

    /**
     * 標準の出力ストリームを出力先として初期化します。
     */
    public DumpStream() {
        this(System.out, false);
    }

    /**
     * 標準の出力ストリームを出力先として初期化します。
     * 
     * @param autoFlush
     *            true の場合 println() メソッドでは出力バッファをフラッシュする
     */
    public DumpStream(boolean autoFlush) {
        this(System.out, autoFlush);
    }

    /**
     * 指定された出力ストリームを出力先として初期化します。
     * 
     * @param out
     *            出力ストリーム
     * @param autoFlush
     *            true の場合 println() メソッドでは出力バッファをフラッシュする
     */
    public DumpStream(OutputStream out, boolean autoFlush) {
        super(out, autoFlush);
    }

    /*
     * print
     */

    /**
     * 登録済みの出力プリンタを使用して出力ストリームへ出力します。
     * 
     * @param o
     *            出力値
     */
    public void print(Object o) {
        try {
            if (o == null) {
                printNull();
            } else if (o.getClass().isArray()) {
                printArray(o);
            } else if (o instanceof Map) {
                printMap((Map) o);
            } else if (o instanceof Collection) {
                printColl((Collection) o);
            } else if (o instanceof java.util.Date) {
                printDate((Date) o);
            } else if (o instanceof Pattern) {
                printPattern((Pattern) o);
            } else {
                super.print(o);
            }
        } catch (Exception e) {
            // no op
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, boolean value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, char value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, double value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, float value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, int value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, long value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルを付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void print(Object title, Object value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(value);
        }
    }

    /**
     * タイトルと名前=値を付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value
     *            出力値
     */
    public void print(Object title, Object name, Object value) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(name);
            super.print('=');
            this.print(value);
        }
    }

    /**
     * タイトルと名前=値１、値２を付けて出力します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value1
     *            出力値１
     * @param value2
     *            出力値２
     */
    public void print(Object title, Object name, Object value1, Object value2) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            this.print(name);
            super.print('=');
            this.print(value1);
            this.print(", ");
            this.print(value2);
        }
    }

    /*
     * println
     */

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, boolean value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, char value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, double value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, float value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, int value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, long value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルを付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param value
     *            出力値
     */
    public void println(Object title, Object value) {
        synchronized (this) {
            this.print(title, value);
            super.println();
        }
    }

    /**
     * タイトルと名前=値を付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value
     *            出力値
     */
    public void println(Object title, Object name, Object value) {
        synchronized (this) {
            this.print(title, name, value);
            super.println();
        }
    }

    /**
     * タイトルと名前=値１、値２を付けて出力し改行します。
     * 
     * @param title
     *            タイトル
     * @param name
     *            値の名前
     * @param value1
     *            出力値１
     * @param value2
     *            出力値２
     */
    public void println(Object title, Object name, Object value1, Object value2) {
        synchronized (this) {
            this.print(title, name, value1, value2);
            super.println();
        }
    }

    /*
     * 空間出力
     */

    /**
     * 引数分の改行を出力します。
     * 
     * @param len
     *            改行数
     */
    public void printnl(int len) {
        synchronized (this) {
            for (int i = 0; i < len; i++) {
                super.println();
            }
        }
    }

    /**
     * 引数分のスペースを出力します。
     * 
     * @param len
     *            スペース数
     */
    public void printsp(int len) {
        synchronized (this) {
            for (int i = 0; i < len; i++) {
                super.print(' ');
            }
        }
    }

    /*
     * 線
     */

    /**
     * 破線を出力します。
     * 
     * @param len
     *            破線数
     */
    public void printLine(int len) {
        synchronized (this) {
            char[] c = new char[len];
            for (int i = 0; i < len; i++) {
                c[i] = '-';
            }
            this.print(c);
        }
    }

    /**
     * 破線を出力して１行分改行します。
     * 
     * @param len
     *            破線数
     */
    public void printLineln(int len) {
        synchronized (this) {
            this.printLine(len);
            super.println();
        }
    }

    /**
     * タイトルを指定して破線を出力します。 "------ タイトル ------"
     * 
     * @param title
     *            タイトル
     * @param len
     *            破線数
     */
    public void printLineln(Object title, int len) {
        synchronized (this) {
            super.println();
            this.printLine(len);
            this.printsp(2);
            this.print(title);
            this.printsp(2);
            this.printLine(len);
            super.println();
        }
    }

    /**
     * 破線前後の改行数を指定して破線を出力します。
     * 
     * @param len
     *            破線数
     * @param beginLine
     *            開始改行
     * @param endLine
     *            終了改行
     */
    public void printLineln(int len, int beginLine, int endLine) {
        synchronized (this) {
            this.printnl(beginLine);
            this.printLine(len);
            this.printnl(endLine);
        }
    }

    /**
     * 二重破線を出力します。
     * 
     * @param len
     *            破線数
     */
    public void printDLine(int len) {
        synchronized (this) {
            char[] c = new char[len];
            for (int i = 0; i < len; i++) {
                c[i] = '=';
            }
            this.print(c);
        }
    }

    /**
     * 二重破線を出力して前後１行分改行します。
     * 
     * @param len
     *            破線数
     */
    public void printDLineln(int len) {
        this.printDLineln(len, 1, 1);
    }

    /**
     * タイトルを指定して二重破線を出力します。 "====== タイトル ======"
     * 
     * @param title
     *            タイトル
     * @param len
     *            破線数
     */
    public void printDLineln(Object title, int len) {
        synchronized (this) {
            super.println();
            this.printDLine(len);
            this.printsp(2);
            this.print(title);
            this.printsp(2);
            this.printDLine(len);
            super.println();
        }
    }

    /**
     * 破線前後の改行数を指定して二重破線を出力します。
     * 
     * @param len
     *            破線数
     * @param beginLine
     *            開始改行
     * @param endLine
     *            終了改行
     */
    public void printDLineln(int len, int beginLine, int endLine) {
        synchronized (this) {
            this.printnl(beginLine);
            this.printDLine(len);
            this.printnl(endLine);
        }
    }

    /*
     * コレクション
     */

    /**
     * 多次元配列を出力します。
     * 
     * <pre>
     *  &lt;ArrayClassType(length)&gt;[item1, item2, &lt;ArrayClassType(length)&gt;[item3-1, item3-2, …], item4]
     * </pre>
     * 
     * 配列以外は print(Object) を呼ぶ
     * 
     * @param a
     *            出力値
     */
    public void printArray(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
            } else if (a.getClass().isArray()) {
                this.printClass(a);
                printDeepArrays(a);
            } else {
                super.print(a);
            }
        }
    }

    /**
     * タイトルと多次元配列を出力します。
     * 
     * <pre>
     *  タイトル : &lt;ArrayClassType(length)&gt;[item1, item2, &lt;ArrayClassType(length)&gt;[item3-1, item3-2, …], item4]
     * </pre>
     * 
     * @param title
     *            タイトル
     * @param a
     *            出力値
     */
    public void printArray(Object title, Object a) {
        synchronized (this) {
            this.print(title);
            super.print(" : ");
            super.println();
            this.printArray(a);
            super.println();
        }
    }

    /**
     * コレクションを出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;[ item1, item2, item3, … ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printColl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.print("[]");
            } else {
                this.printClass(c);
                super.print('[');
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                while (hasNext) {
                    Object o = i.next();
                    this.print((o == c) ? "(this Collection)" : o);
                    hasNext = i.hasNext();
                    if (hasNext) {
                        super.print(", ");
                    }
                }
                super.println(']');
            }
        }
    }

    /**
     * マップを出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ key1:value1, key2:value2, key3:value3, … }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printMap(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.print("{}");
            } else {
                this.printClass(m);
                super.print('{');
                boolean st = true;
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.print((key == m) ? "(this Map)" : key);
                    super.print('=');
                    this.print((value == m) ? "(this Map)" : value);
                }
                super.println('}');
            }
        }
    }

    /**
     * 要素のクラス名を含めコレクションを出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;[ &lt;ClassName&gt;item1, &lt;ClassName&gt;item2, … ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printTypeColl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.print("[]");
            } else {
                this.printClass(c);
                super.print('[');
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                while (hasNext) {
                    Object o = i.next();
                    this.printClass(o);
                    this.print((o == c) ? "(this Collection)" : o);
                    hasNext = i.hasNext();
                    if (hasNext) {
                        super.print(", ");
                    }
                }
                super.println(']');
            }
        }
    }

    /**
     * 要素のクラス名を含めマップを出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ &lt;ClassName&gt;key1:&lt;ClassName&gt;value1, &lt;ClassName&gt;key2:&lt;ClassName&gt;value2, … }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printTypeMap(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.print("{}");
            } else {
                this.printClass(m);
                super.print('{');
                boolean st = true;
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.printClass(key);
                    this.print((key == m) ? "(this Map)" : key);
                    super.print('=');
                    this.printClass(value);
                    this.print((value == m) ? "(this Map)" : value);
                }
                super.println('}');
            }
        }
    }

    /**
     * 多次元配列を表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[
     *      item1
     *      &lt;ArrayClassType(length)&gt;[item2-1, item2-2, item2-3]
     *      item3
     *      …
     *     ]
     * </pre>
     * 
     * @param a
     *            出力値
     */
    public void printArrayTbl(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                super.println();
            } else if (a.getClass().isArray()) {

                this.printClass(a.getClass());
                super.println('[');

                int size = Array.getLength(a);
                for (int i = 0; i < size; i++) {
                    this.printsp(2);
                    this.printArray(Array.get(a, i));
                    super.println();
                }

                super.println(']');
            } else {
                super.println(a);
            }
        }
    }

    /**
     * 多次元配列を表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[
     *       item1, item2, item3
     *       item4, &lt;ArrayClassType(length)&gt;[item5-1, item5-2, item5-3], item6
     *       item7, …
     *     ]
     * </pre>
     * 
     * @param a
     *            出力値
     * @param column
     *            １列分の要素数
     */
    public void printArrayTbl(Object a, int column) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                super.println();

            } else if (a.getClass().isArray()) {
                int size = Array.getLength(a);

                if (size == 0 || column <= 0) {

                    this.printClass(a);
                    this.println("[]");

                } else {

                    this.printClass(a);
                    super.println('[');

                    int row = (size % column == 0) ? (size / column) : (size / column + 1);
                    for (int inx = 0, i = 0; (i < row && inx < size); i++) {
                        this.printsp(2);
                        for (int j = 0; (j < column && inx < size); j++) {
                            this.printArray(Array.get(a, i));
                            if ((j + 1) < column && (inx + 1) < size) {
                                this.print(", ");
                            }
                            inx++;
                        }
                        super.println();
                    }

                    super.println(']');
                }
            } else {
                super.println(a);
            }
        }

    }

    /**
     * コレクションを表形式で出力します。
     * 
     * <pre>
     *    &lt;ClassName&gt;[
     *      item1
     *      item2
     *      item3
     *      …
     *    ]
     * </pre>
     * 
     * @param c
     *            出力値
     */
    public void printCollTbl(Collection c) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
                super.println();
            } else if (c.size() == 0) {
                this.printClass(c);
                this.println("[]");
                return;
            } else {
                this.printClass(c);
                super.println('[');
                Iterator i = c.iterator();
                while (i.hasNext()) {
                    this.printsp(2);
                    Object o = i.next();
                    this.println((o == c) ? "(this Collection)" : o);
                }
                super.println(']');
            }
        }
    }

    /**
     * コレクションを表形式で出力します。
     * 
     * <pre>     
     *     &lt;ClassName&gt;[
     *       item1, item2, item3
     *       item4, item5, item6
     *       item7, …
     *     ]
     * </pre>
     * 
     * @param c
     *            出力値
     * @param column
     *            １列分の要素数
     */
    public void printCollTbl(Collection c, int column) {
        synchronized (this) {
            if (c == null) {
                this.printNull();
                super.println();
            } else if (c.size() == 0 || column <= 0) {
                this.printClass(c);
                this.println("[]");
                return;
            } else {
                Iterator i = c.iterator();
                boolean hasNext = i.hasNext();
                this.printClass(c);
                super.println('[');
                while (hasNext) {
                    this.printsp(2);
                    int icolumn = 0;
                    while (hasNext && icolumn < column) {
                        Object o = i.next();
                        this.print((o == c) ? "(this Collection)" : o);
                        hasNext = i.hasNext();
                        if (hasNext) {
                            super.print(", ");
                        }
                        icolumn++;
                    }
                    super.println();
                }
                super.println(']');
            }
        }
    }

    /**
     * マップを表形式で出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       key1:value1
     *       key2:value2
     *       key3:value3
     *       …
     *     }
     * </pre>
     * 
     * @param m
     *            出力値
     */
    public void printMapTbl(Map m) {
        synchronized (this) {
            if (m == null) {
                this.printNull();
                super.println();
            } else if (m.size() == 0) {
                this.printClass(m);
                this.println("{}");
            } else {
                this.printClass(m);
                super.println('{');
                for (Iterator i = m.entrySet().iterator(); i.hasNext();) {
                    Map.Entry e = (Map.Entry) i.next();
                    Object key = e.getKey();
                    Object value = e.getValue();
                    this.printsp(2);
                    this.print((key == m) ? "(this Map)" : key);
                    super.print('=');
                    this.println((value == m) ? "(this Map)" : value);
                }
                super.println('}');
            }
        }
    }

    /*
     * JavaBean
     */

    /**
     * JavaBean の保有するプロパティを全て出力します。
     * 
     * <pre>
     *  &lt;ClassName&gt;{ property1=value1, property2=value2, … }
     * </pre>
     * 
     * @param bean
     *            出力する JavaBean
     */
    public void printBean(Object bean) {
        synchronized (this) {

            // プロパティをMapに格納して全取得
            Map props;
            try {
                props = HBeans.getProperties(bean);
            } catch (Exception e) {
                super.println();
                return;
            }
            // 出力開始
            // クラス名を出力する
            this.printClass(bean);
            // 全プロパティを出力する
            super.print('{');
            boolean st = true;
            for (Iterator i = props.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                if (!"class".equals(e.getKey())) {
                    if (st) {
                        st = false;
                    } else {
                        this.print(", ");
                    }
                    this.print((String) e.getKey());
                    super.print('=');
                    this.print(e.getValue());
                }
            }
            super.println('}');
        }
    }

    /**
     * 全ての JavaBean の保有するプロパティを全て出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[length]
     *       [
     *       [0]&lt;ClassName&gt;{ property1=value1, property2=value2, … }
     *       [1]&lt;ClassName&gt;{ property1=value1, property2=value2, … }
     *       …
     *     ]
     * </pre>
     * 
     * @param beans
     *            出力する JavaBean の配列
     */
    public void printBeans(Object[] beans) {
        synchronized (this) {
            // 全Beanを出力
            this.printClass(beans);
            super.println();
            super.println('[');
            for (int i = 0; i < beans.length; i++) {
                this.printsp(2);
                super.print('[');
                this.print(i);
                super.print(']');
                this.printBean(beans[i]);
                super.println();
            }
            super.println(']');
        }
    }

    /**
     * JavaBean の保有するプロパティを全て表形式で出力します。
     * 
     * <pre>        
     *     &lt;ClassName&gt;{
     *       property1=value1
     *       property2=value2
     *       property3=value3
     *       …
     *     }
     * </pre>
     * 
     * @param bean
     *            出力する JavaBean
     */
    public void printBeanTbl(Object bean) {
        synchronized (this) {
            // プロパティをMapに格納して全取得
            Map props;
            try {
                props = HBeans.getProperties(bean);
            } catch (Exception e) {
                super.println();
                return;
            }
            // 出力開始
            // クラス名を出力する
            this.print(bean.getClass().getName());
            // 全プロパティを出力する
            super.println('{');
            for (Iterator i = props.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                if (!"class".equals(e.getKey())) {
                    this.printsp(2);
                    this.print((String) e.getKey());
                    super.print('=');
                    this.print(e.getValue());
                    super.println();
                }
            }
            super.println('}');
        }

    }

    /**
     * 全ての JavaBean の保有するプロパティを全て表形式で出力します。
     * 
     * <pre>
     *     &lt;ArrayClassType(length)&gt;[length]
     *     [
     *       [0]&lt;ClassName&gt;{
     *         property1=value1
     *         property2=value2
     *         property3=value3
     *         …
     *       }
     *       [1]&lt;ClassName&gt;{
     *         property1=value1
     *         property2=value2
     *         property3=value3
     *         …
     *       }
     *       …
     *     ]
     * </pre>
     * 
     * @param beans
     *            出力する JavaBean の配列
     */
    public void printBeansTbl(Object[] beans) {
        synchronized (this) {
            // 全Beanを出力
            this.printClass(beans);
            super.println();
            super.println('[');
            for (int i = 0; i < beans.length; i++) {
                this.printsp(2);
                super.print('[' + i + ']');
                this.printBeanTbl(beans[i]);
            }

            super.println(']');
        }
    }

    /**
     * public, static, final であるフィールドを全て出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       public static final [field name1] = [field value1]
     *       public static final transient [field name2] = [field value2]
     *       ･･･
     *     }
     * </pre>
     * 
     * @param c
     *            出力するクラス
     */
    public void printConstants(Class c) {
        synchronized (this) {
            try {
                // クラス名を出力する
                this.printClass(c);
                // 全フィールドを出力する
                Field[] fields = c.getFields();
                super.println('{');
                for (int i = 0; i < fields.length; i++) {
                    int modifiers = fields[i].getModifiers();
                    // public and static and final
                    if (Modifier.isPublic(modifiers) && Modifier.isStatic(modifiers) && Modifier.isFinal(modifiers)) {
                        this.printsp(2);
                        this.print("public static final ");
                        if (Modifier.isTransient(modifiers)) {
                            this.print("transient ");
                        }
                        this.print(fields[i].getName());
                        this.print(" = ");
                        this.println(fields[i].get(null));
                    }
                }
                super.println('}');
            } catch (Exception e) {
                // no op
            }
        }
    }

    /**
     * public であるフィールドを全て出力します。
     * 
     * <pre>
     *     &lt;ClassName&gt;{
     *       public static final java.lang.String foo = "foo"
     *       public static  java.lang.String foo = "foo"
     *       public final  java.lang.String foo = "foo"
     *       public java.lang.String foo = "foo"
     *       public transient  java.lang.String foo = "foo"
     *       public volatile java.lang.String foo = "foo"
     *       ･･･
     *     }  
     * </pre>
     * 
     * @param o
     *            出力するオブジェクト
     */
    public void printFields(Object o) {
        synchronized (this) {
            try {
                // クラス名を出力する
                this.printClass(o);
                // 全フィールドを出力する
                Field[] fields = o.getClass().getFields();
                super.println('{');
                for (int i = 0; i < fields.length; i++) {
                    int modifiers = fields[i].getModifiers();
                    // public
                    if (Modifier.isPublic(modifiers)) {
                        this.printsp(2);
                        this.print("public ");
                        if (Modifier.isStatic(modifiers)) {
                            this.print("static ");
                        }
                        if (Modifier.isFinal(modifiers)) {
                            this.print("final ");
                        }
                        if (Modifier.isTransient(modifiers)) {
                            this.print("transient ");
                        }
                        if (Modifier.isVolatile(modifiers)) {
                            this.print("volatile ");
                        }
                        this.print(fields[i].getType());
                        this.print(' ');
                        this.print(fields[i].getName());
                        this.print(" = ");
                        this.println(fields[i].get(o));
                    }
                }
                super.println('}');
            } catch (Exception e) {
                // no op
            }
        }
    }

    /*
     * Class and null
     */

    /**
     * クラス名を出力します。
     * 
     * @param o
     *            出力するオブジェクト
     */
    public void printClass(Object o) {
        synchronized (this) {
            if (o == null) {
                return;
            }
            Class clazz = (o instanceof Class) ? ((Class) o) : o.getClass();
            super.print('<');
            if (clazz.isArray()) {
                super.print(clazz.getComponentType().getName());
                super.print('[');
                super.print(Array.getLength(o));
                super.print(']');
            } else {
                super.print(clazz.getName());
            }
            super.print('>');
            if (o instanceof Collection) {
                super.print('[');
                super.print(((Collection) o).size());
                super.print(']');
            } else if (o instanceof Map) {
                super.print('[');
                super.print(((Map) o).size());
                super.print(']');
            }
        }
    }

    /**
     * 型情報を付けて出力します。
     * 
     * @param value
     *            出力値
     */
    public void printType(Object value) {
        synchronized (this) {
            this.printClass(value);
            super.print(':');
            this.print(value);
        }
    }

    /**
     * 型情報を付けて出力し改行します。
     * 
     * @param value
     *            出力値
     */
    public void printTypeln(Object value) {
        synchronized (this) {
            this.printClass(value);
            super.print(':');
            this.println(value);
        }
    }

    /**
     * null 値の文字列表現を出力します。
     */
    public void printNull() {
        this.print("null");
    }

    /*
     * 時刻型
     */

    /**
     * 日付型を出力します。
     * 
     * @param date
     *            出力値
     */
    public void printDate(Date date) {
        this.print(formatDate(date));
    }

    /**
     * 現在時刻を出力します。
     */
    public void printCurrentTime() {
        this.printDate(new java.util.Date(System.currentTimeMillis()));
    }

    /**
     * 正規表現パターンを出力します。
     * 
     * @param pattern
     *            正規表現パターン
     */
    public void printPattern(Pattern pattern) {
        super.print(pattern.pattern());
        super.print(',');
        super.print(pattern.flags());
    }

    /*
     * 比較
     */

    /**
     * JavaBean の保有する同一名のプロパティを全て比較しその情報を出力する
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     */
    public void printCompBean(Object from, Object to) {
        synchronized (this) {
            // プロパティをMapに格納して全取得
            Map fromProp;
            Map toProp;
            try {
                fromProp = HBeans.getProperties(from);
                toProp = HBeans.getProperties(to);
            } catch (Exception e) {
                super.println();
                return;
            }
            // 出力開始
            this.printCompMap(fromProp, toProp);
        }

    }

    /**
     * JavaBean の保有する同一名のプロパティを全て比較しその情報を出力する
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     */
    public void printCompMap(Map from, Map to) {
        synchronized (this) {
            // 出力開始
            for (Iterator i = from.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                Object fromValue = e.getValue();
                Object toValue = to.get(e.getKey());

                // 要素を全て厳密に比較する
                if (deepEquals(fromValue, toValue)) {
                    this.println("TRUE", e.getKey(), fromValue, toValue);
                } else {
                    this.println("FALSE", e.getKey(), fromValue, toValue);
                }
            }
        }
    }

    /*
     * Limit
     */

    /**
     * 反復処理を行う要素数の上限値を返却します。
     * 
     * @return 反復処理を行う要素数の上限値
     */
    public int getRepetitionLimit() {
        return repetitionLimit;
    }

    /**
     * 反復処理を行う要素数の上限値を格納します。
     * 
     * @param limit
     *            反復処理を行う要素数の上限値
     */
    public void setRepetitionLimit(int limit) {
        if (0 > limit) {
            throw new IllegalArgumentException("0 > " + limit);
        }
        synchronized (this) {
            this.repetitionLimit = limit;
        }
    }

    /*
     * Helper
     */

    /**
     * 指定された配列の「深層内容」の文字列表現を返却します。<br>
     * REPETITION_LIMITの要素数まで出力します。
     * 
     * @param a
     *            出力する配列
     */
    protected void printDeepArrays(Object a) {
        synchronized (this) {
            if (a == null) {
                this.printNull();
                return;
            }
            Class aClass = a.getClass();
            if (aClass.isArray()) {
                // element is an array of object references
                printDeepArrays(a, new VariableInteger(), 0);
            } else {
                this.print(a);
            }
        }
    }

    /* deepToString(Object[] a) の再起メソッドです。 */
    protected void printDeepArrays(Object a, VariableInteger count, int row) {
        if (a == null) {
            this.printNull();
            return;
        }
        super.print('[');
        for (int i = 0; i < Array.getLength(a); i++) {
            if (count.intValue() >= getRepetitionLimit()) {
                super.print("...");
                break;
            }
            count.increment();

            if (i != 0) {
                this.print(", ");
            }

            Object e = Array.get(a, i);
            if (e == null) {
                this.printNull();
            } else {
                Class eClass = e.getClass();
                if (eClass.isArray()) {
                    // element is an array of object references
                    printDeepArrays(e, count, (row + 1));
                } else {
                    this.print(e);
                }
            }
        }
        super.print(']');
    }

    /**
     * 日付型の書式で日付文字列に変換します。
     * <p>
     * 変換形式：
     * <ul>
     * <li>java.sql.Timestamp = yyyy-mm-dd hh:mm:ss.fffffffff
     * <li>java.sql.Time = hh:mm:ss
     * <li>java.util.Date = yyyy-mm-dd hh:mm:ss.sss
     * <ul>
     * </p>
     * 
     * @param date
     *            日付
     * @return 日付文字列
     */
    protected static Object formatDate(Object date) {
        if (date instanceof Timestamp) {
            String s = formatDate((java.util.Date) date, "yyyy-MM-dd HH:mm:ss");
            return s + '.' + ((Timestamp) date).getNanos();
        } else if (date instanceof Time) {
            return formatDate((java.util.Date) date, "HH:mm:ss");
        } else if (date instanceof Date) {
            return formatDate((java.util.Date) date, "yyyy-MM-dd HH:mm:ss.SSS");
        } else {
            return date;
        }
    }

    /**
     * 日付型をフォーマット変換します。
     * 
     * @param date
     *            日付
     * @param pattern
     *            書式
     * @return 日付文字列
     */
    protected static String formatDate(java.util.Date date, String pattern) {
        return HFmt.formatDate(date, pattern);
    }

    /**
     * 厳密な同一性比較を行う。 <br>
     * null の比較と 配列・多次元配列の内部要素を含む同値性比較を行い、以外は通常の比較と同一。
     * 
     * @param from
     *            比較元
     * @param to
     *            比較先
     * @return 同一の場合True
     */
    protected static boolean deepEquals(Object from, Object to) {
        // どちらかがNullの場合
        if (from == null || to == null) {
            // 双方Nullの場合True
            return (from == to);
        }
        // 双方配列の場合内部要素を含め比較する
        if (from.getClass().isArray() && to.getClass().isArray()) {
            // 配列のコンポーネント型が違う場合False
            if (!from.getClass().getComponentType().equals(from.getClass().getComponentType())) {
                return false;
            }
            // 配列長が違う場合False
            int size = Array.getLength(from);
            if (size != Array.getLength(to)) {
                return false;
            }
            // 内部要素を全比較
            for (int i = 0; i < size; i++) {
                Object o1 = Array.get(from, i);
                Object o2 = Array.get(to, i);
                // 片方がNullの場合False
                if (o1 == null || o2 == null) {
                    if (o1 != o2) {
                        return false;
                    }
                } else {
                    // 再起的に比較
                    if (!deepEquals(o1, o2)) {
                        return false;
                    }
                }
            }
            // 全要素の比較結果同一
            return true;
        }
        // 以外通常比較
        return from.equals(to);

    }
}
