/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.lang;

/**
 * 変更可能な Float 型を提供します。
 */
public final class VariableFloat implements VariableNumber, Comparable {

    /** serialVersionUID */
    private static final long serialVersionUID = -8150734737963552038L;

    /** 変数値. */
    private float value;

    /**
     * Float 型の初期値（0.0）で初期化します.
     */
    public VariableFloat() {
        this((float) 0.0);
    }

    /**
     * 指定の値で初期化します.
     * 
     * @param value
     *            初期値
     */
    public VariableFloat(float value) {
        this.value = value;
    }

    /**
     * 指定の値を float にキャストして初期化します.
     * 
     * @param value
     *            初期値
     */
    public VariableFloat(double value) {
        this.value = (float) value;
    }

    /**
     * 文字列を Float.valueOf(s).floatValue() で変換した値で初期化します.
     * 
     * @param s
     *            初期値
     * @throws NumberFormatException
     *             引数が float として認識出来ない場合
     */
    public VariableFloat(String s) throws NumberFormatException {
        this.value = Float.valueOf(s).floatValue();
    }

    /**
     * 指定の値の Number.floatValue() で初期化します.
     * 
     * @param value
     *            初期値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public VariableFloat(Number value) {
        this.value = value.floatValue();
    }

    /**
     * 指定の値の VariableNumber.floatValue() で初期化します.
     * 
     * @param value
     *            初期値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public VariableFloat(VariableNumber value) {
        this.value = value.floatValue();
    }

    /*
     * math
     */

    /**
     * 現在の値を返却します。
     * 
     * @return 現在の値
     */
    public float get() {
        return this.value;
    }

    /**
     * 新たな値を格納します。
     * 
     * @param value
     *            格納する値
     */
    public void set(float value) {
        this.value = value;
    }

    /**
     * 格納される変数に対して引数を加算します。
     * 
     * @param val
     *            加算する値
     * @return 加算後の値
     */
    public float add(float val) {
        this.value += val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を減算します。
     * 
     * @param val
     *            減算する値
     * @return 減算後の値
     */
    public float subtract(float val) {
        this.value -= val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を乗算します。
     * 
     * @param val
     *            乗算する値
     * @return 乗算後の値
     */
    public float multiply(float val) {
        this.value *= val;
        return this.value;
    }

    /**
     * 格納される変数に対して引数を除算します。
     * 
     * @param val
     *            除算する値
     * @return 除算後の値
     */
    public float divide(float val) {
        this.value /= val;
        return this.value;
    }

    /**
     * 格納される変数に対して値 1 を増分します。
     * 
     * @return 増分後の値
     */
    public float increment() {
        this.value++;
        return this.value;
    }

    /**
     * 格納される変数に対して値 1 を減分します。
     * 
     * @return 減分後の値
     */
    public float decrement() {
        this.value--;
        return this.value;
    }

    /*
     * value
     */

    /**
     * 現在の値を返却します。
     * 
     * @return 現在の値
     */
    public Number getValue() {
        return Boxing.box(this.value);
    }

    /**
     * 指定の値の Number.floatValue() を格納します。
     * 
     * @param value
     *            格納する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public void setValue(Number value) {
        set(value.floatValue());
    }

    /**
     * 指定の値の VariableNumber.floatValue() を格納します。
     * 
     * @param value
     *            格納する値
     * @throws NullPointerException
     *             引数が null の場合
     */
    public void setValue(VariableNumber value) {
        set(value.floatValue());
    }

    /*
     * Number impl
     */

    /**
     * byte で返却します。
     * 
     * @return 格納する値の byte 表現
     */
    public byte byteValue() {
        return (byte) value;
    }

    /**
     * short で返却します。
     * 
     * @return 格納する値の short 表現
     */
    public short shortValue() {
        return (short) value;
    }

    /**
     * int で返却します。
     * 
     * @return 格納する値の int 表現
     */
    public int intValue() {
        return (int) value;
    }

    /**
     * long で返却します。
     * 
     * @return 格納する値の long 表現
     */
    public long longValue() {
        return (long) value;
    }

    /**
     * float で返却します。
     * 
     * @return 格納する値の float 表現
     */
    public float floatValue() {
        return value;
    }

    /**
     * double で返却します。
     * 
     * @return 格納する値の double 表現
     */
    public double doubleValue() {
        return value;
    }

    /*
     * Comparable Impl
     */

    /**
     * 比較順序を返却します。
     * 
     * @param obj
     *            比較するオブジェクト
     * @return 引数の格納値が大きい場合負数、小さい場合整数、同値の場合零を返却します
     * @throws ClassCastException
     *             引数が VariableFloat 以外の場合
     */
    public int compareTo(Object obj) {
        VariableFloat anotherFloat = (VariableFloat) obj;
        float thisVal = this.value;
        float anotherVal = anotherFloat.value;
        return Float.compare(thisVal, anotherVal);
    }

    /*
     * Object
     */

    /**
     * 同値性を検証します、 VariableFloat 型であり格納値が同値の場合は true と見なします。
     * 
     * @param obj
     *            比較先
     * @return VariableFloat 型であり格納値が同値の場合は true
     */
    public boolean equals(Object obj) {
        if (obj instanceof VariableFloat) {
            return (Float.floatToIntBits(((VariableFloat) obj).floatValue()) == Float.floatToIntBits(value));
        }
        return false;
    }

    /**
     * 現在の値をハッシュ値として返却します。
     * 
     * @return ハッシュ値
     */
    public int hashCode() {
        return Float.floatToIntBits(value);
    }

    /**
     * 変数値の文字列表現（String.valueOf(VariableFloat.getValue())）を返却します。
     * 
     * @return 変数値の文字列表現
     */
    public String toString() {
        return String.valueOf(value);
    }

}