/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.helpers;

import java.beans.IndexedPropertyDescriptor;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import shohaku.core.lang.IntrospectionBeansException;
import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.ObjectCreationProxy;

/**
 * JavaBean を制御するヘルパーメソッド群を提供します。 <br>
 * <br>
 * ライブラリの依存性を最小化する方針から Jakarta の BeanUtils 等の高機能ライブラリは使用しません。 <br>
 * コアライブラリには利用頻度の高い機能のみを定義し、必要に応じて他のライブラリを使用します。
 */
public class HBeans {

    /** 空の引数型の配列 */
    public static final Class[] EMPTY_ARG_TYPES = Constants.EMPTY_CLASSES;

    /** 空の引数値の配列 */
    public static final Object[] EMPTY_ARG_VALUES = Constants.EMPTY_OBJECTS;

    /*
     * PropertyDescriptor
     */

    /**
     * クラスのプロパティ定義を不変コレクションで返却します。
     * 
     * @param clazz
     *            クラス
     * @return ビーンのプロパティ定義
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Collection getPropertyDescriptors(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getPropertyDescriptors();
    }

    /**
     * クラスのプロパティ定義をプロパティ名をキーとする不変マップに格納して返却します。
     * 
     * @param clazz
     *            クラス
     * @return プロパティ定義を格納するマップ
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Map getPropertyDescriptorMap(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getPropertyDescriptorMap();
    }

    /**
     * プロパティ名のプロパティ定義を返却します。<br>
     * 指定のプロパティが存在しない場合は null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            プロパティ名
     * @return プロパティ定義
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static PropertyDescriptor getPropertyDescriptor(Class clazz, String propName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getPropertyDescriptor(propName);
    }

    /**
     * プロパティ名のインデックス付きプロパティ定義を返却します。<br>
     * 指定のインデックス付きプロパティが存在しない場合は null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            プロパティ名
     * @return インデックス付きプロパティ定義
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static IndexedPropertyDescriptor getIndexedPropertyDescriptor(Class clazz, String propName) throws IntrospectionBeansException {
        final PropertyDescriptor pd = getPropertyDescriptor(clazz, propName);
        if (pd instanceof IndexedPropertyDescriptor) {
            return (IndexedPropertyDescriptor) pd;
        }
        return null;
    }

    /*
     * Property Method
     */

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの設定メソッドを検索して返却します。 <br>
     * プロパティの設定メソッドが発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            検索するプロパティ名
     * @param paramTypes
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの設定メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Method getAssignmentWriteProperty(Class clazz, String propName, Class[] paramTypes) throws IntrospectionBeansException {
        return getAssignmentMethod(clazz, getWritePropertyMethodName(propName), paramTypes);
    }

    /**
     * パラメータを割り当てられるアクセス可能なプロパティの取得メソッドを検索して返却します。 <br>
     * プロパティの取得メソッドが発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param propName
     *            検索するプロパティ名
     * @param paramTypes
     *            引数のプロパティ型
     * @return アクセス可能なプロパティの取得メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Method getAssignmentReadProperty(Class clazz, String propName, Class[] paramTypes) throws IntrospectionBeansException {
        final Method method = getAssignmentMethod(clazz, getReadPropertyMethodName(propName), paramTypes);
        if (method == null) {
            return getAssignmentMethod(clazz, getReadBooleanPropertyMethodName(propName), paramTypes);
        }
        return method;
    }

    /*
     * Property
     */

    /**
     * ビーンのプロパティをプロパティ名をキーとするマップに格納して返却します。
     * 
     * @param bean
     *            ビーン
     * @return プロパティを格納するマップ
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Map getProperties(Object bean) throws IntrospectionBeansException {
        if (bean == null) {
            throw new NullPointerException("argument is null.");
        }

        final Class clazz = bean.getClass();
        String propName = null;
        try {

            final Collection pds = getPropertyDescriptors(clazz);
            final Map props = new HashMap(pds.size());

            for (Iterator i = pds.iterator(); i.hasNext();) {

                final PropertyDescriptor pd = (PropertyDescriptor) i.next();
                propName = pd.getName();

                if (!"class".equals(propName)) {
                    Method method = pd.getReadMethod();
                    if (method != null) {
                        props.put(propName, method.invoke(bean, (Object[]) null));
                    }
                }

            }
            return props;

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        }
    }

    /**
     * マップの全要素をプロパティに格納して返却します。
     * 
     * @param bean
     *            ビーン
     * @param props
     *            プロパティを格納するマップ
     * @return 引数のビーン
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Object setProperties(Object bean, Map props) throws IntrospectionBeansException {
        if (bean == null) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        String propName = null;
        Object propValue = null;
        try {

            final Map propsDesc = getPropertyDescriptorMap(clazz);

            for (Iterator i = props.entrySet().iterator(); i.hasNext();) {
                Map.Entry e = (Map.Entry) i.next();
                propName = (String) e.getKey();// throw ClassCastException
                propValue = e.getValue();

                // プロパティ定義を取得
                final PropertyDescriptor pd = (PropertyDescriptor) propsDesc.get(propName);
                if (pd == null) {
                    throw new IntrospectionBeansException(HLog.log("not find property. ", clazz, propName));
                }

                // 書き込みメソッドを取得
                Method method = pd.getWriteMethod();
                if (method == null) {
                    throw new IntrospectionBeansException(HLog.log("property has no setter method. ", clazz, propName));
                }

                // 格納する
                method.invoke(bean, createParameterValues(new Object[] { propValue }));
            }
            // 完了
            return bean;

        } catch (ClassCastException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. kay is not String. ", clazz, props), e);
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName), e);
        }

    }

    /**
     * ビーンからプロパティ名の示すプロパティ値を取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param propName
     *            プロパティ名
     * @return プロパティ値
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Object getProperty(Object bean, String propName) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, propName)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final PropertyDescriptor pd = getPropertyDescriptor(clazz, propName);
        if (pd == null) {
            throw new IntrospectionBeansException(HLog.log("not find property. ", clazz, propName));
        }
        return getProperty(bean, pd);
    }

    /**
     * ビーンからプロパティ定義の示すプロパティを取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param pd
     *            プロパティ定義
     * @return プロパティ
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Object getProperty(Object bean, PropertyDescriptor pd) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, pd)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final String propName = pd.getName();
        try {

            final Method method = pd.getReadMethod();
            if (method == null) {
                throw new IntrospectionBeansException(HLog.log("property has no getter method. ", clazz, propName));
            }
            return method.invoke(bean, (Object[]) null);

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("get property err. ", clazz, propName), e);
        }
    }

    /**
     * ビーンからインデックス付きプロパティ定義の示すプロパティを取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param propName
     *            プロパティ名
     * @param index
     *            インデックス
     * @return プロパティ
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Object getIndexedProperty(Object bean, String propName, Integer index) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, propName)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final IndexedPropertyDescriptor ipd = getIndexedPropertyDescriptor(clazz, propName);
        if (ipd == null) {
            throw new IntrospectionBeansException(HLog.log("not find indexed property. ", clazz, propName));
        }
        return getIndexedProperty(bean, ipd, index);
    }

    /**
     * ビーンからインデックス付きプロパティ定義の示すプロパティを取得して返却します。
     * 
     * @param bean
     *            ビーン
     * @param ipd
     *            java.beans.IndexedPropertyDescriptor
     * @param index
     *            インデックス
     * @return プロパティ
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static Object getIndexedProperty(Object bean, IndexedPropertyDescriptor ipd, Integer index) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, ipd)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final String propName = ipd.getName();
        try {

            final Method method = ipd.getIndexedReadMethod();
            if (method == null) {
                throw new IntrospectionBeansException(HLog.log("property has no getter indexed method. ", clazz, propName, index));
            }
            return method.invoke(bean, new Object[] { index });

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("get indexed property err. ", clazz, propName, index), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("get indexed property err. ", clazz, propName, index), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("get indexed property err. ", clazz, propName, index), e);
        }
    }

    /**
     * ビーンからプロパティ名の示すプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param propName
     *            プロパティ名
     * @param newValue
     *            格納する値
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static void setProperty(Object bean, String propName, Object newValue) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, propName)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final PropertyDescriptor pd = getPropertyDescriptor(clazz, propName);
        if (pd == null) {
            throw new IntrospectionBeansException(HLog.log("not find property. ", clazz, propName));
        }
        setProperty(bean, pd, newValue);
    }

    /**
     * ビーンのプロパティ定義の示すプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param pd
     *            プロパティ定義
     * @param newValue
     *            格納する値
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static void setProperty(Object bean, PropertyDescriptor pd, Object newValue) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, pd)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final String propName = pd.getName();
        try {

            final Method method = pd.getWriteMethod();
            if (method == null) {
                throw new IntrospectionBeansException(HLog.log("property has no setter method. ", clazz, propName));
            }
            method.invoke(bean, new Object[] { newValue });

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, newValue), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, newValue), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, newValue), e);
        }
    }

    /**
     * ビーンからプロパティ名の示すインデックス付きプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param propName
     *            プロパティ名
     * @param index
     *            インデックス
     * @param newValue
     *            格納する値
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static void setIndexedProperty(Object bean, String propName, Integer index, Object newValue) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, propName)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final IndexedPropertyDescriptor ipd = getIndexedPropertyDescriptor(clazz, propName);
        if (ipd == null) {
            throw new IntrospectionBeansException(HLog.log("not find indexed property. ", clazz, propName, index));
        }
        setIndexedProperty(bean, ipd, index, newValue);
    }

    /**
     * ビーンのプロパティ定義の示すインデックス付きプロパティに値を格納します。
     * 
     * @param bean
     *            ビーン
     * @param ipd
     *            プロパティ定義
     * @param index
     *            インデックス
     * @param newValue
     *            格納する値
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static void setIndexedProperty(Object bean, IndexedPropertyDescriptor ipd, Integer index, Object newValue) throws IntrospectionBeansException {
        if (HEval.isOrNull(bean, ipd)) {
            throw new NullPointerException("argument is null.");
        }
        final Class clazz = bean.getClass();
        final String propName = ipd.getName();
        try {

            final Method method = ipd.getIndexedWriteMethod();
            if (method == null) {
                throw new IntrospectionBeansException(HLog.log("property has no setter indexed method. ", clazz, propName, index, newValue));
            }
            method.invoke(bean, new Object[] { index, newValue });

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, index, newValue), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, index, newValue), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("set property err. ", clazz, propName, index, newValue), e);
        }
    }

    /**
     * 指定されたクラスに、指定の接頭辞を持つプロパティがあるか検証します。
     * 
     * @param clazz
     *            クラス
     * @param prefix
     *            接尾辞
     * @param propName
     *            プロパティ名
     * @return 接頭辞名を持つプロパティがある場合は true
     * @throws IntrospectionBeansException
     *             プロパティのアクセスに失敗した場合
     */
    public static boolean containsProperty(Class clazz, String prefix, String propName) throws IntrospectionBeansException {
        return containsMethod(clazz, propertyAsMethodName(prefix, propName));
    }

    /*
     * Method
     */

    /**
     * 同一のメソッド名のアクセス可能な Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変マップ
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Map getMethodGroup(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getMethodGroup();
    }

    /**
     * 指定された名前のアクセス可能な Method を不変コレクションに格納して返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return 不変コレクション、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Collection getMethods(Class clazz, String methodName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getMethods(methodName);
    }

    /**
     * 指定された名前と引数が無い、アクセス可能な Method を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getMethod(Class clazz, String methodName) throws IntrospectionBeansException {
        return getMethod(clazz, methodName, EMPTY_ARG_TYPES);
    }

    /**
     * 指定された名前とパラメータ型のアクセス可能な Method を返却します。 <br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @param paramTypes
     *            パラメータ型
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getMethod(Class clazz, String methodName, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getMethod(methodName, paramTypes);
    }

    /**
     * 指定された名前と割り当て可能なパラメータ型で最初に見付かったアクセス可能な Method を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @param paramTypes
     *            パラメータ型
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getAssignmentMethod(Class clazz, String methodName, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAssignmentMethod(methodName, paramTypes);
    }

    /**
     * 指定された名前とパラメータ型でアクセス可能な Method があるか検証します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return メソッド名に一致する Method がある場合は true
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static boolean containsMethod(Class clazz, String methodName) throws IntrospectionBeansException {
        return (null != getBeanDescriptor(clazz).getMethods(methodName));
    }

    /*
     * Accessible Method
     */

    /**
     * 同一のメソッド名の特権でアクセス可能な Method を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変マップ
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Map getAccessibleMethodGroup(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleMethodGroup();
    }

    /**
     * 指定された名前の特権でアクセス可能な Method を不変コレクションに格納して返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return 不変コレクション、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Collection getAccessibleMethods(Class clazz, String methodName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleMethods(methodName);
    }

    /**
     * 指定された名前と引数が無い特権でアクセス可能な Method を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getAccessibleMethod(Class clazz, String methodName) throws IntrospectionBeansException {
        return getAccessibleMethod(clazz, methodName, EMPTY_ARG_TYPES);
    }

    /**
     * 指定された名前とパラメータ型の特権でアクセス可能な Method を返却します。 <br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @param paramTypes
     *            パラメータ型
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getAccessibleMethod(Class clazz, String methodName, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleMethod(methodName, paramTypes);
    }

    /**
     * 指定された名前と割り当て可能なパラメータ型で最初に見付かった特権でアクセス可能な Method を返却します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @param paramTypes
     *            パラメータ型
     * @return メソッド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static Method getAssignmentAccessibleMethod(Class clazz, String methodName, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAssignmentAccessibleMethod(methodName, paramTypes);
    }

    /**
     * 指定された名前とパラメータ型で特権でアクセス可能な Method があるか検証します。
     * 
     * @param clazz
     *            クラス
     * @param methodName
     *            メソッド名
     * @return メソッド名に一致する Method がある場合は true
     * @throws IntrospectionBeansException
     *             メソッドへのアクセスに失敗した場合
     */
    public static boolean containsAccessibleMethod(Class clazz, String methodName) throws IntrospectionBeansException {
        return (null != getBeanDescriptor(clazz).getAccessibleMethods(methodName));
    }

    /*
     * invoke Method
     */

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。
     * 
     * @param obj
     *            実行するインスタンス
     * @param method
     *            メソッド
     * @param paramValue
     *            パラメータ値
     * @return メソッドの返却値
     * @throws IntrospectionBeansException
     *             メソッドの呼出に失敗した場合
     */
    public static Object invokeMethod(Object obj, Method method, Object paramValue) throws IntrospectionBeansException {
        return invokeMethod(obj, method, new Object[] { paramValue });
    }

    /**
     * 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。
     * 
     * @param obj
     *            実行するインスタンス
     * @param method
     *            メソッド
     * @param paramValues
     *            パラメータ値
     * @return メソッドの返却値
     * @throws IntrospectionBeansException
     *             メソッドの呼出に失敗した場合
     */
    public static Object invokeMethod(Object obj, Method method, Object[] paramValues) throws IntrospectionBeansException {
        if (HEval.isOrNull(method, paramValues)) {
            throw new NullPointerException("argument is null.");
        }
        try {

            if (isStatic(method)) {
                return method.invoke(null, createParameterValues(paramValues));
            }
            if (obj == null) {
                throw new NullPointerException("argument obj is null.");
            }
            return method.invoke(obj, createParameterValues(paramValues));

        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("method invocation err. ", method, paramValues), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("method invocation err. ", method, paramValues), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("method invocation err. ", method, paramValues), e);
        }
    }

    /*
     * Field
     */

    /**
     * 同一のフィールド名のアクセス可能な Field を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変マップ
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Map getFieldGroup(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getFieldGroup();
    }

    /**
     * 指定された名前のアクセス可能な Field を不変コレクションに格納して返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return 不変コレクション、発見できない場合 null
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Collection getFields(Class clazz, String fieldName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getFields(fieldName);
    }

    /**
     * 指定された名前で最初に見付かったのアクセス可能な Field を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return フィールド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Field getField(Class clazz, String fieldName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getField(fieldName);
    }

    /**
     * 指定されたクラスに、指定のフィールド名に一致する、アクセス可能な Field があるか検証します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return フィールド名に一致する Field がある場合は true
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static boolean containsField(Class clazz, String fieldName) throws IntrospectionBeansException {
        return (null != getBeanDescriptor(clazz).getFields(fieldName));
    }

    /*
     * invoke Field
     */

    /**
     * オブジェクトからフィールド名の示すフィールド値を取得して返却します。
     * 
     * @param clazz
     *            フィールドを保有するクラス
     * @param obj
     *            フィールドを保有するオブジェクト
     * @param fieldName
     *            フィールド名
     * @return フィールドの値
     * @throws IntrospectionBeansException
     *             フィールドの呼出に失敗した場合
     */
    public static Object getFieldValue(Class clazz, Object obj, String fieldName) throws IntrospectionBeansException {
        try {
            return getField(clazz, fieldName).get(obj);
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("get field err. ", clazz, fieldName), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("get field err. ", clazz, fieldName), e);
        }
    }

    /**
     * オブジェクトからフィールド名の示すフィールド値を設定して元の値を返却します。
     * 
     * @param clazz
     *            フィールドを保有するクラス
     * @param obj
     *            フィールドを保有するオブジェクト
     * @param fieldName
     *            フィールド名
     * @param newValue
     *            格納するフィールド値
     * @return 元のフィールド値
     * @throws IntrospectionBeansException
     *             フィールドの呼出に失敗した場合
     */
    public static Object setFieldValue(Class clazz, Object obj, String fieldName, Object newValue) throws IntrospectionBeansException {
        try {
            final Field field = getField(clazz, fieldName);
            final Object oldValue = field.get(obj);
            field.set(obj, newValue);
            return oldValue;
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("set field err. ", clazz, fieldName, newValue), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("set field err. ", clazz, fieldName, newValue), e);
        }
    }

    /*
     * Accessible Field
     */

    /**
     * 同一のフィールド名の特権でアクセス可能な Field を不変コレクションでマッピングした不変マップを返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変マップ
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Map getAccessibleFieldGroup(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleFieldGroup();
    }

    /**
     * 指定された名前の特権でアクセス可能な Field を不変コレクションに格納して返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return 不変コレクション、発見できない場合 null
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Collection getAccessibleFields(Class clazz, String fieldName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleFields(fieldName);
    }

    /**
     * 指定された名前で最初に見付かったの特権でアクセス可能な Field を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return フィールド、発見できない場合 null
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static Field getAccessibleField(Class clazz, String fieldName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAccessibleField(fieldName);
    }

    /**
     * 指定されたクラスに、指定のフィールド名に一致する、特権でアクセス可能な Field があるか検証します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return フィールド名に一致する Field がある場合は true
     * @throws IntrospectionBeansException
     *             フィールドのアクセスに失敗した場合
     */
    public static boolean containsAccessibleField(Class clazz, String fieldName) throws IntrospectionBeansException {
        return (null != getBeanDescriptor(clazz).getAccessibleFields(fieldName));
    }

    /*
     * invoke Accessible Field
     */

    /**
     * オブジェクトからフィールド名の示すフィールド値を特権で取得して返却します。
     * 
     * @param clazz
     *            フィールドを保有するクラス
     * @param obj
     *            フィールドを保有するオブジェクト
     * @param fieldName
     *            フィールド名
     * @return フィールドの値
     * @throws IntrospectionBeansException
     *             フィールドの呼出に失敗した場合
     */
    public static Object getAccessibleFieldValue(Class clazz, Object obj, String fieldName) throws IntrospectionBeansException {
        try {
            return getAccessibleField(clazz, fieldName).get(obj);
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("get field err. ", clazz, fieldName), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("get field err. ", clazz, fieldName), e);
        }
    }

    /**
     * オブジェクトからフィールド名の示すフィールド値を特権で設定して元の値を返却します。
     * 
     * @param clazz
     *            フィールドを保有するクラス
     * @param obj
     *            フィールドを保有するオブジェクト
     * @param fieldName
     *            フィールド名
     * @param newValue
     *            格納するフィールド値
     * @return 元のフィールド値
     * @throws IntrospectionBeansException
     *             フィールドの呼出に失敗した場合
     */
    public static Object setAccessibleFieldValue(Class clazz, Object obj, String fieldName, Object newValue) throws IntrospectionBeansException {
        try {
            final Field field = getAccessibleField(clazz, fieldName);
            final Object oldValue = field.get(obj);
            field.set(obj, newValue);
            return oldValue;
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("set field err. ", clazz, fieldName, newValue), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("set field err. ", clazz, fieldName, newValue), e);
        }
    }

    /*
     * Constants
     */

    /**
     * ビーンの public static final なフィールドを名前と値でマッピングした不変マップに格納して返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変マップ
     * @throws IntrospectionBeansException
     *             フィールドの取得に失敗した場合
     */
    public static Map getConstantFieldMap(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getConstantFieldMap();
    }

    /**
     * ビーンの public static final なフィールドの値を返却します。
     * 
     * @param clazz
     *            クラス
     * @param fieldName
     *            フィールド名
     * @return public static final なフィールドの値
     * @throws IntrospectionBeansException
     *             フィールドの取得に失敗した場合
     */
    public static Object getConstantFieldValue(Class clazz, String fieldName) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getConstantField(fieldName);
    }

    /*
     * Constructor
     */

    /*  */
    /**
     * アクセス可能な Constructor を不変コレクションに格納して返却します。
     * 
     * @param clazz
     *            クラス
     * @return 不変コレクション
     * @throws IntrospectionBeansException
     *             コンストラクタの取得に失敗した場合
     */
    public static Collection getConstructors(Class clazz) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getConstructors();
    }

    /**
     * 引数が無いアクセス可能な Constructor を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @return コンストラクタ、発見できない場合 null
     * @throws IntrospectionBeansException
     *             コンストラクタの取得に失敗した場合
     */
    public static Constructor getConstructor(Class clazz) throws IntrospectionBeansException {
        return getConstructor(clazz, EMPTY_ARG_TYPES);
    }

    /**
     * 指定されたパラメータ型のアクセス可能な Constructor を返却します。<br>
     * 発見できない場合 null を返却します。
     * 
     * @param clazz
     *            クラス
     * @param paramTypes
     *            パラメータ型
     * @return コンストラクタ、発見できない場合 null
     * @throws IntrospectionBeansException
     *             コンストラクタの取得に失敗した場合
     */
    public static Constructor getConstructor(Class clazz, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getConstructor(paramTypes);
    }

    /*  */
    /**
     * 指定された割り当て可能なパラメータ型で最初に見付かったのアクセス可能な Constructor を返却します。
     * 
     * @param clazz
     *            クラス
     * @param paramTypes
     *            パラメータ型
     * @return コンストラクタ、発見できない場合 null
     * @throws IntrospectionBeansException
     *             コンストラクタの取得に失敗した場合
     */
    public static Constructor getAssignmentConstructor(Class clazz, Class[] paramTypes) throws IntrospectionBeansException {
        return getBeanDescriptor(clazz).getAssignmentConstructor(paramTypes);
    }

    /*
     * new instance
     */

    /**
     * 空コンストラクタからインスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @return 生成されたインスタンス
     * @throws IntrospectionBeansException
     *             インスタンスの生成に失敗した場合
     */
    public static Object newInstance(Class clazz) throws IntrospectionBeansException {
        return newInstance(clazz, EMPTY_ARG_TYPES, EMPTY_ARG_VALUES);
    }

    /**
     * 指定されたパラメータ値を割当て可能なコンストラクタを検索し、最初に見付かったコンストラクタからインスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @param paramValues
     *            パラメータ値
     * @return 生成されたインスタンス
     * @throws IntrospectionBeansException
     *             インスタンスの生成に失敗した場合
     */
    public static Object newInstance(Class clazz, Object[] paramValues) throws IntrospectionBeansException {
        return newInstance(clazz, toClassArray(paramValues), paramValues);
    }

    /**
     * 指定されたパラメータ型を割当て可能なコンストラクタを検索し、最初に見付かったコンストラクタからインスタンスを生成して返却します。
     * 
     * @param clazz
     *            クラス
     * @param paramTypes
     *            パラメータ型
     * @param paramValues
     *            パラメータ値
     * @return 生成されたインスタンス
     * @throws IntrospectionBeansException
     *             インスタンスの生成に失敗した場合
     */
    public static Object newInstance(Class clazz, Class[] paramTypes, Object[] paramValues) throws IntrospectionBeansException {
        try {
            return getAssignmentConstructor(clazz, paramTypes).newInstance(createParameterValues(paramValues));
        } catch (SecurityException e) {
            throw new IntrospectionBeansException(HLog.log("new instance err. ", clazz, paramTypes, paramValues), e);
        } catch (IllegalArgumentException e) {
            throw new IntrospectionBeansException(HLog.log("new instance err. ", clazz, paramTypes, paramValues), e);
        } catch (InstantiationException e) {
            throw new IntrospectionBeansException(HLog.log("new instance err. ", clazz, paramTypes, paramValues), e);
        } catch (IllegalAccessException e) {
            throw new IntrospectionBeansException(HLog.log("new instance err. ", clazz, paramTypes, paramValues), e);
        } catch (InvocationTargetException e) {
            throw new IntrospectionBeansException(HLog.log("new instance err. ", clazz, paramTypes, paramValues), e);
        }
    }

    /*
     * Utils
     */

    /**
     * 第一引数に対して第二引数が全て割り当て可能か検証します。
     * 
     * @param fromTypes
     *            比較元のクラスリスト
     * @param toTypes
     *            比較先のクラスリスト
     * @return 第一引数に対して第二引数が全て割り当て可能の場合は true
     */
    public static boolean isAssignmentCompatible(Class[] fromTypes, Class[] toTypes) {
        if (fromTypes.length != toTypes.length) {
            return false;
        }
        for (int i = 0; i < fromTypes.length; i++) {
            if (!isAssignmentCompatible(fromTypes[i], toTypes[i])) {
                return false;
            }
        }
        return true;
    }

    /**
     * 第一引数に対して第二引数が割り当て可能か検証します。
     * 
     * @param fromType
     *            比較元のクラス
     * @param toType
     *            比較先のクラス
     * @return 第一引数に対して第二引数が割り当て可能の場合は true
     */
    public static boolean isAssignmentCompatible(Class fromType, Class toType) {
        // 単純に割り当て可能か検証する
        if (fromType.isAssignableFrom(toType)) {
            return true;
        }
        // ラッパー型が一致する場合許可
        return HClass.boxEquals(fromType, toType);
    }

    /**
     * オブジェクトの配列を基にその要素のクラスの配列を生成して返却する。
     * 
     * @param objects
     *            生成基のオブジェクトの配列
     * @return クラスの配列
     */
    public static Class[] toClassArray(Object[] objects) {
        Class[] cs = new Class[objects.length];
        for (int i = 0; i < objects.length; i++) {
            cs[i] = objects[i].getClass();
        }
        return cs;
    }

    /**
     * 指定の接頭辞を持つプロパティのメソッド名を返却します。
     * 
     * @param prefix
     *            接尾辞
     * @param propName
     *            プロパティ名
     * @return プロパティのメソッド名
     */
    public static String propertyAsMethodName(String prefix, String propName) {
        return prefix + HCnv.capitalize(propName);
    }

    /*
     * Utils
     */

    static boolean isStatic(Member member) {
        return Modifier.isStatic(member.getModifiers());
    }

    static BeanIntrospectDescriptor getBeanDescriptor(Class clazz) throws IntrospectionBeansException {
        return BeanIntrospectDescriptor.forClass(clazz);
    }

    /* 新規のパラメータの値を生成して返却します。 */
    static Object[] createParameterValues(Object[] srcValues) throws IntrospectionBeansException {
        final Object[] newValues;
        try {
            newValues = new Object[srcValues.length];
            for (int i = 0; i < srcValues.length; i++) {
                Object o = srcValues[i];
                if (o instanceof ObjectCreationProxy) {
                    o = ((ObjectCreationProxy) o).create();
                }
                newValues[i] = o;
            }
        } catch (ObjectCreationException e) {
            if (e.getCause() instanceof IntrospectionBeansException) {
                throw (IntrospectionBeansException) e.getCause();
            }
            throw new IntrospectionBeansException(HLog.log("creat parameters err. ", srcValues), e);
        }
        return newValues;
    }

    /* Get プロパティのメソッド名を返却します。 */
    static String getReadPropertyMethodName(String propName) {
        return getPropertyMethodName("get", propName);
    }

    /* Get プロパティのメソッド名を返却します。 */
    static String getReadBooleanPropertyMethodName(String propName) {
        return getPropertyMethodName("is", propName);
    }

    /* Set プロパティのメソッド名を返却します。 */
    static String getWritePropertyMethodName(String propName) {
        return getPropertyMethodName("set", propName);
    }

    /* プロパティのメソッド名を返却します。 */
    static String getPropertyMethodName(String prefix, String propName) {
        if (HEval.isEmpty(propName)) {
            return prefix;
        }
        char[] chars = propName.toCharArray();
        chars[0] = Character.toUpperCase(chars[0]);
        return new StringBuffer(prefix).append(chars).toString();
    }

}
