/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections.cache;

import java.io.Serializable;
import java.util.WeakHashMap;

import shohaku.core.collections.Cache;

/**
 * 弱参照でキーでエントリを格納するハッシュテーブル構造を持つキャッシュを提供します。<br>
 * リサイズオペレーションでは削除されるエントリに規準は有りません。 <br>
 * <br>
 * このキャッシュはデータの格納に java.util.WeakHashMap を使用するため、その特性を継承しています。<br>
 * 特性の詳細は java.util.WeakHashMap を参照してください。
 * 
 * @see java.util.WeakHashMap
 */
public class WeakHashCache extends AbstractCache implements Cloneable, Serializable {

    /* serialVersionUID */
    private static final long serialVersionUID = 874263548114519647L;

    /* デフォルトの最大キャッシュサイズ */
    static final int MAXIMUM_SIZE = 1 << 10;

    /* デフォルトの初期容量 */
    static final int INITIAL_CAPACITY = 32;

    /* デフォルトの負荷係数 */
    static final float LOAD_FACTOR = 0.75f;

    /**
     * デフォルトの初期容量 (32) と負荷係数 (0.75f)、最大キャッシュサイズ (1 << 10) で初期化します。
     */
    public WeakHashCache() {
        this(INITIAL_CAPACITY);
    }

    /**
     * 指定された初期容量とデフォルトの負荷係数 (0.75f)、最大キャッシュサイズ (1 << 10) で初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     */
    public WeakHashCache(int initialCapacity) {
        this(initialCapacity, LOAD_FACTOR);
    }

    /**
     * 指定された初期容量と負荷係数とデフォルトの最大キャッシュサイズ (1 << 10) で初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     */
    public WeakHashCache(int initialCapacity, float loadFactor) {
        this(initialCapacity, loadFactor, MAXIMUM_SIZE);
    }

    /**
     * 指定されたキャッシュの全てのエントリを格納して初期化します。
     * 
     * @param t
     *            キャッシュ
     */
    public WeakHashCache(Cache t) {
        this(Math.max((int) (t.size() / LOAD_FACTOR) + 1, INITIAL_CAPACITY), LOAD_FACTOR, MAXIMUM_SIZE);
    }

    /**
     * 指定された初期容量と負荷係数、最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param maxSize
     *            最大キャッシュサイズ
     */
    public WeakHashCache(int initialCapacity, float loadFactor, int maxSize) {
        this(initialCapacity, loadFactor, maxSize, null);
    }

    /**
     * 指定された初期容量と負荷係数、最大キャッシュサイズで初期化します。
     * 
     * @param initialCapacity
     *            初期容量
     * @param loadFactor
     *            負荷係数
     * @param maxSize
     *            最大キャッシュサイズ
     * @param t
     *            キャッシュ
     */
    WeakHashCache(int initialCapacity, float loadFactor, int maxSize, Cache t) {
        super(new WeakHashMap(initialCapacity, loadFactor), maxSize, t);
    }

    /*
     * Cloneable
     */

    /**
     * クローンを生成して返却します。
     * 
     * @return このオブジェクトのクローン
     * @see java.lang.Object#clone()
     */
    public Object clone() {
        synchronized (mutex) {
            WeakHashCache result = null;
            try {
                result = (WeakHashCache) super.clone();
            } catch (CloneNotSupportedException e) {
                // assert false;
            }
            return result;
        }
    }
}
