/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.RandomAccess;

import shohaku.core.collections.decorator.DecoratedList;
import shohaku.core.collections.decorator.DecoratedRandomAccessList;
import shohaku.core.lang.Predicate;

/**
 * List インタフェースのユーティリティメソッドを提供します。
 */
public class ListUtils {

    /**
     * 実装型に固有の機能へのアクセスを制限したリストを返却します。
     * 
     * @param list
     *            ラップするリスト
     * @return 実装を装飾したリスト
     */
    public static List unextendsList(List list) {
        if (list instanceof RandomAccess) {
            return new DecoratedRandomAccessList(list);
        } else {
            return new DecoratedList(list);
        }

    }

    /**
     * リストに反復子の要素を全て追加します。
     * 
     * @param list
     *            追加先のリスト
     * @param elements
     *            追加する反復子
     * @return 引数のリスト参照
     */
    public static List addAll(List list, Iterator elements) {
        while (elements.hasNext()) {
            list.add(elements.next());
        }
        return list;
    }

    /**
     * リスト内の指定された位置に、反復子の要素を全て追加します。<br>
     * 現在その位置にある要素と後続の要素は右に移動します (インデックス値が増加)。
     * 
     * @param list
     *            追加先のリスト
     * @param index
     *            最初の要素を挿入する位置のインデックス
     * @param elements
     *            追加する反復子
     * @return 引数のリスト参照
     */
    public static List addAll(List list, int index, Iterator elements) {
        int i = index;
        while (elements.hasNext()) {
            list.add(i++, elements.next());
        }
        return list;
    }

    /**
     * リスト内の指定された位置に、反復子の要素を全て追加します。<br>
     * 現在その位置にある要素と後続の要素は右に移動します (インデックス値が増加)。
     * 
     * @param list
     *            追加先のリスト
     * @param index
     *            最初の要素を挿入する位置のインデックス
     * @param elements
     *            追加する反復子
     * @param predicate
     *            要素を追加するか評価するクロージャ
     * @return 引数のリスト参照
     */
    public static List addAll(List list, int index, Iterator elements, Predicate predicate) {
        int i = index;
        while (elements.hasNext()) {
            Object o = elements.next();
            if (predicate.evaluate(o)) {
                list.add(i++, elements.next());
            }
        }
        return list;
    }

    /**
     * クロージャが真と評価する要素をリストに追加します。
     * 
     * @param list
     *            対象のリスト
     * @param index
     *            最初の要素を挿入する位置のインデックス
     * @param elements
     *            追加する値のコレクション
     * @param predicate
     *            要素を追加するか評価するクロージャ
     * @return リストの参照
     */
    public static List addAll(List list, int index, Collection elements, Predicate predicate) {
        return addAll(list, index, elements.iterator(), predicate);
    }

}
