/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.lang.reflect.Array;
import java.util.Collection;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;

import shohaku.core.collections.decorator.DecoratedIterator;
import shohaku.core.collections.decorator.UnmodifiableIterator;
import shohaku.core.helpers.HEval;
import shohaku.core.lang.Boxing;
import shohaku.core.lang.Predicate;

/**
 * Iterator インタフェースのユーティリティメソッドを提供します。
 */
public class IteratorUtils {

    /*
     * Empty
     */

    /* 空要素の反復子。 */
    static class EmptyIterator implements Iterator {
        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            return false;
        }

        public Object next() {
            throw new NoSuchElementException();
        }

    }

    /* 無限に反復する反復子。 */
    static class InfiniteLoopIterator implements Iterator {
        private final Object v;

        InfiniteLoopIterator(Object infiniteValue) {
            this.v = infiniteValue;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            return true;
        }

        public Object next() {
            return v;
        }

    }

    /* 列挙を反復子のインタフェースに適合させるアダプタ。 */
    static class EnumerationIteratorAdapter implements Iterator {
        private final Enumeration e;

        EnumerationIteratorAdapter(Enumeration e) {
            this.e = e;
        }

        public boolean hasNext() {
            return e.hasMoreElements();
        }

        public Object next() {
            if (!e.hasMoreElements()) {
                throw new NoSuchElementException();
            }
            return e.nextElement();
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }
    }

    /*
     * Array
     */

    static abstract class AbstractArrayIterator implements Iterator {
        protected final int length;

        protected int index = -1;

        AbstractArrayIterator(Object a) {
            if (a == null) {
                throw new NullPointerException();
            }
            this.length = Array.getLength(a);
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            if ((index + 1) < this.length) {
                return true;
            }
            return false;
        }

        public Object next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            return nextObject(++index);
        }

        protected abstract Object nextObject(int inx);
    }

    /*
     * Object Array
     */

    static class ObjectArrayIterator extends AbstractArrayIterator {
        private final Object[] a;

        ObjectArrayIterator(Object[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return a[inx];
        }
    }

    static class CharacterArrayIterator extends AbstractArrayIterator {
        private final char[] a;

        CharacterArrayIterator(char[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class BooleanArrayIterator extends AbstractArrayIterator {
        private final boolean[] a;

        BooleanArrayIterator(boolean[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class ByteArrayIterator extends AbstractArrayIterator {
        private final byte[] a;

        ByteArrayIterator(byte[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class ShortArrayIterator extends AbstractArrayIterator {
        private final short[] a;

        ShortArrayIterator(short[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class IntegerArrayIterator extends AbstractArrayIterator {
        private final int[] a;

        IntegerArrayIterator(int[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class LongArrayIterator extends AbstractArrayIterator {
        private final long[] a;

        LongArrayIterator(long[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class FloatArrayIterator extends AbstractArrayIterator {
        private final float[] a;

        FloatArrayIterator(float[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class DoubleArrayIterator extends AbstractArrayIterator {
        private final double[] a;

        DoubleArrayIterator(double[] a) {
            super(a);
            this.a = a;
        }

        protected Object nextObject(int inx) {
            return Boxing.box(a[inx]);
        }
    }

    static class CharSequenceIterator implements Iterator {
        private final CharSequence chars;

        private int index = -1;

        CharSequenceIterator(CharSequence cs) {
            if (cs == null) {
                throw new NullPointerException();
            }
            this.chars = cs;
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        public boolean hasNext() {
            if ((index + 1) < chars.length()) {
                return true;
            }
            return false;
        }

        public Object next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            return Boxing.box(chars.charAt(++index));
        }

    }

    static class CompositeIterator implements Iterator {
        final Iterator[] iters;

        private Iterator i;

        private int index = -1;

        CompositeIterator(Iterator[] is) {
            this.iters = is;
        }

        public void remove() {
            if (i == null) {
                throw new IllegalStateException();
            }
            i.remove();
        }

        public boolean hasNext() {
            if (i == null) {
                if ((index + 1) < iters.length) {
                    i = iters[++index];
                } else {
                    return false;
                }
            }
            while (!i.hasNext()) {
                if ((index + 1) < iters.length) {
                    i = iters[++index];
                } else {
                    return false;
                }
            }
            return true;
        }

        public Object next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            return i.next();
        }
    }

    static abstract class FilterIterator implements Iterator {

        private final Object NULL = new Object();

        private final Iterator i;

        private Object next = NULL;

        private Boolean hasNext = null;

        FilterIterator(Iterator iter) {
            this.i = iter;
        }

        public void remove() {
            if (this.next == NULL) {
                throw new IllegalStateException();
            }
            i.remove();
        }

        public boolean hasNext() {
            if (this.hasNext != null) {
                return this.hasNext.booleanValue();
            }
            this.hasNext = Boolean.FALSE;
            while (i.hasNext()) {
                Object e = i.next();
                if (accept(e)) {
                    this.next = e;
                    this.hasNext = Boolean.TRUE;
                    break;
                }
            }
            return this.hasNext.booleanValue();
        }

        public Object next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            this.hasNext = null;
            return this.next;
        }

        abstract boolean accept(Object e);
    }

    static class PredicateIterator extends FilterIterator {
        private final Predicate predicate;

        PredicateIterator(Iterator iter, Predicate predicate) {
            super(iter);
            this.predicate = predicate;
        }

        boolean accept(Object e) {
            return predicate.evaluate(e);
        }

    }

    /**
     * 複数の反復子を合成した反復子を生成して返却します。<br>
     * 配列インデックス順に全ての反復子を走査します。
     * 
     * @param is
     *            合成する反復子の配列
     * @return 複数の反復子を合成した反復子
     */
    public static Iterator compositeIterator(Iterator[] is) {
        return new CompositeIterator(is);
    }

    /**
     * 空要素の反復子を生成して返却します。
     * 
     * @return 空要素の反復子
     */
    public static Iterator emptyIterator() {
        return new EmptyIterator();
    }

    /**
     * 無限に反復し null を返し続ける更新不可の反復子を生成して返却します。<br>
     * この反復子は終了しません。 常に hasNext() == true を返しまた next() == null を返し続けます。
     * 
     * @return 無限に反復する反復子
     */
    public static Iterator infiniteLoopIterator() {
        return new InfiniteLoopIterator(null);
    }

    /**
     * 無限に反復し infiniteValue を返し続ける更新不可の反復子を生成して返却します。<br>
     * この反復子は終了しません。 常に hasNext() == true を返しまた next() == infiniteValue を返し続けます。
     * 
     * @param infiniteValue
     *            next() から無限回に返される値
     * @return 無限に反復する反復子
     */
    public static Iterator infiniteLoopIterator(Object infiniteValue) {
        return new InfiniteLoopIterator(infiniteValue);
    }

    /*
     * Array Iterator
     */

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     * @throws IllegalArgumentException
     *             引数が配列以外の場合
     */
    public static Iterator arrayIterator(Object a) {
        if (a == null) {
            throw new NullPointerException();
        }
        if (HEval.isArray(a)) {
            Class type = a.getClass().getComponentType();
            if (type.isPrimitive()) {
                if (type == Character.TYPE) {
                    return asIterator((char[]) a);
                } else if (type == Boolean.TYPE) {
                    return asIterator((boolean[]) a);
                } else if (type == Byte.TYPE) {
                    return asIterator((byte[]) a);
                } else if (type == Short.TYPE) {
                    return asIterator((short[]) a);
                } else if (type == Integer.TYPE) {
                    return asIterator((int[]) a);
                } else if (type == Long.TYPE) {
                    return asIterator((long[]) a);
                } else if (type == Float.TYPE) {
                    return asIterator((float[]) a);
                } else if (type == Double.TYPE) {
                    return asIterator((double[]) a);
                }
            } else {
                return asIterator((Object[]) a);
            }
        }
        throw new IllegalArgumentException("is not Array:" + a);
    }

    /**
     * 文字シーケンスの要素を反復する更新不可の反復子を生成して返却します。<br>
     * next() で返される要素は Character 型です。
     * 
     * @param cs
     *            反復する文字シーケンス
     * @return 文字シーケンスの要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(CharSequence cs) {
        return new CharSequenceIterator(cs);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(Object[] a) {
        return new ObjectArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(char[] a) {
        return new CharacterArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(boolean[] a) {
        return new BooleanArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(byte[] a) {
        return new ByteArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(short[] a) {
        return new ShortArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(int[] a) {
        return new IntegerArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(long[] a) {
        return new LongArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(float[] a) {
        return new FloatArrayIterator(a);
    }

    /**
     * 配列の要素を反復する更新不可の反復子を生成して返却します。
     * 
     * @param a
     *            反復する配列
     * @return 配列の要素を反復する更新不可の反復子
     */
    public static Iterator asIterator(double[] a) {
        return new DoubleArrayIterator(a);
    }

    /**
     * 列挙を反復子のインタフェースに適合させて返却します。<br>
     * 返却される反復子は更新不可と為ります。
     * 
     * @param a
     *            列挙
     * @return 列挙を反復子のインタフェースに適合させた更新不可の反復子
     */
    public static Iterator asIterator(Enumeration a) {
        return new EnumerationIteratorAdapter(a);
    }

    /*
     * Functor
     */

    /**
     * 真偽を評価するクロージャが、真と評価する値のみを返す反復子を返却します。
     * 
     * @param i
     *            ラップする反復子
     * @param predicate
     *            真偽を評価するクロージャ
     * @return 真と評価する値のみを返す反復子
     */
    public static Iterator predicateIterator(Iterator i, Predicate predicate) {
        return new PredicateIterator(i, predicate);
    }

    /*
     * Wrapper
     */

    /**
     * 更新不可の反復子に装飾して返却します。
     * 
     * @param i
     *            ラップする反復子
     * @return 更新不可の反復子
     */
    public static Iterator unmodifiableIterator(Iterator i) {
        return new UnmodifiableIterator(i);
    }

    /**
     * 実装型に固有の機能へのアクセスを制限した反復子を返却します。
     * 
     * @param i
     *            ラップする反復子
     * @return 実装を装飾した反復子
     */
    public static Iterator unextendsIterator(Iterator i) {
        return new DecoratedIterator(i);
    }

    /**
     * オブジェクトをコレクションとして解析して推測させる妥当な反復子を生成して返却します。<br>
     * 現在有効なデータ型は 配列、Collection、Enumeration、Iterator、Map、CharSequence と為ります。
     * 
     * @param collection
     *            生成基のコレクション
     * @return コレクションの型から推測させる妥当な反復子
     */
    public static Iterator collectionIterator(Object collection) {
        final Iterator i;
        if (HEval.isArray(collection)) {
            i = arrayIterator(collection);
        } else if (collection instanceof Collection) {
            i = ((Collection) collection).iterator();
        } else if (collection instanceof Iterator) {
            i = (Iterator) collection;
        } else if (collection instanceof Map) {
            i = ((Map) collection).entrySet().iterator();
        } else if (collection instanceof CharSequence) {
            i = asIterator((CharSequence) collection);
        } else if (collection instanceof Enumeration) {
            i = asIterator((Enumeration) collection);
        } else {
            throw new IllegalArgumentException("is not collection type. " + collection);
        }
        return i;
    }

    /**
     * 指定された要素数を可能ならばスキップします。
     * 
     * @param i
     *            スキップする反復子
     * @param skipCount
     *            可能ならばスキップする要素数
     * @return 引数の反復子の参照
     */
    public static Iterator shift(Iterator i, int skipCount) {
        for (int count = 0; skipCount > count && i.hasNext(); count++) {
            i.next();
        }
        return i;
    }

}
