/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.collections;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import shohaku.core.functor.FFactory;

/**
 * オブジェクトをメモリ上にキャッシュする為に有用な機能を追加した、マップ構造のインタフェースを定義します。<br>
 * また、このインターフェースは独立したデータ型ではなく、Map インターフェースを基にするアダプタとして定義されています。<br>
 * <br>
 * マップに対して、サイズの上限指定やエントリの順序、参照モデルの応用等の付加機能を提供するメモリベースの簡易キャッシュです。 <br>
 * <br>
 * このオブジェクトはスレッドセーフである必要があります。<br>
 * また複数のオペレーションをスレッドセーフに制御する為に getMutex() から適切な同期オブジェクトを返す必要があります。
 */
public interface Cache {

    /**
     * キャッシュをスレッドセーフに制御する同期を取る為の参照を返却します。<br>
     * 
     * <pre>
     *    Cache c = new Cache();
     *
     *    synchronized(c.getMutex()) {  // Synchronizing mutex object!
     *        if(c.containsKey("name")){
     *            c.put("name", "Alexander");
     *        }
     *        ...
     *    }
     * </pre>
     * 
     * @return キャッシュをスレッドセーフに制御する同期オブジェクト
     */
    Object getMutex();

    /**
     * 現在のキャッシュサイズを返します。
     * 
     * @return 現在のキャッシュサイズ
     */
    int size();

    /**
     * キャッシュをクリアします。
     */
    void clear();

    /**
     * 最大キャッシュサイズを返却します。<br>
     * maxSize が負数の場合は上限無しを示します。
     * 
     * @return 最大キャッシュサイズ。
     */
    int getMaxSize();

    /**
     * 最大キャッシュサイズを設定します。<br>
     * maxSize が上限無し（負数）以外で (maxSize < size()) の場合はリサイズを行います。
     * 
     * @param maxSize
     *            最大キャッシュサイズ
     */
    void setMaxSize(int maxSize);

    /**
     * キャッシュサイズが上限に達しているか検証します。
     * 
     * @return キャッシュサイズが上限に達している場合は true
     */
    boolean isLimit();

    /**
     * 指定のキャッシュ数にリサイズします。<br>
     * 削除されるエントリの規準は実装に依存します。<br>
     * 実際には (newSize < size()) の場合のみリサイズされます。<br>
     * newSize が負数の場合は IllegalArgumentException を発生させます。
     * 
     * @param newSize
     * @return 実際にリサイズされた場合は true
     * @throws IllegalArgumentException
     *             size が負数の場合
     */
    boolean resize(int newSize);

    /**
     * キャッシュが空の場合に true を返却します。
     * 
     * @return キャッシュが空の場合に true
     */
    boolean isEmpty();

    /**
     * 指定のキーが存在する場合は true を返却します。
     * 
     * @param key
     *            検証するキー
     * @return 指定のキーが存在する場合は true
     */
    boolean containsKey(Object key);

    /**
     * 指定のキャッシュデータが存在する場合は true を返却します。
     * 
     * @param value
     *            検証するキャッシュデータ
     * @return 指定のキャッシュデータが存在する場合は true
     */
    boolean containsValue(Object value);

    /**
     * 指定のキーに対応するキャッシュデータを返却します。<br>
     * キーが存在しない場合に生成ファンクタが登録されているならば生成ファンクタから生成して返却し、以外は null を返します。
     * 
     * @param key
     *            キー
     * @return 指定のキーに対応するキャッシュデータ
     */
    Object get(Object key);

    /**
     * 登録されている、生成ファンクタを返却します。<br>
     * 登録されていない場合は null を返します
     * 
     * @return 生成ファンクタ
     */
    FFactory getFactory();

    /**
     * 指定のキーが存在しない場合にデータを生成する、生成ファンクタを登録します。<br>
     * 生成ファンクタから生成されたデータは引数のキーでキャッシュされます。<br>
     * FFactory.create(Object o) の引数にはキーが渡されます。
     * 
     * @param factory
     *            生成ファンクタ
     */
    void setFactory(FFactory factory);

    /**
     * 指定のキーに対応するキャッシュエントリを削除してキャッシュデータを返却します。<br>
     * キーが存在しない場合は null を返します。
     * 
     * @param key
     *            キー
     * @return 削除されたキャッシュデータ、キーが存在しない場合は null
     */
    Object remove(Object key);

    /**
     * 指定のキーとキャッシュデータを格納します。<br>
     * エントリが追加されると同時に (getMaxSize() < (size() + 1)) の場合はリサイズされます。<br>
     * 既にキーが存在する場合はそのキャッシュデータを上書きします。<br>
     * 上書きされた場合は元のキャッシュデータを返却します、以外は null を返却します。
     * 
     * @param key
     *            キー
     * @param value
     *            キャッシュデータ
     * @return 上書きされた場合は元のキャッシュデータ、以外は null
     */
    Object put(Object key, Object value);

    /**
     * 指定されたキャッシュのエントリを全て格納します。<br>
     * エントリが追加されると同時に (getMaxSize() < (size() + 1)) の場合はリサイズされます。
     * 
     * @param t
     *            キャッシュ
     */
    void putAll(Cache t);

    /**
     * 指定されたマップのエントリをキャッシュエントリとして全て格納します。<br>
     * エントリが追加されると同時に (getMaxSize() < (size() + 1)) の場合はリサイズされます。
     * 
     * @param t
     *            マップ
     */
    void putAll(Map t);

    /**
     * キャッシュエントリのセットビューを返却します、要素は Map.Entry 型です。<br>
     * このビューのオペレーションは getMutex() のオブジェクトにより同期されます。<br>
     * <br>
     * セットはキャッシュと連動しているので、キャッシュに対する変更はセットに反映され、また、セットに対する変更はキャッシュに反映されます。 <br>
     * セットに対する反復の処理中にキャッシュが変更された場合は、反復の結果は保証されません。 <br>
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って キャッシュから対応するマッピングを削除する要素削除処理をサポートします。 <br>
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * 
     * @return キャッシュデータのコレクションビュー
     */
    Set entrySet();

    /**
     * キャッシュデータのコレクションビューを返却します。<br>
     * このビューのオペレーションは getMutex() のオブジェクトにより同期されます。<br>
     * <br>
     * コレクションはキャッシュと連動しているので、キャッシュに対する変更はコレクションに反映され、また、コレクションに対する変更はキャッシュに反映されます。 <br>
     * コレクションに対する反復の処理中にキャッシュが変更された場合は、反復の結果は保証されません。 <br>
     * コレクションは、Iterator.remove、Collection.remove、removeAll、retainAll、および clear の各オペレーションを使って キャッシュから対応するマッピングを削除する要素削除処理をサポートします。 <br>
     * add オペレーションと addAll オペレーションは、コレクションではサポートされていません。
     * 
     * @return キャッシュデータのコレクションビュー
     */
    Collection values();

    /**
     * キーのセットビューを返却します。<br>
     * このビューのオペレーションは getMutex() のオブジェクトにより同期されます。<br>
     * <br>
     * セットはキャッシュと連動しているので、キャッシュに対する変更はセットに反映され、また、セットに対する変更はキャッシュに反映されます。 <br>
     * セットに対する反復の処理中にキャッシュが変更された場合は、反復の結果は保証されません。 <br>
     * セットは、Iterator.remove、Set.remove、removeAll、retainAll、および clear の各オペレーションを使って キャッシュから対応するマッピングを削除する要素削除処理をサポートします。 <br>
     * add オペレーションと addAll オペレーションは、セットではサポートされていません。
     * 
     * @return キーのセットビュー
     */
    Set keySet();

}