/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans.dynamic;

import shohaku.core.helpers.HBeans;
import shohaku.core.lang.IntrospectionBeansException;

/**
 * コンストラクタまたはファクトリメソッドの情報を格納し、実行時に指定されたクラスを元にオブジェクトを生成する機能を提供します。 <br>
 * <br>
 * この機能はコンストラクタとファクトリメソッドの情報を同等に扱います。 <br>
 * その為、生成基のクラスと生成されるオブジェクトは同一とは限りません。
 */
public class FactoryMethodDesc extends FeatureDesc {

    /* メソッド名。 */
    private String methodName;

    /* 引数の情報のリスト。 */
    private ArgumentsDesc arguments;

    /**
     * デフォルトコンストラクタ。
     */
    public FactoryMethodDesc() {
        // no op
    }

    /**
     * 引数のプロパティを格納して初期化します。<br>
     * メソッド名の無いファクトリ（コンストラクタ）として認識します。
     * 
     * @param params
     *            引数情報のリスト
     */
    public FactoryMethodDesc(ArgumentsDesc params) {
        setArguments(params);
    }

    /**
     * 引数のプロパティを格納して初期化します。<br>
     * ファクトリメソッドとして認識します。
     * 
     * @param methodName
     *            メソッド名
     * @param params
     *            引数情報のリスト
     */
    public FactoryMethodDesc(String methodName, ArgumentsDesc params) {
        setMethodName(methodName);
        setArguments(params);
    }

    /**
     * 指定されたクラスからメソッドを呼び出し結果を返却します。
     * 
     * @param clazz
     *            処理対象のクラス
     * @return メソッドの戻り値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object invoke(Class clazz) throws IntrospectionBeansException {
        return invoke(clazz, null);
    }

    /**
     * 指定されたインスタンスからメソッドを呼び出し結果を返却します。
     * 
     * @param obj
     *            処理対象のインスタンス
     * @return メソッドの戻り値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object invoke(Object obj) throws IntrospectionBeansException {
        if (obj == null) {
            throw new NullPointerException("argument is null.");
        }
        return invoke(obj.getClass(), obj);
    }

    /* 指定されたインスタンスとクラスからメソッドまたはコンストラクタを呼び出し結果を返却します。 */
    Object invoke(Class clazz, Object obj) throws IntrospectionBeansException {
        if (clazz == null) {
            throw new NullPointerException("argument is null.");
        }
        if (isConstructor()) {
            return invokeNewInstance(clazz);
        } else {
            return invokeMethod(clazz, obj);
        }
    }

    /* 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。 */
    Object invokeMethod(Class clazz, Object obj) throws IntrospectionBeansException {
        return HBeans.invokeMethod(obj, HBeans.getAssignmentMethod(clazz, getMethodName(), getArgumentTypes()), getArgumentValues());
    }

    /* 指定されたクラスからコンストラクタを呼び出しインスタンスを生成して返却します。 */
    Object invokeNewInstance(Class clazz) throws IntrospectionBeansException {
        return HBeans.newInstance(clazz, getArgumentTypes(), getArgumentValues());
    }

    /*
     * Property
     */

    /**
     * コンストラクタの場合は true を返す。
     * 
     * @return コンストラクタの場合は true
     */
    public boolean isConstructor() {
        return (getMethodName() == null || getMethodName().length() == 0);
    }

    /**
     * メソッド名を返却します。
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * メソッド名を格納します。
     * 
     * @param name
     *            メソッド名
     */
    public void setMethodName(String name) {
        this.methodName = name;
    }

    /**
     * 引数情報のリストを返却します。
     * 
     * @return 引数情報のリスト
     */
    public ArgumentsDesc getArguments() {
        return arguments;
    }

    /**
     * 引数情報のリストを格納します。
     * 
     * @param params
     *            引数情報のリスト
     */
    public void setArguments(ArgumentsDesc params) {
        this.arguments = params;
    }

    /* 引数の値を返却します。 */
    Object[] getArgumentValues() {
        return (arguments == null) ? new Object[0] : arguments.getArgumentValues();
    }

    /* 引数の型情報を返却します。 */
    Class[] getArgumentTypes() {
        return (arguments == null) ? new Class[0] : arguments.getArgumentTypes();
    }
}
