/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.core.beans.dynamic;

import shohaku.core.helpers.HBeans;
import shohaku.core.lang.IntrospectionBeansException;

/**
 * クラスに関する情報を格納し、実行時に指定されたクラスを生成、制御する機能を提供します。
 */
public class ClassDesc extends FeatureDesc {

    /* フィールド情報。 */
    private FieldDesc[] fields = new FieldDesc[0];

    /* 初期化メソッド情報。 */
    private MethodDesc[] initMethods = new MethodDesc[0];

    /* メソッド情報。 */
    private MethodDesc[] methods = new MethodDesc[0];

    /* 破棄メソッド情報。 */
    private MethodDesc[] destroyMethods = new MethodDesc[0];

    /* 生成メソッド情報。 */
    private FactoryMethodDesc factory;

    /**
     * デフォルトコンストラクタ。
     */
    public ClassDesc() {
        // no op
    }

    /**
     * 指定されたクラスから、インスタンスを生成して返却します。
     * 
     * @param clazz
     *            生成元のクラス
     * @return 生成されたインスタンス
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object createInstance(Class clazz) throws IntrospectionBeansException {
        if (clazz == null) {
            throw new NullPointerException("argument is null.");
        }

        // new instance
        Object o = null;
        if (null != getFactory()) {
            o = getFactory().invoke(clazz);
        } else {
            o = HBeans.newInstance(clazz);
        }
        // invoke init methods
        for (int i = 0; i < initMethods.length; i++) {
            MethodDesc m = initMethods[i];
            m.invoke(o);
        }
        return o;
    }

    /**
     * 指定されたインスタンスに対して破棄処理を実行します。
     * 
     * @param o
     *            破棄するインスタンス
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public void destroyInstance(Object o) throws IntrospectionBeansException {
        if (o == null) {
            throw new NullPointerException("argument is null.");
        }
        // invoke init methods
        for (int i = 0; i < destroyMethods.length; i++) {
            MethodDesc m = destroyMethods[i];
            m.invoke(o);
        }
    }

    /*
     * Property
     */

    /**
     * 生成メソッド情報を返却します。
     * 
     * @return 生成メソッド情報
     */
    public FactoryMethodDesc getFactory() {
        return factory;
    }

    /**
     * 生成メソッド情報を格納します。
     * 
     * @param factory
     *            生成メソッド情報
     */
    public void setFactory(FactoryMethodDesc factory) {
        this.factory = factory;
    }

    /**
     * フィールド情報の配列を返却します。
     * 
     * @return フィールド情報の配列
     */
    public FieldDesc[] getFields() {
        return (FieldDesc[]) fields.clone();
    }

    /**
     * フィールド情報の配列を返却します。
     * 
     * @param fs
     *            フィールド情報の配列
     */
    public void setFields(FieldDesc[] fs) {
        if (fs == null) {
            throw new NullPointerException("argument is null.");
        }
        this.fields = fs;
    }

    /**
     * 初期化メソッド情報を返却します。
     * 
     * @return 初期化メソッド情報
     */
    public MethodDesc[] getInitMethods() {
        return (MethodDesc[]) initMethods.clone();
    }

    /**
     * メソッド情報の配列を返却します。
     * 
     * @param ms
     *            メソッド情報の配列
     */
    public void setInitMethods(MethodDesc[] ms) {
        if (ms == null) {
            throw new NullPointerException("argument is null.");
        }
        this.initMethods = ms;
    }

    /**
     * メソッド情報の配列を返却します。
     * 
     * @return メソッド情報の配列
     */
    public MethodDesc[] getMethods() {
        return (MethodDesc[]) methods.clone();
    }

    /**
     * メソッド情報の配列を返却します。
     * 
     * @param ms
     *            メソッド情報の配列
     */
    public void setMethods(MethodDesc[] ms) {
        if (ms == null) {
            throw new NullPointerException("argument is null.");
        }
        this.methods = ms;
    }

    /**
     * メソッド情報の配列を返却します。
     * 
     * @return メソッド情報の配列
     */
    public MethodDesc[] getDestroyMethods() {
        return (MethodDesc[]) destroyMethods.clone();
    }

    /**
     * メソッド情報の配列を返却します。
     * 
     * @param ms
     *            メソッド情報の配列
     */
    public void setDestroyMethods(MethodDesc[] ms) {
        if (ms == null) {
            throw new NullPointerException("argument is null.");
        }
        this.destroyMethods = ms;
    }

}
