/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans.binder;

import java.lang.reflect.Method;
import java.util.Map;

import shohaku.core.helpers.HBeans;
import shohaku.core.helpers.HEval;
import shohaku.core.helpers.HLog;
import shohaku.core.lang.IntrospectionBeansException;

/**
 * メソッドを更新規則と名前で拘束された引数と関連づける機能を提供します。
 */
public class BindMethod {

    /* 実行基のオブジェクト型。 */
    private Class objectType;

    /* 実行基のインスタンス。 */
    private Object srcObject;

    /* メソッド。 */
    private Method method;

    /* メソッド名。 */
    private String methodName;

    /* 引数の拘束情報。 */
    private BindArgumentsDesc bindArgumentsDesc;

    /* パラメータ型。 */
    private Class[] parameterTypes;

    /* パラメータ値。 */
    private Object[] parameterValues;

    /* パラメータの拘束名。 */
    private String[] parameterNames;

    /* パラメータの拘束タイプ。 */
    private BindArgumentRule[] parameterRules;

    /* toString()をキャッシュする。 */
    private String _toString;

    /*
     * デフォルトコンストラクタ。
     */
    private BindMethod(Class objectType, Object srcObject, String methodName, BindArgumentsDesc args) throws IntrospectionBeansException {
        this.methodName = methodName;
        this.objectType = objectType;
        this.srcObject = srcObject;
        this.bindArgumentsDesc = args;
        this.parameterTypes = args.getArgumentTypes();
        this.parameterRules = args.getArgumentRules();
        this.parameterNames = args.getArgumentNames();
        this.parameterValues = args.getArgumentValues();
        init();
    }

    /**
     * メソッドを呼び出し結果を返却します。
     * 
     * @param values
     *            名前で拘束されるメソッドの引数値
     * @return メソッドの戻り値
     * @throws IntrospectionBeansException
     *             処理の呼出に失敗した場合
     */
    public Object invoke(Map values) throws IntrospectionBeansException {
        return invokeMethod(getSrcObject(), getMethod(), values);
    }

    /**
     * 実行基のオブジェクト型を返却します。
     * 
     * @return 実行基のオブジェクト型
     */
    public Class getObjectType() {
        return objectType;
    }

    /**
     * メソッド名を返却します。
     * 
     * @return メソッド名
     */
    public String getMethodName() {
        return methodName;
    }

    /**
     * 実行基のオブジェクトを返却します．
     * 
     * @return 実行基のオブジェクト
     */
    public Object getSrcObject() {
        return srcObject;
    }

    /**
     * メソッドを返却します．
     * 
     * @return メソッド
     */
    public Method getMethod() {
        return method;
    }

    /**
     * メソッドの引数情報を返却します。
     * 
     * @return 引数情報
     */
    public BindArgumentsDesc getBindArgumentsDesc() {
        return bindArgumentsDesc;
    }

    /**
     * このオブジェクトの文字列表現を返却します。
     * 
     * @return このオブジェクトの文字列表現
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return _toString;
    }

    /*
     * private
     */

    /* メソッドの拘束を実行する。 */
    private void init() throws IntrospectionBeansException {
        Class[] types = parameterTypes;
        Object[] values = parameterValues;
        String[] names = parameterNames;
        BindArgumentRule[] rules = parameterRules;
        if (!HEval.isEqualArraySize(new Object[] { types, values, names, rules })) {
            throw new IntrospectionBeansException("parameter is illegal.");
        }
        if (HEval.isOrEmpty(names)) {
            throw new IntrospectionBeansException("parameter is illegal.");
        }
        try {
            initMethod(getMatchingAccessibleMethod());
        } catch (NoSuchMethodException e) {
            throw new IntrospectionBeansException("parameter is illegal.", e);
        }
        // toString()を構築
        buildToString();
    }

    /* メソッドを格納します． */
    private void initMethod(Method initMethod) {
        this.method = initMethod;
    }

    /* 指定されたインスタンスとクラスからメソッドを呼び出し結果を返却します。 */
    private Object invokeMethod(Object obj, Method invokeMethod, Map params) throws IntrospectionBeansException {

        Object[] values = (Object[]) parameterValues.clone();
        String[] names = parameterNames;
        BindArgumentRule[] rules = parameterRules;
        for (int i = 0; i < values.length; i++) {
            BindArgumentRule rule = rules[i];
            String name = names[i];

            // final
            if (BindArgumentRule.FINAL == rule) {
                if (params.containsKey(name)) {
                    throw new IntrospectionBeansException("illegal parameter name is final. " + name);
                }
            } else // overwrite
            if (BindArgumentRule.OVERWRITE == rule) {
                if (params.containsKey(name)) {
                    values[i] = params.get(name);
                }
            } else // required
            if (BindArgumentRule.REQUIRED == rule) {
                if (params.containsKey(name)) {
                    values[i] = params.get(name);
                } else {
                    throw new IntrospectionBeansException("required parameter isn't included. " + name);
                }
            }

        }
        return HBeans.invokeMethod(obj, invokeMethod, values);
    }

    /* 指定されたクラスから処理の対象となるメソッドオブジェクトを返却します。 */
    private Method getMatchingAccessibleMethod() throws NoSuchMethodException, IntrospectionBeansException {
        Method m = HBeans.getAssignmentMethod(getObjectType(), getMethodName(), parameterTypes);
        if (m == null) {
            throw new NoSuchMethodException(HLog.log("not find Method. ", getObjectType(), getMethodName(), parameterTypes));
        }
        return m;
    }

    /* toString()を構築する */
    private void buildToString() {
        Class[] types = parameterTypes;
        Object[] values = parameterValues;
        String[] names = parameterNames;
        BindArgumentRule[] rules = parameterRules;
        StringBuffer sb = new StringBuffer();
        sb.append("bind method=");
        sb.append(getMethod());
        sb.append('{');
        for (int i = 0; i < types.length; i++) {
            sb.append('{');
            sb.append(rules[i]);
            sb.append(',');
            sb.append(names[i]);
            sb.append(',');
            sb.append(values[i]);
            sb.append('}');
        }
        sb.append('}');
        this._toString = sb.toString();
    }

    /*
     * static
     */

    /**
     * メソッドの拘束を実行し生成されたオブジェクトを返却します。
     * 
     * @param objectType
     *            実行基のオブジェクト型
     * @param srcObject
     *            実行基のインスタンス（静的なメソッドの場合は null を指定する）
     * @param methodName
     *            メソッド名
     * @param args
     *            引数の拘束情報
     * @return 拘束されたメソッド情報
     * @throws IntrospectionBeansException
     *             メソッド生成または拘束に失敗した場合
     */
    public static BindMethod bind(Class objectType, Object srcObject, String methodName, BindArgumentsDesc args) throws IntrospectionBeansException {
        return new BindMethod(objectType, srcObject, methodName, args);
    }

}
