/*
 * shohaku
 * Copyright (C) 2006  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.core.beans;

import java.util.Iterator;
import java.util.Map;

import shohaku.core.functor.FTransformer;
import shohaku.core.functor.FunctorException;
import shohaku.core.lang.IllegalIdentifierException;

/**
 * 型変換を行う機能をマッピングする仕組みを提供します。
 */
public interface TypeTransformer {

    /**
     * 格納する変換ファンクタのエントリ数を返却します。
     * 
     * @return 格納するエントリ数
     */
    int size();

    /**
     * クラス型の要素を持つ反復子を生成して返却します。
     * 
     * @return クラスの反復子
     */
    Iterator getTypes();

    /**
     * 指定のクラスに対応するエントリがある場合は true を返却します。
     * 
     * @param clazz
     *            検証するクラス
     * @return 指定のクラスに対応するエントリがある場合は true
     */
    boolean contains(Class clazz);

    /**
     * 指定のクラスに対応する変換ファンクタを返却します。<br>
     * 対応する変換ファンクタが発見出来ない場合は null が返ります。
     * 
     * @param clazz
     *            クラス
     * @return 指定のクラスに対応する変換ファンクタ
     */
    FTransformer get(Class clazz);

    /**
     * 指定のクラスから推測される変換ファンクタを返却します。<br>
     * このメソッドはクラスから推測的に対応クラスを探します。<br>
     * 推測される変換ファンクタが発見出来ない場合は null が返ります。
     * 
     * @param clazz
     *            クラス
     * @return 指定のクラスから推測される変換ファンクタ
     */
    FTransformer find(Class clazz);

    /**
     * 指定のクラスから推測される変換ファンクタから変換を実行して結果を返却します。<br>
     * 推測の規準は find(Class clazz) と同じ規準となります。<br>
     * 推測される変換ファンクタが発見出来ない場合は IllegalIdentifierException が返ります。<br>
     * このメソッドは transform(clazz, o, java.util.Collections.EMPTY_MAP) と同意です。
     * 
     * @param clazz
     *            クラス
     * @param o
     *            変換対象のオブジェクト
     * @return 変換ファンクタの処理結果
     * @throws IllegalIdentifierException
     *             対象の変換ファンクタが発見出来ない場合
     * @throws FunctorException
     *             予測外の問題から変換に失敗した場合
     */
    Object transform(Class clazz, Object o);

    /**
     * 指定のクラスから推測される変換ファンクタから変換を実行して結果を返却します。<br>
     * 推測の規準は find(Class clazz) と同じ規準となります。<br>
     * 推測される変換ファンクタが発見出来ない場合は IllegalIdentifierException が返ります。
     * 
     * @param clazz
     *            クラス
     * @param o
     *            変換対象のオブジェクト
     * @param args
     *            変換パラメータ
     * @return 変換ファンクタの処理結果
     * @throws IllegalIdentifierException
     *             対象の変換ファンクタが発見出来ない場合
     * @throws FunctorException
     *             予測外の問題から変換に失敗した場合
     */
    Object transform(Class clazz, Object o, Map args);

}