/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.nodes;

import java.util.Iterator;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import shohaku.core.collections.IteratorUtils;
import shohaku.core.helpers.Eval;
import shohaku.core.lang.LangUtils;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.util.cel.CELBinder;
import shohaku.core.util.cel.CELContext;
import shohaku.core.util.cel.CELFormatException;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentContext;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeContext;

/**
 * デフォルトノードを構成するためのユーティリティーメソッドを提供します。<br>
 * 依存を避けるため、このクラスはデフォルトノードから直接利用される事は有りません。
 */
public class NodeCompositeUtilities {

    /** 有効なBOOL型の文字列表記。 */
    public static final String[] BOOL_STRINGS = new String[] { "true", "false", "yes", "no", "on", "off", "1", "0" };

    /** 有効なBOOL型のTRUEを示す文字列表記。 */
    public static final String[] TRUE_STRINGS = new String[] { "true", "yes", "on", "1" };

    /**
     * 有効なBOOL型の文字列表記か検証します。
     * 
     * @param value
     *            検証する文字列
     * @return 有効なBOOL型の文字列表記の場合のみ <code>true</code>
     */
    public static boolean isBoolean(String value) {
        return Eval.isContains(BOOL_STRINGS, value.toLowerCase());
    }

    /**
     * 文字列に対応するBOOL値を返却します。
     * 
     * @param value
     *            解析する文字列
     * @return BOOL型のTRUEを示す文字列表記の場合のみ <code>true</code>
     */
    public static boolean getBoolean(String value) {
        return Eval.isContains(TRUE_STRINGS, value.toLowerCase());
    }

    /**
     * テキスト情報を評価して返却します。 <br>
     * 全ての子のテキスト型ノードを一つの文字列に連結して返却します。
     * 
     * @param context
     *            ノードコンテキスト
     * @return テキスト情報
     */
    public static String getTextValueToString(NodeContext context) {
        StringBuffer sb = new StringBuffer();
        Iterator i = context.textIterator();
        while (i.hasNext()) {
            sb.append(((Node) i.next()).getNodeValue());
        }
        return sb.toString().trim();
    }

    /**
     * テキスト情報を評価して返却します。 <br>
     * 全ての子の値型ノードの文字列表現を一つの文字列に連結して返却します。
     * 
     * @param context
     *            ノードコンテキスト
     * @return テキスト情報
     */
    public static String getChildrenValueToString(NodeContext context) {
        StringBuffer sb = new StringBuffer();
        Iterator i = context.valueIterator();
        while (i.hasNext()) {
            sb.append(((Node) i.next()).getNodeValue());
        }
        return sb.toString().trim();
    }

    /**
     * 指定された値をCEL式として解釈し、その結果の値を返却します。
     * 
     * @param context
     *            ノードコンテキスト
     * @param value
     *            解析する文字列
     * @return CEL式の実行結果の値
     */
    public static Object toCELValue(NodeContext context, String value) {
        try {
            return CELBinder.getObjectCreationBinder().getValue(value, new CELContextImpl(context));
        } catch (CELFormatException e) {
            throw new GinkgoException("value:" + value, e);
        }
    }

    /*
     * class
     */

    private static class CELContextImpl implements CELContext {

        final NodeContext context;

        CELContextImpl(NodeContext context) {
            this.context = context;

        }

        public Iterator nameIterator() {
            Iterator[] iters = new Iterator[2];
            iters[0] = context.getDocument().getNodeValueIdIterator();
            iters[1] = context.getDocumentContext().getAttributeNameIterator();
            return IteratorUtils.compositeIterator(iters);
        }

        public Object getValue(String name) {
            return NodeCompositeUtilities.getReferenceValue(context, name);
        }

        public Object setValue(String name, Object value) {
            return context.getDocumentContext().setAttribute(name, value);
        }

        public boolean containsName(String name) {
            return NodeCompositeUtilities.isReferenceContainsName(context, name);
        }
    }

    /**
     * 指定された識別子に対応するノードの値を返却します。
     * 
     * @param context
     *            ノードコンテキスト
     * @param id
     *            識別子
     * @return 識別子が示すノードの値
     */
    public static Object getReferenceNodeValue(NodeContext context, String id) {
        return context.getDocument().getNodeValueById(id);
    }

    /**
     * 引数の名前を持つドキュメントの属性値を返却します。 <br>
     * 指定された属性が存在しない場合は<code>null</code>が返されます。
     * 
     * @param context
     *            ノードコンテキスト
     * @param name
     *            属性名
     * @return 属性値
     */
    public static Object getDocumentContextAttribute(NodeContext context, String name) {
        DocumentContext docContext = context.getDocumentContext();
        return docContext.getAttribute(name);
    }

    /**
     * 指定された識別子に対応するノードの値、又はコンテキスト属性を返却します。<br>
     * 識別子に対応するノードの値が存在しない場合にコンテキスト属性を検索し双方に存在しない場合は <code>null</code> を返却します。
     * 
     * @param context
     *            ノードコンテキスト
     * @param id
     *            識別子
     * @return 識別子が示すノードの値、又はコンテキスト属性
     */
    public static Object getReferenceValue(NodeContext context, String id) {
        Object o = null;
        if (!Eval.isBlankOrSpace(id)) {
            if (context.getDocument().containsId(id)) {
                o = getReferenceNodeValue(context, id);
            } else {
                o = getDocumentContextAttribute(context, id);
            }
        }
        return o;
    }

    /**
     * 指定された識別子に対応するノードの値、又はコンテキスト属性が存在するか検証します。
     * 
     * @param context
     *            ノードコンテキスト
     * @param id
     *            識別子
     * @return 識別子が示すノードの値、又はコンテキスト属性が存在する場合 <code>true</code>
     */
    public static boolean isReferenceContainsName(NodeContext context, String id) {
        boolean ret = false;
        if (!Eval.isBlankOrSpace(id)) {
            ret = context.getDocument().containsId(id);
            if (!ret) {
                ret = context.getDocumentContext().containsName(id);
            }
        }
        return ret;
    }

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param context
     *            ノードのコンテキスト情報
     * @param className
     *            クラスを示す文字列
     * @return ロードされたクラス
     */
    public static Class loadClass(NodeContext context, String className) {
        try {
            return LangUtils.loadClass(className, context.getClassLoader());
        } catch (NoSuchResourceException e) {
            throw new GinkgoException("Class couldn't be created. name:" + className, e);
        }
    }

    /* 拡張の数値書式の正規表現パターン。 */
    private static final Pattern numberPattern = Pattern
            .compile("^[-+]?(?:0|0X|0x)?[0-9-a-fA-F](?:[-+0-9-a-fA-F,]*[-+0-9-a-fA-F])?(?:\\.[-+0-9-a-fA-F]+)?$");

    /* 拡張の数値書式の拡張文字を削除するパターン */
    private static final Pattern numberRemovePattern = Pattern.compile("^\\+|,");

    /**
     * 数値表現の拡張書式をJavaの数値規約の書式に変換する。
     * 
     * @param value
     *            数値の文字シーケンス
     * @return Java書式の数値文字列
     * @throws NumberFormatException
     *             書式が数値表現として解析に失敗した場合
     */
    public static String toJavaNumberString(CharSequence value) {

        if (numberPattern.matcher(value).matches()) {
            return numberRemovePattern.matcher(value).replaceAll("");
        } else {
            throw new NumberFormatException("value:" + value);
        }
    }

    /* 拡張の数値書式の正規表現パターン。 */
    private static final Pattern referenceExPattern = Pattern.compile("([#$%]\\{(?:\\\\\\}|[^}])*\\})");

    /* 拡張の数値書式の拡張文字を削除するパターン */
    private static final Pattern referenceExRemovePattern = Pattern.compile("\\\\\\}");

    /**
     * 指定の文字列を参照構文として解析し変換し返却します。
     * 
     * @param document
     *            ドキュメント
     * @param s
     *            変換元の文字列
     * @return 変換後の文字列
     */
    public static String filterReferenceExpression(Document document, String s) {

        StringBuffer sb = new StringBuffer(s.length());
        Matcher m = referenceExPattern.matcher(s);

        int i = 0;
        while (i < s.length()) {
            if (m.find(i)) {
                sb.append(s.substring(i, m.start(1)));
                // get value
                String refex = referenceExRemovePattern.matcher(m.group(1)).replaceAll("}");
                char mark = refex.charAt(0);
                String id = refex.substring(2, refex.length() - 1);
                Object value = getReferenceExpressionValue(document, mark, id);
                sb.append(value);
                i = m.end(1);
            } else {
                sb.append(s.substring(i));
                break;
            }
        }
        return sb.toString();
    }

    private static Object getReferenceExpressionValue(Document document, char mark, String id) {
        Object o = null;
        if (mark == '$') {
            o = document.getContext().getAttribute(id);
        } else {
            o = document.getNodeValueById(id);
        }
        return o;
    }

}
