/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.nodes;

import shohaku.core.helpers.Eval;
import shohaku.ginkgo.AbstractNode;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.NodeContext;

/**
 * デフォルトノードの抽象実装を提供し実装の負担を最小化します。
 */
public abstract class AbstractDefaultNode extends AbstractNode {

    /* テキストをキャッシュする. */
    private String cacheText;

    /* 有効なノード種別のキャッシュ。 */
    private int[] types;

    /* ノードを構成するユーティリティ機能。 */
    private NodeCompositeFeature feature;

    /*
     * Node Type's
     */

    /**
     * 指定されたノードの種別に属する場合Trueを返却します。
     * 
     * @param type
     *            検証する種別
     * @return 指定されたノードの種別に属する場合True
     */
    public boolean isType(int type) {
        if (types == null) {
            types = getTypes();
        }
        return Eval.isContains(types, type);
    }

    /**
     * 有効なノード種別を返却します。
     * 
     * @return 有効なノード種別
     */
    abstract protected int[] getTypes();

    /*
     * Text
     */

    /**
     * テキストのキャッシュを返却します。
     * 
     * @return テキストのキャッシュ
     */
    protected String getCacheText() {
        return cacheText;
    }

    /**
     * テキストのキャッシュを格納します。
     * 
     * @param s
     *            テキストのキャッシュ
     */
    protected void setCacheText(String s) {
        this.cacheText = s;
    }

    /**
     * テキスト情報を評価して返却します。
     * 
     * @return テキスト情報
     */
    public String getText() {
        if (null == getCacheText()) {
            setCacheText(getFeature().getTextValue(this));
        }
        return getCacheText();
    }

    /*
     * NodeCompositeFeature
     */

    /**
     * ノードを構成するユーティリティ機能を返却します。 <br>
     * 戻り値は<code>NodeCompositeFeature</code>クラスまたはサブクラスのインスタンス。
     * 
     * @return ノードを構成するユーティリティ機能
     */
    protected NodeCompositeFeature getFeature() {
        return feature;
    }

    /* ノードを構成するユーティリティ機能を格納します。 */
    private void setFeature(NodeCompositeFeature feature) {
        this.feature = feature;
    }

    /*
     * Event
     */

    /**
     * 解析処理を開始する直前に初期化の通知を受ける。 <br>
     * オーバライドした場合必ず親クラスの同メソッドを呼び出してください。
     * 
     * @param nodeContext
     *            ノードのコンテキスト情報
     * @throws GinkgoException
     *             構成情報例外。
     */
    public void initialize(NodeContext nodeContext) {
        super.initialize(nodeContext);
        // Feature
        NodeCompositeRule rule = getContext().getNodeCompositeRule();
        Object f = rule.getFeature(NodeCompositeFeature.class);
        setFeature((f != null) ? (NodeCompositeFeature) f : new NodeCompositeFeature());
    }

}
