/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.helpers;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import shohaku.core.collections.Group;
import shohaku.core.collections.group.HashListGroup;
import shohaku.core.collections.params.Parameters;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;
import shohaku.ginkgo.Ginkgo;
import shohaku.ginkgo.GinkgoException;
import shohaku.ginkgo.Node;
import shohaku.ginkgo.NodeAttributesRule;
import shohaku.ginkgo.NodeCompositeRule;
import shohaku.ginkgo.NodeCompositeRuleFactory;
import shohaku.ginkgo.NodeRule;
import shohaku.ginkgo.Substitutor;

/**
 * XML形式で定義されたデータから <code>DefaultNodeCompositeRule</code> （又はそのサブクラス）を生成する機能を提供します。
 */
public class XMLNodeCompositeRuleFactory implements NodeCompositeRuleFactory {

    /*
     * static initialize
     */

    /* 構成情報を読み取る構成ルール情報を生成を示す。 */
    private static final DefaultNodeCompositeRule ruleNodeCompositeRule;
    static {

        DefaultNodeCompositeRule rule = new DefaultNodeCompositeRule();

        List nodes = new ArrayList(48);

        // 処理頻度の高いノード
        putNode(nodes, "/composite-rule/nodes/node", "SimpleNode");
        putNode(nodes, "/composite-rule/tags/tag", "SimpleNode");

        // 処理頻度の中程度のノード
        // composite-rule/rules
        putNode(nodes, "/composite-rule/rules/xml-composite-rule", "nodes.core.beans.ObjectNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/property", "nodes.core.beans.SetPropertyInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/method", "nodes.core.beans.MethodInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/method/arg", "nodes.core.beans.ArgumentInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/init", "nodes.core.beans.FactoryMethodInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/init/arg", "nodes.core.beans.ArgumentInfoNode");
        // composite-rule/node/substitutor
        putNode(nodes, "/composite-rule/nodes/node/substitutor", "nodes.core.beans.ObjectNode");
        // composite-rule/node/attributes
        putNode(nodes, "/composite-rule/nodes/node/attributes", "SimpleNode");
        putNode(nodes, "/composite-rule/nodes/node/attributes/attribute", "SimpleNode");
        // composite-rule/node/x-attributes
        putNode(nodes, "/composite-rule/nodes/node/x-attributes", "SimpleNode");
        putNode(nodes, "/composite-rule/nodes/node/x-attributes/x-attribute", "SimpleNode");
        // 値ノード
        // base
        putNode(nodes, "/desc", "nodes.core.base.CommentNode");
        putNode(nodes, "/text", "nodes.core.base.CharDataNode");
        putNode(nodes, "/string", "nodes.core.base.StringNode");
        putNode(nodes, "/string/append", "nodes.core.base.StringNode");
        putNode(nodes, "/char", "nodes.core.base.CharacterNode");
        putNode(nodes, "/byte", "nodes.core.base.ByteNode");
        putNode(nodes, "/short", "nodes.core.base.ShortNode");
        putNode(nodes, "/int", "nodes.core.base.IntgerNode");
        putNode(nodes, "/long", "nodes.core.base.LongNode");
        putNode(nodes, "/double", "nodes.core.base.DoubleNode");
        putNode(nodes, "/float", "nodes.core.base.FloatNode");
        putNode(nodes, "/boolean", "nodes.core.base.BooleanNode");
        putNode(nodes, "/type", "nodes.core.base.ClassNode");
        putNode(nodes, "/ref", "nodes.core.base.ReferenceNode");
        // math
        putNode(nodes, "bigDecimal", "nodes.core.math.BigDecimalNode");
        putNode(nodes, "bigInteger", "nodes.core.math.BigIntegerNode");
        // util
        putNode(nodes, "/date", "nodes.core.util.DateNode");
        putNode(nodes, "/uri", "nodes.core.util.URINode");
        putNode(nodes, "/regex", "nodes.core.util.RegexNode");
        // collection
        putNode(nodes, "/array", "nodes.core.collection.ArrayNode");
        putNode(nodes, "/list", "nodes.core.collection.ListNode");
        putNode(nodes, "/set", "nodes.core.collection.SetNode");
        putNode(nodes, "/map", "nodes.core.collection.MapNode");
        putNode(nodes, "/map/entry", "nodes.core.collection.SingletonMapNode");
        putNode(nodes, "/group", "nodes.core.collection.GroupNode");
        putNode(nodes, "/group/entry", "nodes.core.collection.SingletonGroupNode");
        putNode(nodes, "/parameters", "nodes.core.collection.ParametersNode");
        putNode(nodes, "/parameters/param", "nodes.core.collection.NamedSingletonMapNode");
        // bean
        putNode(nodes, "/object", "nodes.core.beans.ObjectNode");
        putNode(nodes, "/object/property", "nodes.core.beans.SetPropertyInfoNode");
        putNode(nodes, "/object/method", "nodes.core.beans.MethodInfoNode");
        putNode(nodes, "/object/method/arg", "nodes.core.beans.ArgumentInfoNode");
        putNode(nodes, "/object/init", "nodes.core.beans.FactoryMethodInfoNode");
        putNode(nodes, "/object/init/arg", "nodes.core.beans.ArgumentInfoNode");
        putNode(nodes, "/object/factory", "nodes.core.beans.FactoryMethodInfoNode");
        putNode(nodes, "/object/factory/arg", "nodes.core.beans.ArgumentInfoNode");
        putNode(nodes, "/call", "nodes.core.beans.CallMethodNode");
        putNode(nodes, "/call/arg", "nodes.core.beans.ArgumentInfoNode");

        // 処理が一度のみのノード
        putNode(nodes, "/composite-rule", "SimpleNode");
        putNode(nodes, "/composite-rule/rules", "SimpleNode");
        putNode(nodes, "/composite-rule/nodes", "SimpleNode");
        putNode(nodes, "/composite-rule/tags", "SimpleNode");

        Object[] nodeRuleSet = new Object[nodes.size() * 2];
        int j = 0;
        for (Iterator i = nodes.iterator(); i.hasNext();) {
            NodeRule nodeRule = new NodeRule();

            Object[] entry = (Object[]) i.next();
            String pattern = (String) entry[0];
            String className = (String) entry[1];

            nodeRule.setPattern(pattern);
            nodeRule.setNodeClass(className);

            nodeRuleSet[j++] = pattern;
            nodeRuleSet[j++] = nodeRule;
        }
        rule.addNodeRuleParameters(null, new Parameters(nodeRuleSet));

        ruleNodeCompositeRule = rule;
    }

    private static void putNode(List nodes, String id, String className) {
        Object[] e = new Object[2];
        e[0] = id;
        e[1] = "shohaku.ginkgo." + className;
        nodes.add(e);
    }

    /*
     * implement NodeCompositeRuleLoader
     */

    /* 読み取りに使用するクラスリーダ。 */
    private ClassLoader classLoader;

    /**
     * 定義ファイルへのリソースパスから<code>NodeCompositeRule</code>を生成して返却します。
     * 
     * @param id
     *            定義ファイルへのリソースパス
     * @return <code>NodeCompositeRule</code>の実装クラスのインスタンス
     * @see shohaku.ginkgo.NodeCompositeRuleFactory#create(java.lang.String)
     */
    public NodeCompositeRule create(String id) {
        if (classLoader == null) {
            return createImpl(id, XMLNodeCompositeRuleFactory.class.getClassLoader());
        } else {
            return createImpl(id, classLoader);
        }
    }

    /**
     * 引数の入力ストリームから<code>NodeCompositeRule</code>を生成して返却します。
     * 
     * @param inStream
     *            入力ストリーム
     * @return <code>NodeCompositeRule</code>の実装クラスのインスタンス
     */
    public NodeCompositeRule create(InputStream inStream) {
        if (classLoader == null) {
            return createImpl(inStream, XMLNodeCompositeRuleFactory.class.getClassLoader());
        } else {
            return createImpl(inStream, classLoader);
        }
    }

    /**
     * 読み取りに使用するクラスリーダを返却します。
     * 
     * @return クラスリーダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * 読み取りに使用するクラスリーダを格納します。
     * 
     * @param loader
     *            クラスリーダ
     */
    public void setClassLoader(ClassLoader loader) {
        classLoader = loader;
    }

    /*
     * private static
     */

    /* 引数のシステムIDから DefaultNodeCompositeRule を生成して返却します。 */
    private static DefaultNodeCompositeRule createImpl(String systemId, ClassLoader classLoader) {

        InputStream inStream = null;
        try {
            inStream = ResourceLoader.getResourceAsStream(systemId, classLoader);
        } catch (NoSuchResourceException e) {
            throw new GinkgoException("NodeCompositeRule couldn't be created.", e);
        }
        if (inStream == null) {
            throw new GinkgoException("NodeCompositeRule couldn't be created.");
        }
        return createImpl(inStream, classLoader);

    }

    /* 引数の入力ストリームから DefaultNodeCompositeRule を生成して返却します。 */
    private static DefaultNodeCompositeRule createImpl(InputStream inStream, ClassLoader classLoader) {
        try {

            Node root = load(inStream, ruleNodeCompositeRule, classLoader);
            return init(root);

        } catch (Exception e) {
            throw new GinkgoException("NodeCompositeRule couldn't be created.", e);
        }
    }

    /* 入力ストリームからノードリストを読み込む。 */
    private static Node load(InputStream inStream, DefaultNodeCompositeRule rule, ClassLoader classLoader) {

        // XML構成ルールファイルの解析
        Ginkgo ginkgo = new Ginkgo();
        if (classLoader != null) {
            ginkgo.setClassLoader(classLoader);
        }
        // XMLの解析
        ginkgo.parse(rule, inStream);

        // ノードを保管する
        Node root = ginkgo.getDocument().getContext().getRoot();

        return root;
    }

    /* ノードを初期化および登録します。 */
    private static DefaultNodeCompositeRule init(Node root) {

        DefaultNodeCompositeRule rule = null;
        HashMap nodeMap = new HashMap();
        HashListGroup tagGroup = new HashListGroup();

        for (Iterator i = root.getContext().elementIterator(); i.hasNext();) {
            Node n = (Node) i.next();
            String nm = n.getContext().getNodeName();
            if ("rules".equals(nm)) {
                rule = getDefaultNodeCompositeRule(n);
            } else if ("nodes".equals(nm)) {
                nodeTag(n, nodeMap);
            } else if ("tags".equals(nm)) {
                tagTag(n, tagGroup);
            }
        }
        if (rule == null) {
            rule = new DefaultNodeCompositeRule();
        }
        for (Iterator i = tagGroup.iterator(); i.hasNext();) {
            Group.Entry e = (Group.Entry) i.next();
            String nsuri = (String) e.getKey();
            List tags = (List) e.getValues();

            Object[] nodeRuleSet = new Object[tags.size() * 2];
            int k = 0;
            for (Iterator j = tags.iterator(); j.hasNext();) {
                String[] tag = (String[]) j.next();
                NodeRule nodeRule = new NodeRule();
                String pattern = tag[0];
                String nodeId = tag[1];

                Object[] nodeAtts = (Object[]) nodeMap.get(nodeId);
                if (null == nodeAtts) {
                    throw new NullPointerException("node doesn't exist. nodeId:" + nodeId);
                }
                String className = (String) nodeAtts[0];
                Parameters rules = (Parameters) nodeAtts[1];
                Parameters extend = (Parameters) nodeAtts[2];
                Substitutor substitutor = (Substitutor) nodeAtts[3];

                nodeRule.setPattern(pattern);
                nodeRule.setNodeClass(className);
                nodeRule.setNodeAttributesRules(rules);
                nodeRule.setXAttributes(extend);
                nodeRule.setSubstitutor(substitutor);

                nodeRuleSet[k++] = pattern;
                nodeRuleSet[k++] = nodeRule;
            }
            rule.addNodeRuleParameters(nsuri, new Parameters(nodeRuleSet));
        }
        return rule;
    }

    /* DefaultNodeCompositeRule を生成して返却します。 */
    private static DefaultNodeCompositeRule getDefaultNodeCompositeRule(Node rulesTag) {
        Iterator i = rulesTag.getContext().elementIterator("xml-composite-rule");
        if (i.hasNext()) {
            Node n = (Node) i.next();
            return (DefaultNodeCompositeRule) n.getNodeValue();
        }
        return new DefaultNodeCompositeRule();
    }

    /* NODESタグの情報を格納します。 */
    private static void nodeTag(Node nodesTag, HashMap nodeMap) {
        for (Iterator i = nodesTag.getContext().elementIterator("node"); i.hasNext();) {
            Node node = (Node) i.next();
            String nodeId = node.getAttribute("id");

            Object[] nodeAtts = new Object[4];
            nodeMap.put(nodeId, nodeAtts);
            nodeAtts[0] = node.getAttribute("class");
            for (Iterator j = node.getContext().elementIterator(); j.hasNext();) {
                Node n = (Node) j.next();
                String nm = n.getContext().getNodeName();
                if ("attributes".equals(nm)) {
                    setNodeAttributes(nodeAtts, n);
                } else if ("x-attributes".equals(nm)) {
                    setXNodeAttributes(nodeAtts, n);
                } else if ("substitutor".equals(nm)) {
                    nodeAtts[3] = n.getNodeValue();
                }
            }
        }
    }

    private static void setNodeAttributes(Object[] nodeAtts, Node n) {

        HashMap m = new HashMap();
        for (Iterator i = n.getContext().elementIterator("attribute"); i.hasNext();) {
            Node att = (Node) i.next();
            String name = att.getAttribute("name");
            String alias = att.getAttribute("alias");
            String defaultValue = att.getAttribute("defaultValue");
            NodeAttributesRule rule = new NodeAttributesRule(name, alias, defaultValue);
            m.put(name, rule);
        }
        nodeAtts[1] = new Parameters(m);
    }

    private static void setXNodeAttributes(Object[] nodeAtts, Node n) {
        HashMap m = new HashMap();
        for (Iterator i = n.getContext().elementIterator("x-attribute"); i.hasNext();) {
            Node att = (Node) i.next();
            String name = att.getAttribute("name");
            String value = att.getAttribute("value");
            m.put(name, value);
        }
        nodeAtts[2] = new Parameters(m);

    }

    /* TAGSタグの情報を格納します。 */
    private static void tagTag(Node tagsTag, HashListGroup tagGroup) {
        String nsuri = tagsTag.getAttribute("nsuri");
        List tags = new ArrayList();
        for (Iterator i = tagsTag.getContext().elementIterator("tag"); i.hasNext();) {
            Node n = (Node) i.next();
            tags.add(new String[] { n.getAttribute("name"), n.getAttribute("node") });
        }
        tagGroup.addAll(((null == nsuri) ? "" : nsuri), tags);
    }
}
