/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.ginkgo.helpers;

import java.io.InputStream;
import java.util.Hashtable;

import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * <code>XMLNodeCompositeRuleFactory</code> から <code>NodeCompositeRule</code> を生成し保管する機能を提供します。 <br>
 * 一度読み込まれたリソースはキャッシュされるため２度目以降には読み取りは行われません。
 */
public class NodeCompositeRuleLoader {

    /* 構成ルールをキャシュします。 */
    private static final class CompositeRuleCache {
        /* リソースパスで構成ルールをマッピングする。 */
        static final Hashtable cache = new Hashtable();
    }

    /**
     * 構成ルールを生成して返却します。
     * 
     * @param rulePath
     *            構成ルールファイルへのクラスパス
     * @param classLoader
     *            生成に使用するクラスローダ
     * @return 構成ルール
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static NodeCompositeRule getNodeCompositeRule(String rulePath, ClassLoader classLoader)
            throws NoSuchResourceException {
        synchronized (CompositeRuleCache.cache) {
            NodeCompositeRule rule = (NodeCompositeRule) CompositeRuleCache.cache.get(rulePath);
            if (rule == null) {
                rule = loadNodeCompositeRule(rulePath, classLoader);
                CompositeRuleCache.cache.put(rulePath, rule);
            }
            return rule;
        }
    }

    /* 構成ルールを読み取り生成して返却します。 */
    private static NodeCompositeRule loadNodeCompositeRule(String rulePath, ClassLoader classLoader)
            throws NoSuchResourceException {
        InputStream inStream = ResourceLoader.getResourceAsStream(rulePath, classLoader);
        XMLNodeCompositeRuleFactory factory = new XMLNodeCompositeRuleFactory();
        factory.setClassLoader(classLoader);
        return factory.create(inStream);
    }

}
