/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.ginkgo;

import java.util.HashMap;
import java.util.Iterator;

import shohaku.core.helpers.Eval;

/**
 * XMLドキュメントの構成情報を表現する機能を提供します。
 */
public class Document {

    /* ドキュメントのコンテキスト情報。 */
    private final DocumentContext context;

    /* 親ドキュメント。 */
    private final Document parent;

    /* ドキュメントの公開識別子。 */
    private String id;

    /* 識別子を持つノードの値を保管します。 */
    private final HashMap nodeValueMap;

    /* 子の構成処理に必要な情報のみを保持したドキュメントで有るかを示す。 */
    private final boolean isPreserve;

    /* 初期化します（XML解析時に生成されます）。 */
    Document(Ginkgo ginkgo, NodeCompositeRule rule, Document parent) {
        this.context = new DocumentContext(ginkgo, rule);
        this.parent = parent;
        this.nodeValueMap = new HashMap();
        this.isPreserve = false;

    }

    /* Preserve Document を初期化します。 */
    private Document(Document srcDocument) {
        this.context = null;
        this.parent = srcDocument.parent;
        this.id = srcDocument.id;
        this.nodeValueMap = srcDocument.nodeValueMap;
        this.isPreserve = true;
    }

    /**
     * ドキュメントのコンテキスト情報を返却します。
     * 
     * @return ドキュメントのコンテキスト情報
     */
    public DocumentContext getContext() {
        return context;
    }

    /**
     * 親ドキュメントを返却します。
     * 
     * @return 親ドキュメント
     */
    public Document getParent() {
        return parent;
    }

    /**
     * ドキュメントの公開識別子を返却します。
     * 
     * @return ドキュメントの公開識別子
     */
    public String getId() {
        return id;
    }

    /**
     * ドキュメントの公開識別子を格納します。
     * 
     * @param documentId
     *            ドキュメントの公開識別子
     */
    void setId(String documentId) {
        this.id = documentId;
    }

    /*
     * ID
     */

    /**
     * 全てのノードの値の識別子の反復子を返す。
     * 
     * @return 全てのノードの値の識別子
     */
    public Iterator getNodeValueIdIterator() {
        return this.nodeValueMap.keySet().iterator();
    }

    /**
     * 引数のノードの値の識別子が存在する場合<code>true</code>を返却します。
     * 
     * @param id
     *            ID
     * @return 引数のノードの値の識別子が存在する場合<code>true</code>
     */
    public boolean containsId(String id) {
        // return this.nodeValueMap.containsKey(id);
        if (id == null) {
            throw new NullPointerException();
        }
        if (Eval.isContains(id, ':')) {
            int off = id.indexOf(':');
            String documentId = id.substring(0, off);
            String nodeId = id.substring(++off);
            return containsId(documentId, nodeId);
        } else {
            return this.nodeValueMap.containsKey(id);
        }
    }

    /* ドキュメント識別子と識別子が示すノードの値が存在する場合<code>true</code>を返却します。 */
    private boolean containsId(String documentId, String nodeId) {
        if (documentId.equals(getId())) {
            return this.nodeValueMap.containsKey(nodeId);
        }
        if (null != getParent()) {
            return getParent().containsId(documentId, nodeId);
        }
        return false;
    }

    /**
     * IDが示すノードの値を返す。
     * 
     * @param id
     *            ID
     * @return IDが示すノードの値
     * @throws NullPointerException
     *             <code>id</code>が<code>null</code>の場合発生する
     */
    public Object getNodeValueById(String id) {
        if (id == null) {
            throw new NullPointerException();
        }
        if (Eval.isContains(id, ':')) {
            int off = id.indexOf(':');
            String documentId = id.substring(0, off);
            String nodeId = id.substring(++off);
            return getNodeValueById(documentId, nodeId);
        } else {
            return this.nodeValueMap.get(id);
        }
    }

    /* ドキュメントIDとIDが示すノードの値を返す。 */
    private Object getNodeValueById(String documentId, String nodeId) {
        if (documentId.equals(getId())) {
            return this.nodeValueMap.get(nodeId);
        }
        if (null != getParent()) {
            return getParent().getNodeValueById(documentId, nodeId);
        }
        return null;
    }

    /* IDを持つノードの値を追加します。 */
    void addReferenceNodeValue(Node node) {
        if (node == null) {
            throw new NullPointerException();
        }
        if (node.isType(Node.TYPE_VALUE)) {
            String id = node.getId();
            if (!Eval.isBlank(id)) {
                if (this.nodeValueMap.containsKey(id)) {
                    throw new GinkgoException("That ID has already been registered. id:" + id + ".");
                }
                this.nodeValueMap.put(id, node.getNodeValue());
            }
        }

    }

    /**
     * 子の構成処理に必要な情報のみを保持したドキュメントの場合<code>true</code>を返却します。
     * 
     * @return 子の構成処理に必要な情報のみを保持したドキュメントの場合<code>true</code>
     */
    public boolean isPreserve() {
        return this.isPreserve;
    }

    /**
     * 子のドキュメントの構成処理に必要な情報のみを保持したドキュメントを返却します。<br>
     * 
     * @return 子の構成処理に必要な情報のみを保持したドキュメント
     */
    public Document getPreserveDocument() {
        return new Document(this);
    }

    /*
     * Ginkgo.parse
     */

    /**
     * ノードの解析処理プロセスの開始の通知を受けます。
     * 
     * @param node
     *            ノード
     */
    void startElement(Node node) {
        // no op
    }

    /**
     * ノードの解析処理プロセスの終了の通知を受けます。
     * 
     * @param node
     *            ノード
     */
    void endElement(Node node) {
        addReferenceNodeValue(node);
    }

}
