/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.kosho;

import java.io.File;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URI;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;

import shohaku.core.collections.Group;
import shohaku.core.collections.params.Parameters;

/**
 * 多様な型でリソースデータを取得する機能を持つラッパーオブジェクトを提供します。
 */
public class DefaultKoshoResources implements KoshoResources {

    /** 保管するコンフィギュレーションリソース。 */
    protected KoshoResources lookup;

    /**
     * デフォルトコンストラクタ。
     */
    public DefaultKoshoResources() {
        super();
    }

    /**
     * コンフィギュレーションリソースを保管して初期化します。
     * 
     * @param resources
     *            コンフィギュレーションリソース
     */
    public DefaultKoshoResources(KoshoResources resources) {
        setResources(resources);
    }

    /**
     * 保管するコンフィギュレーションリソースを返却します。
     * 
     * @return コンフィギュレーションリソース
     */
    public KoshoResources getResources() {
        return lookup;
    }

    /**
     * コンフィギュレーションリソースを格納します。
     * 
     * @param resources
     *            コンフィギュレーションリソース
     */
    public void setResources(KoshoResources resources) {
        if (resources == null) {
            throw new NullPointerException("KoshoResources as null");
        }
        lookup = resources;
    }

    /**
     * 全てのリソース識別子を含む反復子を返却します。
     * 
     * @return 全てのリソース識別子を含む反復子
     * @see shohaku.kosho.KoshoResources#keyIterator()
     */
    public Iterator keyIterator() {
        return lookup.keyIterator();
    }

    /**
     * リソースを Object 型で取得します。 <br>
     * 指定された識別子が存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            リソース名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @see shohaku.kosho.KoshoResources#getObject(java.lang.String)
     */
    public Object getObject(String key) {
        return lookup.getObject(key);
    }

    /**
     * 指定された文字列型の識別子が示す値を返却します。 <br>
     * 指定された識別子が存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            リソース識別子
     * @param defaultValue
     *            デフォルト値
     * @return リソース識別子が示す値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @see shohaku.kosho.KoshoResources#getObject(java.lang.String, java.lang.Object)
     */
    public Object getObject(String key, Object defaultValue) {
        return lookup.getObject(key, defaultValue);
    }

    /**
     * 識別子を持つリソースが存在するかを返却します。
     * 
     * @param key
     *            リソース識別子
     * @return 指定された識別子が含まれている場合 true
     * @throws NullPointerException
     *             key が null の場合発生する
     * @see shohaku.kosho.KoshoResources#containsKey(java.lang.String)
     */
    public boolean containsKey(String key) {
        return lookup.containsKey(key);
    }

    /**
     * 識別子を持つリソースが存在し、指定クラスとキャスト可能な関係に有るを返却します。<br>
     * 値がNullの場合キャスト可能 false 返却します。
     * 
     * @param key
     *            リソース識別子
     * @param type
     *            キャスト可能な関係に有るか検証するクラス
     * @return 指定された識別子が含まれている場合 true
     * @throws NullPointerException
     *             key または type が null の場合発生する
     * @see shohaku.kosho.KoshoResources#containsKey(java.lang.String, java.lang.Class)
     */
    public boolean containsKey(String key, Class type) {
        return lookup.containsKey(key, type);
    }

    /**
     * データの文字列表現(Object#toString())を取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値の文字列表現
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public String getToString(String key) {
        Object o = getObject(key);
        if (o == null) {
            return null;
        }
        return o.toString();
    }

    /**
     * データの文字列表現(Object#toString())を取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値の文字列表現
     * @throws NullPointerException
     *             key が null の場合発生する
     */
    public String getToString(String key, Object defaultValue) {
        Object o = getObject(key, defaultValue);
        if (o == null) {
            return null;
        }
        return o.toString();
    }

    /**
     * データを String 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public String getString(String key) {
        return (String) getObject(key);
    }

    /**
     * データを String 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public String getString(String key, Object defaultValue) {
        return (String) getObject(key, defaultValue);
    }

    /**
     * データを Boolean 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Boolean getBoolean(String key) {
        return (Boolean) getObject(key);
    }

    /**
     * データを Boolean 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Boolean getBoolean(String key, Object defaultValue) {
        return (Boolean) getObject(key, defaultValue);
    }

    /**
     * データを Character 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Character getCharacter(String key) {
        return (Character) getObject(key);
    }

    /**
     * データを Character 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Character getCharacter(String key, Object defaultValue) {
        return (Character) getObject(key, defaultValue);
    }

    /**
     * データを Number 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Number getNumber(String key) {
        return (Number) getObject(key);
    }

    /**
     * データを Number 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Number getNumber(String key, Object defaultValue) {
        return (Number) getObject(key, defaultValue);
    }

    /**
     * データを Byte 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Byte getByte(String key) {
        return (Byte) getObject(key);
    }

    /**
     * データを Byte 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Byte getByte(String key, Object defaultValue) {
        return (Byte) getObject(key, defaultValue);
    }

    /**
     * データを Short 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Short getShort(String key) {
        return (Short) getObject(key);
    }

    /**
     * データを Short 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Short getShort(String key, Object defaultValue) {
        return (Short) getObject(key, defaultValue);
    }

    /**
     * データを Integer 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Integer getInteger(String key) {
        return (Integer) getObject(key);
    }

    /**
     * データを Integer 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Integer getInteger(String key, Object defaultValue) {
        return (Integer) getObject(key, defaultValue);
    }

    /**
     * データを Long 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Long getLong(String key) {
        return (Long) getObject(key);
    }

    /**
     * データを Long 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Long getLong(String key, Object defaultValue) {
        return (Long) getObject(key, defaultValue);
    }

    /**
     * データを Float 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Float getFloat(String key) {
        return (Float) getObject(key);
    }

    /**
     * データを Float 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Float getFloat(String key, Object defaultValue) {
        return (Float) getObject(key, defaultValue);
    }

    /**
     * データを Double 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Double getDouble(String key) {
        return (Double) getObject(key);
    }

    /**
     * データを Double 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Double getDouble(String key, Object defaultValue) {
        return (Double) getObject(key, defaultValue);
    }

    /**
     * データを BigInteger 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigInteger getBigInteger(String key) {
        return (BigInteger) getObject(key);
    }

    /**
     * データを BigInteger 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigInteger getBigInteger(String key, Object defaultValue) {
        return (BigInteger) getObject(key, defaultValue);
    }

    /**
     * データを BigDecimal 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigDecimal getBigDecimal(String key) {
        return (BigDecimal) getObject(key);
    }

    /**
     * データを BigDecimal 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public BigDecimal getBigDecimal(String key, Object defaultValue) {
        return (BigDecimal) getObject(key, defaultValue);
    }

    /**
     * データを java.util.Date 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public java.util.Date getDate(String key) {
        return (java.util.Date) getObject(key);
    }

    /**
     * データを java.util.Date 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public java.util.Date getDate(String key, Object defaultValue) {
        return (java.util.Date) getObject(key, defaultValue);
    }

    /**
     * データを java.io.File 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public File getFile(String key) {
        return (File) getObject(key);
    }

    /**
     * データを java.io.File 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public File getFile(String key, Object defaultValue) {
        return (File) getObject(key, defaultValue);
    }

    /**
     * データを java.util.regex.Pattern 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Pattern getRegex(String key) {
        return (Pattern) getObject(key);
    }

    /**
     * データを java.util.regex.Pattern 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Pattern getRegex(String key, Object defaultValue) {
        return (Pattern) getObject(key, defaultValue);
    }

    /**
     * データを java.net.URI 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public URI getURI(String key) {
        return (URI) getObject(key);
    }

    /**
     * データを java.net.URI 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public URI getURI(String key, Object defaultValue) {
        return (URI) getObject(key, defaultValue);
    }

    /**
     * データを java.util.Properties 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Properties getProperties(String key) {
        return (Properties) getObject(key);
    }

    /**
     * データを java.util.Properties 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Properties getProperties(String key, Object defaultValue) {
        return (Properties) getObject(key, defaultValue);
    }

    /**
     * データを java.util.Collection 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Collection getCollection(String key) {
        return (Collection) getObject(key);
    }

    /**
     * データを java.util.Collection 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Collection getCollection(String key, Object defaultValue) {
        return (Collection) getObject(key, defaultValue);
    }

    /**
     * 配列型で定義された値を java.lang.Object 型で取得します。 <br>
     * 取得された値が配列型でない場合 ClassCastException が発生します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が配列型でない場合に発生する
     */
    public Object getArrays(String key) {
        Object o = getObject(key);
        if (o == null) {
            return null;
        }
        if (o.getClass().isArray()) {
            return o;
        } else {
            throw new ClassCastException("Not Array Object");
        }

    }

    /**
     * 配列型で定義された値を java.lang.Object 型で取得します。 <br>
     * 取得された値が配列型でない場合 ClassCastException が発生します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が配列型でない場合に発生する
     */
    public Object getArrays(String key, Object defaultValue) {
        Object o = getArrays(key);
        if (o == null) {
            o = defaultValue;
        }
        return o;
    }

    /**
     * データを java.util.List 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public List getList(String key) {
        return (List) getObject(key);
    }

    /**
     * データを java.util.List 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public List getList(String key, Object defaultValue) {
        return (List) getObject(key, defaultValue);
    }

    /**
     * データを java.util.Set 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Set getSet(String key) {
        return (Set) getObject(key);
    }

    /**
     * データを java.util.Set 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Set getSet(String key, Object defaultValue) {
        return (Set) getObject(key, defaultValue);
    }

    /**
     * データを java.util.Map 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Map getMap(String key) {
        return (Map) getObject(key);
    }

    /**
     * データを java.util.Map 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Map getMap(String key, Object defaultValue) {
        return (Map) getObject(key, defaultValue);
    }

    /**
     * データを shohaku.core.collections.Group 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Group getGroup(String key) {
        return (Group) getObject(key);
    }

    /**
     * データを shohaku.core.collections.Group 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Group getGroup(String key, Object defaultValue) {
        return (Group) getObject(key, defaultValue);
    }

    /**
     * データを shohaku.core.collections.params.Parameters 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Parameters getParameters(String key) {
        return (Parameters) getObject(key);
    }

    /**
     * データを shohaku.core.collections.params.Parameters 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Parameters getParameters(String key, Object defaultValue) {
        return (Parameters) getObject(key, defaultValue);
    }

    /**
     * クラス型で定義された値を java.lang.Class 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>null</code>を返却します。
     * 
     * @param key
     *            データ名
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Class getType(String key) {
        return (Class) getObject(key);
    }

    /**
     * クラス型で定義された値を java.lang.Class 型で取得します。 <br>
     * 指定されたキーが存在しない場合<code>defaultValue</code>を返却します。
     * 
     * @param key
     *            データ名
     * @param defaultValue
     *            デフォルト値
     * @return 指定された値
     * @throws NullPointerException
     *             key が null の場合発生する
     * @throws ClassCastException
     *             取得値が戻り値の型にキャスト出来ない場合
     */
    public Class getType(String key, Object defaultValue) {
        return (Class) getObject(key, defaultValue);
    }

}
