/*
 * shohaku
 * Copyright (C) 2005  tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
package shohaku.kosho;

import java.io.InputStream;

import shohaku.core.util.LocaleResourceBundleBaseCalculator;
import shohaku.core.util.XResourceBundle;
import shohaku.core.util.XResourceBundleBaseCalculator;
import shohaku.core.util.XResourceBundleCreater;
import shohaku.ginkgo.Document;
import shohaku.ginkgo.DocumentCompositeRule;
import shohaku.ginkgo.NodeCompositeRule;

/**
 * <code>Ginkgo API</code> 用いた階層化リソースバンドルの生成機能の抽象実装を提供します。
 */
public abstract class AbstractGinkgoResourceBundleCreater implements XResourceBundleCreater {

    private final XResourceBundleBaseCalculator bundleBaseCalculator;

    /**
     * ロケールの束縛基準で初期化します。
     */
    public AbstractGinkgoResourceBundleCreater() {
        this(new LocaleResourceBundleBaseCalculator());
    }

    /**
     * 指定の束縛基準を格納して初期化します。
     * 
     * @param calculator
     *            リソースを束ねるための束縛基準
     */
    public AbstractGinkgoResourceBundleCreater(XResourceBundleBaseCalculator calculator) {
        this.bundleBaseCalculator = calculator;
    }

    /**
     * 格納されている束縛基準を返却します。
     * 
     * @return 格納されている束縛基準
     */
    public XResourceBundleBaseCalculator getBundleBaseCalculator() {
        return this.bundleBaseCalculator;
    }

    /**
     * <code>getXMLResourceBundleClass()</code> で返されるクラスのクラスローダを返却します。
     * 
     * @return <code>getXMLResourceBundleClass()</code> で返されるクラスのクラスローダ
     */
    public ClassLoader getDefaultClassLoader() {
        return getGinkgoResourceBundleClass().getClassLoader();
    }

    /**
     * <code>getXMLResourceBundle()</code> から返される <code>AbstractGinkgoResourceBundle</code> 実装を返却します。
     * 
     * @param loader
     *            クラスリーダ
     * @param parent
     *            親バンドル、親がない場合<code>null</code>
     * @param bundleBase
     *            リソースの束縛基準
     * @param baseName
     *            規定名
     * @param bundleName
     *            リソースの固有名
     * @return <code>AbstractGinkgoResourceBundle</code> 実装インスタンス
     */
    public XResourceBundle createBundle(ClassLoader loader, XResourceBundle parent, Object bundleBase, String baseName,
            String bundleName) {

        InputStream stream = KoshoHelper.getGinkgoBundleInputStream(loader, bundleName);
        if (stream != null) {
            DocumentCompositeRule docRule = getDocumentCompositeRule(bundleBase);
            NodeCompositeRule nodeRule = getBundleNodeCompositeRule(loader, baseName);
            Document parentDoc = (parent == null) ? null : ((AbstractGinkgoResourceBundle) parent).getDocument();
            try {
                return getGinkgoResourceBundle(loader, parent, parentDoc, bundleBase, stream, docRule, nodeRule);
            } finally {
                try {
                    stream.close();
                } catch (Exception e) {
                    // no op
                }
            }
        }
        return null;
    }

    /*
     * private
     */

    private NodeCompositeRule getBundleNodeCompositeRule(ClassLoader loader, String baseName) {
        return KoshoHelper.findGinkgoBundleNodeCompositeRule(getGinkgoResourceBundleClass(), loader, baseName);
    }

    private DocumentCompositeRule getDocumentCompositeRule(Object bundleBase) {
        return KoshoHelper.getGinkgoBundleDocumentCompositeRule(getBundleBaseCalculator(), bundleBase);
    }

    /*
     * abstract
     */

    /**
     * <code>Ginkgo API</code> 用いた階層化リソースバンドルを生成して返却します。
     * 
     * @param loader
     *            クラスリーダ
     * @param parent
     *            親バンドル、親がない場合<code>null</code>
     * @param parentDoc
     *            親ドキュメント、親がない場合<code>null</code>
     * @param bundleBase
     *            リソースの束縛基準
     * @param stream
     *            リソースの入力ストリーム
     * @param docRule
     *            リソースのドキュメント構成ルール
     * @param nodeRule
     *            リソースのノード構成ルール
     * @return 階層化リソースバンドルの新規インスタンス
     */
    protected abstract AbstractGinkgoResourceBundle getGinkgoResourceBundle(ClassLoader loader, XResourceBundle parent,
            Document parentDoc, Object bundleBase, InputStream stream, DocumentCompositeRule docRule,
            NodeCompositeRule nodeRule);

    /**
     * <code>Ginkgo API</code> 用いた階層化リソースバンドルのクラスを返します。<br>
     * <code>getXMLResourceBundle()</code> で返されるクラスと同じ <code>AbstractGinkgoResourceBundle</code> の実装クラスで有る必要があります。
     * 
     * @return <code>Ginkgo API</code> 用いた階層化リソースバンドルのクラス
     */
    protected abstract Class getGinkgoResourceBundleClass();

}