/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.node;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import shohaku.composer.CompositeContext;
import shohaku.composer.CompositeException;
import shohaku.composer.DocumentContext;
import shohaku.composer.Node;
import shohaku.composer.NodeContext;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;

/**
 * デフォルトノードを構成するためのユーティリティーを提供します。
 */
public class NodeCompositeUtilities {

    /** プリミティブ型のクラスと省略名のマッピング。 */
    public static final Map PRIMITIVE_FOR_TYPE_MAP;
    static {
        HashMap m = new HashMap();
        //primitive
        m.put("byte", Byte.TYPE);
        m.put("short", Short.TYPE);
        m.put("int", Integer.TYPE);
        m.put("long", Long.TYPE);
        m.put("float", Float.TYPE);
        m.put("double", Double.TYPE);
        m.put("boolean", Boolean.TYPE);
        m.put("char", Character.TYPE);
        m.put("void", Void.TYPE);
        PRIMITIVE_FOR_TYPE_MAP = Collections.unmodifiableMap(m);
    }

    /** 基本的な型のクラスと省略名のマッピング。 */
    public static final Map SHORT_NAME_FOR_TYPE_MAP;
    static {
        HashMap m = new HashMap();
        //primitive
        m.putAll(PRIMITIVE_FOR_TYPE_MAP);
        //primitive object
        m.put("Byte", Byte.class);
        m.put("Short", Short.class);
        m.put("Integer", Integer.class);
        m.put("Long", Long.class);
        m.put("Float", Float.class);
        m.put("Double", Double.class);
        m.put("Boolean", Boolean.class);
        m.put("Character", Character.class);
        m.put("Void", Void.class);
        //object
        m.put("Object", Object.class);
        m.put("Class", Class.class);
        m.put("System", System.class);
        m.put("String", String.class);
        SHORT_NAME_FOR_TYPE_MAP = Collections.unmodifiableMap(m);
    }

    /** プリミティブ型のクラスとクラス名のマッピング。 */
    public static final Map PRIMITIVE_CLASS_NAME_MAP;
    static {
        HashMap m = new HashMap();
        //primitive
        m.put(Byte.TYPE, "B");
        m.put(Short.TYPE, "S");
        m.put(Integer.TYPE, "I");
        m.put(Long.TYPE, "J");
        m.put(Float.TYPE, "F");
        m.put(Double.TYPE, "D");
        m.put(Boolean.TYPE, "Z");
        m.put(Character.TYPE, "C");
        m.put(Void.TYPE, "V");
        PRIMITIVE_CLASS_NAME_MAP = Collections.unmodifiableMap(m);
    }

    /*
     * Helper
     */

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param s
     *            クラスを示す文字列
     * @param context
     *            ノードのコンテキスト情報
     * @return ロードされたクラス
     */
    public static Class loadClass(String s, NodeContext context) {
        return loadClass(s, context.getDocumentContext());
    }

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param s
     *            クラスを示す文字列
     * @param context
     *            ドキュメントのコンテキスト情報
     * @return ロードされたクラス
     */
    public static Class loadClass(String s, DocumentContext context) {
        return loadClass(s, context.getCompositeContext());
    }

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param s
     *            クラスを示す文字列
     * @param context
     *            解析処理のコンテキスト情報
     * @return ロードされたクラス
     */
    public static Class loadClass(String s, CompositeContext context) {
        return loadClass(s, context.getClassLoader());
    }

    /**
     * 指定された文字列からクラスをロードし返却します。
     * 
     * @param s
     *            クラスを示す文字列
     * @param loader
     *            クラスローダー
     * @return ロードされたクラス
     */
    public static Class loadClass(String s, ClassLoader loader) {
        try {

            //拡張の配列表記の検証
            int offset = s.length();
            StringBuffer aSymbol = new StringBuffer();
            while (s.charAt(offset - 2) == '[' && s.charAt(offset - 1) == ']') {
                offset -= 2;
                aSymbol.append('[');
            }
            boolean isArray = (aSymbol.length() > 0);

            //クラス名を取得（配列の場合は基クラス）
            String className = (isArray) ? s.substring(0, offset) : s;

            //拡張のクラス表記の検証
            Class c = (Class) SHORT_NAME_FOR_TYPE_MAP.get(className);
            //配列型以外の拡張のクラス表記は返却
            if (c != null && !isArray) {
                return c;
            }
            //配列の場合は配列の表記を付加する
            if (c != null && isArray) {
                String primitive = (String) PRIMITIVE_CLASS_NAME_MAP.get(c);
                if (primitive == null) {
                    className = aSymbol + "L" + c.getName() + ";";
                } else {
                    className = aSymbol + primitive;
                }
            }

            //クラスリード（拡張表記以外は変更なし）
            return ResourceLoader.getClass(className, loader);

        } catch (NoSuchResourceException e) {
            throw new CompositeException("Class情報のロードに失敗しました class name:" + s, e);
        }

    }

    /* 拡張の数値書式の正規表現パターン。 */
    private static final Pattern numberPattern = Pattern
            .compile("^[-+]?(?:0|0X|0x)?[0-9-a-fA-F](?:[-+0-9-a-fA-F,]*[-+0-9-a-fA-F])?(?:\\.[-+0-9-a-fA-F]+)?$");

    /* 拡張の数値書式の拡張文字を削除するパターン */
    private static final Pattern numberRemovePattern = Pattern.compile("^\\+|,");

    /**
     * 数値表現の拡張書式をJavaの数値規約の書式に変換する。
     * 
     * @param value
     *            数値の文字シーケンス
     * @return Java書式の数値文字列
     * @throws NumberFormatException
     *             書式が数値表現として解析に失敗した場合
     */
    public static String toJavaNumberString(CharSequence value) {

        if (numberPattern.matcher(value).matches()) {
            return numberRemovePattern.matcher(value).replaceAll("");
        } else {
            throw new NumberFormatException("value:" + value);
        }
    }

    //    public static String convertNumberString(CharSequence value) {
    //        int len = value.length();
    //        StringBuffer num = new StringBuffer(len);
    //        int period = 0;
    //        boolean sign = false;
    //        for (int i = 0; i < len; i++) {
    //            char c = value.charAt(i);
    //            if (i == 0) {
    //                if (',' == c) {
    //                    throw new NumberFormatException("comma is illegal at the head. value:" + value);
    //                }
    //                sign = ('-' == c || '+' == c);
    //            } else {
    //                if (i == 1 && sign && ',' == c) {
    //                    throw new NumberFormatException("comma is illegal at the head. value:" + value);
    //                }
    //                if (',' == c && period > 0) {
    //                    throw new NumberFormatException("comma is illegal after the decimal point. value:" + value);
    //                }
    //            }
    //            if ('.' == c) {
    //                period++;
    //            }
    //            if (',' != c) {
    //                num.append(c);
    //            }
    //        }
    //        if (num.length() == 0) {
    //            throw new NumberFormatException("value:" + value);
    //        }
    //        return num.toString();
    //    }
    /**
     * 公開ノードを示す区切り文字':'を判別して、指定されたIDに対応するノードを返す。
     * 
     * @param id
     *            ID
     * @return IDが示すノード
     * @throws NullPointerException
     *             id が null の場合発生する
     */
    public static Node getReferenceNodeById(Node node, String id) {
        return getReferenceNodeById(node.getNodeContext(), id);
    }

    /**
     * 公開ノードを示す区切り文字':'を判別して、指定されたIDに対応するノードを返す。
     * 
     * @param id
     *            ID
     * @return IDが示すノード
     * @throws NullPointerException
     *             id が null の場合発生する
     */
    public static Node getReferenceNodeById(NodeContext nodeContext, String id) {
        return getReferenceNodeById(nodeContext.getDocumentContext(), nodeContext.getCompositeContext(), id);
    }

    /**
     * 公開ノードを示す区切り文字':'を判別して、指定されたIDに対応するノードを返す。
     * 
     * @param id
     *            ID
     * @return IDが示すノード
     * @throws NullPointerException
     *             id が null の場合発生する
     */
    public static Node getReferenceNodeById(DocumentContext docContext, CompositeContext compContext, String id) {
        if (id == null) {
            throw new NullPointerException();
        }
        if (-1 == id.lastIndexOf(':')) {
            return docContext.getNodeById(id);
        } else {
            return compContext.getPublicNodeById(id);
        }
    }

    /* 拡張の数値書式の正規表現パターン。 */
    private static final Pattern referenceExPattern = Pattern.compile("([#$%]\\{(?:\\\\\\}|[^}])*\\})");

    /* 拡張の数値書式の拡張文字を削除するパターン */
    private static final Pattern referenceExRemovePattern = Pattern.compile("\\\\\\}");

    /**
     * 指定の文字列を参照構文として解析し変換し返却します。
     * 
     * @param context
     *            ドキュメントのコンテキスト情報
     * @param s
     *            変換元の文字列
     * @return 変換後の文字列
     */

    public static String filterReferenceExpression(DocumentContext context, String s) {

        StringBuffer sb = new StringBuffer(s.length());
        Matcher m = referenceExPattern.matcher(s);

        int i = 0;
        while (i < s.length()) {
            if (m.find(i)) {
                sb.append(s.substring(i, m.start(1)));
                //get value
                String refex = referenceExRemovePattern.matcher(m.group(1)).replaceAll("}");
                char mark = refex.charAt(0);
                String id = refex.substring(2, refex.length() - 1);
                Object value = getReferenceExpValue(context, mark, id);
                sb.append(value);
                i = m.end(1);
            } else {
                sb.append(s.substring(i));
                break;
            }
        }
        return sb.toString();
    }

    //    public static String filterReferenceExpression(DocumentContext context, String s) {
    //
    //        final int _nonum = -1;
    //        final char _nochar = '\u0000';
    //
    //        StringBuffer sb = new StringBuffer(s);
    //
    //        char[] chars = s.toCharArray();
    //        int begin = _nonum;
    //        int end = _nonum;
    //        int shift = 0;
    //        char mark = _nochar;
    //        char c1 = _nochar;
    //        char c2 = _nochar;
    //        char c3 = _nochar;
    //        for (int i = 2; i < chars.length; i++) {
    //            if (i == 2) {
    //                //begin index
    //                c1 = chars[0];
    //                c2 = chars[1];
    //                c3 = chars[2];
    //                if (c1 == '\\' && contain("\\#$%{}", c2)) {
    //                    sb.replace(0, 2, String.valueOf(c2));
    //                    shift += 1 - 2;
    //                    c2 = _nochar;
    //                } else {
    //                    //check start
    //                    if ((c1 == '%' || c1 == '$' || c1 == '#') && c2 == '{') {
    //                        begin = i - 1 + shift;
    //                        mark = c1;
    //                    }
    //                }
    //            } else {
    //                //next index
    //                c1 = c2;
    //                c2 = c3;
    //                c3 = chars[i];
    //            }
    //
    //            if (c1 != '\\' && c2 == '\\' && contain("\\#$%{}", c3)) {
    //                sb.replace(i - 1 + shift, i + 1 + shift, String.valueOf(c3));
    //                shift += 1 - 2;
    //                c3 = _nochar;
    //            } else {
    //                //check start
    //                if (begin == _nonum && contain("#$%", c2) && c3 == '{') {
    //                    begin = i + shift;
    //                    mark = c2;
    //                }
    //                //check end
    //                if (begin != _nonum && c2 != '\\' && c3 == '}') {
    //                    end = i + shift;
    //                }
    //                //setting value
    //                if (begin != _nonum && end != _nonum) {
    //                    //get value
    //                    String val = String.valueOf(getReferenceExpValue(context, mark, sb.substring(begin + 1, end)));
    //                    //set value
    //                    sb.replace(begin - 1, end + 1, val);
    //                    //shift index (value.length - src.length)
    //                    shift += (val.length() - ((end + 1) - (begin - 1)));
    //                    //init chars
    //                    mark = _nochar;
    //                    begin = _nonum;
    //                    end = _nonum;
    //                }
    //            }
    //        }
    //        return sb.toString();
    //    }

    private static boolean contain(String chars, char c) {
        return (0 <= chars.indexOf(c));
    }

    private static Object getReferenceExpValue(DocumentContext context, char mark, String id) {
        Object o = null;
        if (mark == '%') {
            o = context.getCompositeContext().getAttribute(id);
        } else if (mark == '$') {
            o = context.getAttribute(id);
        } else {
            Node val = NodeCompositeUtilities.getReferenceNodeById(context, context.getCompositeContext(), id);
            if (val != null && val.isType(Node.TYPE_VALUE)) {
                o = val.getNodeValue();
            }
        }
        return o;
    }

}
