/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.node;

import java.util.Iterator;

import shohaku.composer.CompositeContext;
import shohaku.composer.DocumentContext;
import shohaku.composer.Node;

/**
 * デフォルトノードを構成するための機能を提供します。 <br>
 * このクラスは <code>CompositeRule.getFeature(Class)</code> から <code>NodeCompositeFeature.class</code>
 * をキーとして取得し各ノードが利用します。 <br>
 * デフォルトノードを使用する場合はこのクラスまたはサブクラスが必ず登録されている必要があります。
 */
public class NodeCompositeFeature {

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @return 指定された名前の属性値が示す参照値
     */
    public Class getTextToClass(Node node) {
        return getAttributeToClass(node, AbstractBaseNode.TEXT_ALIAS);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @return 指定された名前の属性値が示す参照値
     */
    public Class getAttributeToClass(Node node, String name) {
        String className = node.getAttribute(name);
        if (null == className) {
            return null;
        }
        return NodeCompositeUtilities.loadClass(className, node.getNodeContext());
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>defaultValue</code> が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値が示す参照値又は <code>defaultValue</code>
     */
    public Class getAttributeToClass(Node node, String name, Class defaultValue) {
        Class o = getAttributeToClass(node, name);
        if (null == o) {
            return defaultValue;
        }
        return o;
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @return 指定された名前の属性値が示す参照値
     */
    public Object getTextRef(Node node) {
        return getAttributeRef(node, AbstractBaseNode.TEXT_ALIAS);
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param node
     *            ノードのコンテキスト情報
     * @param name
     *            属性名
     * @return 指定された名前の属性値が示す参照値
     */
    public Object getAttributeRef(Node node, String name) {
        String s = node.getAttribute(name);
        if (null == s) {
            return null;
        }
        s = s.trim();

        Object o = null;
        if (s.length() != 0) {
            if (s.charAt(0) == '%') {
                String id = s.substring(1);
                o = getCompositeContextAttribute(node, id);
            } else if (s.charAt(0) == '$') {
                String id = s.substring(1);
                o = getDocumentContextAttribute(node, id);
            } else {
                String id = (s.charAt(0) == '#') ? s.substring(1) : s;
                o = getReferenceNodeValue(node, id);
            }
        }
        return o;
    }

    /**
     * 指定された名前を持つ属性の値を参照構文として解析して参照値を返却します。 <br>
     * 指定された名前の属性又は属性値の示す参照値が存在しない場合は、 <code>defaultValue</code> が返されます。
     * 
     * @param node
     *            ノード
     * @param name
     *            属性名
     * @param defaultValue
     *            指定の属性が存在しない場合に返却される値
     * @return 指定された名前の属性値が示す参照値又は <code>defaultValue</code>
     */
    public Object getAttributeRef(Node node, String name, Object defaultValue) {
        Object o = getAttributeRef(node, name);
        if (null == o) {
            return defaultValue;
        }
        return o;
    }

    /**
     * テキスト情報を評価して返却します。 <br>
     * 全ての子の値型ノードの文字列表現を一つの文字列に連結して返却します。
     * 
     * @param node
     *            ノード
     * @return テキスト情報
     */
    public String getTextString(Node node) {
        StringBuffer sb = new StringBuffer();
        Iterator i = node.getNodeContext().childIterator();
        while (i.hasNext()) {
            Node n = (Node) i.next();
            if (n.isType(Node.TYPE_VALUE)) {
                sb.append(n.getNodeValue());
            }
        }
        return sb.toString().trim();
    }

    private final String[] BOOL_STRINGS = new String[] { "true", "false", "yes", "no", "on", "off", "1", "0" };

    /**
     * @param value
     * @return
     */
    public boolean isBoolean(String value) {
        for (int i = 0; i < BOOL_STRINGS.length; i++) {
            if (BOOL_STRINGS[i].equalsIgnoreCase(value)) {
                return true;
            }
        }
        return false;
    }

    private final String[] TRUE_STRINGS = new String[] { "true", "yes", "on", "1" };

    /**
     * @param value
     * @return
     */
    public boolean getBoolean(String value) {
        for (int i = 0; i < TRUE_STRINGS.length; i++) {
            if (TRUE_STRINGS[i].equalsIgnoreCase(value)) {
                return true;
            }
        }
        return false;
    }

    /**
     * @param value
     * @return
     */
    public String toNumberString(CharSequence value) {
        return NodeCompositeUtilities.toJavaNumberString(value);
    }

    /*
     * private
     */

    //    private CharSequence trim(CharSequence chars, char trim) {
    //        int len = chars.length();
    //        int st = 0;
    //        for (int i = 0; i < len; i++) {
    //            char c = chars.charAt(i);
    //            if (trim == c) {
    //                st++;
    //            } else {
    //                break;
    //            }
    //        }
    //        for (int i = (len - 1); i >= 0; i--) {
    //            char c = chars.charAt(i);
    //            if (trim == c) {
    //                len--;
    //            } else {
    //                break;
    //            }
    //        }
    //        return chars.subSequence(st, len);
    //    }
    private Object getReferenceNodeValue(Node node, String id) {
        Object o = null;
        Node val = NodeCompositeUtilities.getReferenceNodeById(node, id);
        if (val != null && val.isType(Node.TYPE_VALUE)) {
            o = val.getNodeValue();
        }
        return o;
    }

    private Object getCompositeContextAttribute(Node node, String id) {
        CompositeContext context = node.getNodeContext().getCompositeContext();
        return context.getAttribute(id);
    }

    private Object getDocumentContextAttribute(Node node, String id) {
        DocumentContext context = node.getNodeContext().getDocumentContext();
        return context.getAttribute(id);
    }

}
