/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.node;

import shohaku.composer.AbstractNode;
import shohaku.composer.Composer;
import shohaku.composer.CompositeContext;
import shohaku.composer.CompositeException;
import shohaku.composer.CompositeRule;
import shohaku.composer.DocumentContext;
import shohaku.composer.NodeContext;

/**
 * デフォルト実装でのテンプレート実装を提供します。
 */
public abstract class AbstractBaseNode extends AbstractNode {

    /*
     * Node Type's
     */

    /* 有効なノード種別のキャッシュ。 */
    private int[] types;

    /**
     * 指定されたノードの種別に属する場合Trueを返却します。
     * 
     * @param type
     *            検証する種別
     * @return 指定されたノードの種別に属する場合True
     */
    public boolean isType(int type) {
        if (types == null) {
            types = getTypes();
        }
        for (int i = 0; i < types.length; i++) {
            if (types[i] == type) {
                return true;
            }
        }
        return false;
    }

    /**
     * 有効なノード種別を返却します。
     * 
     * @return 有効なノード種別
     */
    abstract protected int[] getTypes();

    /*
     * Text
     */

    /** テキスト情報を属性として表すための別名。 */
    public static final String TEXT_ALIAS = "%text";

    /* テキストをキャッシュする. */
    private String cacheText;

    /**
     * テキストのキャッシュを返却します。
     * 
     * @return テキストのキャッシュ
     */
    protected String getCacheText() {
        return cacheText;
    }

    /**
     * テキストのキャッシュを格納します。
     * 
     * @param s
     *            テキストのキャッシュ
     */
    protected void setCacheText(String s) {
        this.cacheText = s;
    }

    /**
     * テキスト情報を評価して返却します。
     * 
     * @return テキスト情報
     */
    public String getText() {
        return getAttribute(TEXT_ALIAS);
    }

    /**
     * テキスト情報を評価して返却します。 <br>
     * デフォルト実装では全ての子の値型ノードの文字列表現を一つの文字列に連結して返却します。 テキストの属性表現との整合性を保つため、サブクラスが振る舞いを変える場合は getText()
     * ではなくこのメソッドをオーバライドしてください。
     * 
     * @return テキスト情報
     */
    protected String toTextString() {
        if (null == getCacheText()) {
            setCacheText(getFeature().getTextString(this));
        }
        return getCacheText();
    }

    /*
     * Attribute
     */

    /**
     * 属性を検索し存在しない場合は拡張属性を検索し値を返却します。 <br>
     * 属性名が存在しない場合 <code>null</code> を返す。
     * 
     * @param name
     *            属性名
     * @return 属性値
     */
    public String getAttribute(String name) {
        String _name = getNodeContext().getExAttribute("*" + name);
        if (_name != null) {
            name = _name;
        }
        String value = null;
        if (TEXT_ALIAS.equals(name)) {
            value = toTextString();
        } else {
            value = getNodeContext().getElemAttribute(name);
            if (value == null) {
                value = getNodeAttribute(name);
            }
            if (value == null) {
                value = getNodeContext().getExAttribute(name);
            }
        }
        return value;
    }

    /**
     * 指定された名前を持つノード固有の属性の値を返却します。 <br>
     * 指定された名前が存在しない場合は、 <code>null</code> が返されます。
     * 
     * @param name
     *            属性名
     * @return 指定された名前の属性値
     */
    protected String getNodeAttribute(String name) {
        return null;
    }

    /*
     * NodeCompositeFeature
     */

    /** ノードを構成するユーティリティ機能。 */
    private NodeCompositeFeature feature;

    /**
     * ノードを構成するユーティリティ機能を返却します。 <br>
     * 戻り値は NodeCompositeFeature クラスまたはサブクラスのインスタンス。
     * 
     * @return ノードを構成するユーティリティ機能
     */
    protected NodeCompositeFeature getFeature() {
        return feature;
    }

    /* ノードを構成するユーティリティ機能を格納します。 */
    private void setFeature(NodeCompositeFeature feature) {
        this.feature = feature;
    }

    /*
     * Helper
     */

    /**
     * ドキュメントのコンテキスト情報を返却します。
     * 
     * @return ドキュメントのコンテキスト情報
     */
    protected DocumentContext getDocumentContext() {
        return getNodeContext().getDocumentContext();
    }

    /**
     * 解析処理のコンテキスト情報を返却します。
     * 
     * @return 解析処理のコンテキスト情報
     */
    protected CompositeContext getCompositeContext() {
        return getNodeContext().getCompositeContext();
    }

    /**
     * 解析処理を実行する Composer を返却します。
     * 
     * @return 解析処理を実行する Composer。
     */
    protected Composer getComposer() {
        return getCompositeContext().getComposer();
    }

    /**
     * 解析処理に使用する CompositeRule を返却します。
     * 
     * @return CompositeRule。
     */
    public CompositeRule getCompositeRule() {
        return getDocumentContext().getCompositeRule();
    }

    /**
     * 解析処理に使用する ClassLoader を返します.
     * 
     * @return 解析処理に使用する ClassLoader
     */
    protected ClassLoader getClassLoader() {
        return getComposer().getClassLoader();
    }

    /*
     * Event
     */

    /**
     * 解析処理を開始する直前に初期化の通知を受ける。 <br>
     * オーバライドした場合必ず親クラスの同メソッドを呼び出してください。
     * 
     * @param nodeContext
     *            ノードのコンテキスト情報
     * @throws CompositeException
     *             構成情報例外。
     */
    public void initialize(NodeContext nodeContext) {
        super.initialize(nodeContext);
        //Feature
        CompositeRule rule = getCompositeRule();
        Object f = rule.getFeature(NodeCompositeFeature.class);
        setFeature((f != null) ? (NodeCompositeFeature) f : new NodeCompositeFeature());
    }

}
