/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.helpers;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import shohaku.composer.Composer;
import shohaku.composer.CompositeException;
import shohaku.composer.CompositeRule;
import shohaku.composer.CompositeRuleFactory;
import shohaku.composer.Node;
import shohaku.composer.NodeRule;
import shohaku.composer.Substitutor;
import shohaku.core.collections.Parameters;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;

/**
 * XML形式で定義されたデータから DefaultCompositeRule （又はそのサブクラス）を生成するファクトリを提供します。
 */
public class XMLCompositeRuleFactory implements CompositeRuleFactory {

    /*
     * static initialize
     */

    /* 構成情報を読み取る構成ルール情報を生成を示す。 */
    private static final DefaultCompositeRule ruleCompositeRule;
    static {

        DefaultCompositeRule rule = new DefaultCompositeRule();

        List nodes = new ArrayList(48);

        //処理頻度の高いノード
        putNode(nodes, "/composite-rule/nodes/node", "SimpleNode");
        putNode(nodes, "/composite-rule/tags/tag", "SimpleNode");

        //処理頻度の中程度のノード
        //composite-rule/rules
        putNode(nodes, "/composite-rule/rules/xml-composite-rule", "node.beans.ObjectNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/property", "node.beans.SetPropertyInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/method", "node.beans.MethodInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/method/arg", "node.beans.ParameterInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/init", "node.beans.FactoryMethodInfoNode");
        putNode(nodes, "/composite-rule/rules/xml-composite-rule/init/arg", "node.beans.ParameterInfoNode");
        //composite-rule/node/substitutor
        putNode(nodes, "/composite-rule/nodes/node/substitutor", "node.beans.ObjectNode");
        //composite-rule/node/attributes
        putNode(nodes, "/composite-rule/nodes/node/attributes", "node.collection.MapNode");
        putNode(nodes, "/composite-rule/nodes/node/attributes/attribute", "node.collection.NamedTextSingletonMapNode");
        //値ノード
        //base
        putNode(nodes, "/desc", "node.base.DescriptionNode");
        putNode(nodes, "/text", "node.base.CharDataNode");
        putNode(nodes, "/string", "node.base.StringNode");
        putNode(nodes, "/val", "node.base.TextValueNode");
        putNode(nodes, "/char", "node.base.CharacterNode");
        putNode(nodes, "/decimal", "node.base.DecimalNode");
        putNode(nodes, "/byte", "node.base.ByteNode");
        putNode(nodes, "/short", "node.base.ShortNode");
        putNode(nodes, "/int", "node.base.IntgerNode");
        putNode(nodes, "/long", "node.base.LongNode");
        putNode(nodes, "/double", "node.base.DoubleNode");
        putNode(nodes, "/float", "node.base.FloatNode");
        putNode(nodes, "/boolean", "node.base.BooleanNode");
        putNode(nodes, "/type", "node.base.ClassNode");
        putNode(nodes, "/ref", "node.base.ReferenceNode");
        //util
        putNode(nodes, "/date", "node.util.DateNode");
        putNode(nodes, "/time", "node.util.TimeNode");
        putNode(nodes, "/timestamp", "node.util.TimestampNode");
        putNode(nodes, "/uri", "node.util.URINode");
        putNode(nodes, "/regex", "node.util.RegexNode");
        //collection
        putNode(nodes, "/array", "node.collection.ArrayNode");
        putNode(nodes, "/map", "node.collection.MapNode");
        putNode(nodes, "/map/keyValue", "node.collection.SingletonMapNode");
        putNode(nodes, "/list", "node.collection.ListNode");
        putNode(nodes, "/set", "node.collection.SetNode");
        //bean
        putNode(nodes, "/object", "node.beans.ObjectNode");
        putNode(nodes, "/object/property", "node.beans.SetPropertyInfoNode");
        putNode(nodes, "/object/method", "node.beans.MethodInfoNode");
        putNode(nodes, "/object/method/arg", "node.beans.ParameterInfoNode");
        putNode(nodes, "/object/init", "node.beans.FactoryMethodInfoNode");
        putNode(nodes, "/object/init/arg", "node.beans.ParameterInfoNode");
        putNode(nodes, "/call", "node.beans.CallMethodNode");
        putNode(nodes, "/call/arg", "node.beans.ParameterInfoNode");

        //処理が一度のみのノード
        putNode(nodes, "/composite-rule", "SimpleNode");
        putNode(nodes, "/composite-rule/rules", "SimpleNode");
        putNode(nodes, "/composite-rule/nodes", "SimpleNode");
        putNode(nodes, "/composite-rule/tags", "SimpleNode");

        Object[] nodeRuleSet = new Object[nodes.size() * 2];
        int j = 0;
        for (Iterator i = nodes.iterator(); i.hasNext();) {
            NodeRule nodeRule = new NodeRule();

            Object[] entry = (Object[]) i.next();
            String pattern = (String) entry[0];
            String className = (String) entry[1];
            Parameters exAttr = (Parameters) entry[2];

            nodeRule.setPattern(pattern);
            nodeRule.setNodeClass(className);
            nodeRule.setExAttributes(exAttr);

            nodeRuleSet[j++] = pattern;
            nodeRuleSet[j++] = nodeRule;
        }
        rule.setNodeRuleParameters(new Parameters(nodeRuleSet));

        ruleCompositeRule = rule;
    }

    private static void putNode(List nodes, String id, String className) {
        putNode(nodes, id, className, null);
    }

    private static void putNode(List nodes, String id, String className, Object[] atts) {
        Object[] e = new Object[3];
        e[0] = id;
        e[1] = "shohaku.composer." + className;
        if (null == atts) {
            e[2] = Parameters.EMPTY_PARAMETERS;
        } else {
            e[2] = new Parameters(atts);
        }
        nodes.add(e);
    }

    /*
     * implement CompositeRuleFactory
     */

    /* 読み取りに使用するクラスリーダ。 */
    private ClassLoader classLoader;

    /**
     * 引数のシステムIDから CompositeRule を生成して返却します。 システムIDへの定義ファイルへのリソースパスを指定します。
     * 
     * @param systemId
     *            システムID
     * @return CompositeRule の実装クラスのインスタンス
     */
    public CompositeRule create(String systemId) {
        if (classLoader == null) {
            return createImpl(systemId, XMLCompositeRuleFactory.class.getClassLoader());
        } else {
            return createImpl(systemId, classLoader);
        }
    }

    /**
     * 引数の入力ストリームから CompositeRule を生成して返却します。
     * 
     * @param inStream
     *            入力ストリーム
     * @return CompositeRule の実装クラスのインスタンス
     */
    public CompositeRule create(InputStream inStream) {
        if (classLoader == null) {
            return createImpl(inStream, XMLCompositeRuleFactory.class.getClassLoader());
        } else {
            return createImpl(inStream, classLoader);
        }
    }

    /**
     * 読み取りに使用するクラスリーダを返却します。
     * 
     * @return クラスリーダ
     */
    public ClassLoader getClassLoader() {
        return classLoader;
    }

    /**
     * 読み取りに使用するクラスリーダを格納します。
     * 
     * @param loader
     *            クラスリーダ
     */
    public void setClassLoader(ClassLoader loader) {
        classLoader = loader;
    }

    /*
     * private static
     */

    /* 引数のシステムIDから DefaultCompositeRule を生成して返却します。 */
    private static DefaultCompositeRule createImpl(String systemId, ClassLoader classLoader) {

        InputStream inStream = null;
        try {
            inStream = ResourceLoader.getResourceAsStream(systemId, classLoader);
        } catch (NoSuchResourceException e) {
            throw new CompositeException("構成ルールファイルの読取に失敗しました。", e);
        }
        if (inStream == null) {
            throw new CompositeException("構成ルールファイルの読取に失敗しました。");
        }
        return createImpl(inStream, classLoader);

    }

    /* 引数の入力ストリームから DefaultCompositeRule を生成して返却します。 */
    private static DefaultCompositeRule createImpl(InputStream inStream, ClassLoader classLoader) {
        try {

            Node root = load(inStream, ruleCompositeRule, classLoader);
            return init(root);

        } catch (Exception e) {
            throw new CompositeException("構成ルールファイルの読取に失敗しました。", e);
        }
    }

    /* 入力ストリームからノードリストを読み込む。 */
    private static Node load(InputStream inStream, DefaultCompositeRule rule, ClassLoader classLoader) {

        //XML構成ルールファイルの解析
        Composer composer = new Composer();
        if (classLoader != null) {
            composer.setClassLoader(classLoader);
        }
        //XMLの解析
        composer.parse(rule, inStream);

        //ノードを保管する
        Node root = composer.getRoot();

        return root;
    }

    /* ノードを初期化および登録します。 */
    private static DefaultCompositeRule init(Node root) {

        DefaultCompositeRule rule = null;
        HashMap nodeMap = new HashMap();
        HashMap tagMap = new HashMap();

        for (Iterator i = root.getNodeContext().elementIterator(); i.hasNext();) {
            Node n = (Node) i.next();
            String nm = n.getNodeContext().getNodeName();
            if ("rules".equals(nm)) {
                rule = getDefaultCompositeRule(n);
            } else if ("nodes".equals(nm)) {
                nodeTag(n, nodeMap);
            } else if ("tags".equals(nm)) {
                tagTag(n, tagMap);
            }
        }
        if (rule == null) {
            rule = new DefaultCompositeRule();
        }

        Object[] nodeRuleSet = new Object[tagMap.size() * 2];
        int j = 0;
        for (Iterator i = tagMap.keySet().iterator(); i.hasNext();) {
            NodeRule nodeRule = new NodeRule();
            String pattern = (String) i.next();
            String nodeId = (String) tagMap.get(pattern);

            Object[] nodeAtts = (Object[]) nodeMap.get(nodeId);
            String className = (String) nodeAtts[0];
            Parameters exAttributes = (Parameters) nodeAtts[1];
            Substitutor substitutor = (Substitutor) nodeAtts[2];

            nodeRule.setPattern(pattern);
            nodeRule.setNodeClass(className);
            nodeRule.setExAttributes(exAttributes);
            nodeRule.setSubstitutor(substitutor);

            nodeRuleSet[j++] = pattern;
            nodeRuleSet[j++] = nodeRule;
        }
        rule.setNodeRuleParameters(new Parameters(nodeRuleSet));
        return rule;
    }

    /* DefaultCompositeRule を生成して返却します。 */
    private static DefaultCompositeRule getDefaultCompositeRule(Node rulesTag) {
        Iterator i = rulesTag.getNodeContext().elementIterator("xml-composite-rule");
        if (i.hasNext()) {
            Node ruleInfo = (Node) i.next();
            return (DefaultCompositeRule) ruleInfo.getNodeValue();
        }
        return new DefaultCompositeRule();
    }

    /* NODESタグの情報を格納します。 */
    private static void nodeTag(Node nodesTag, HashMap nodeMap) {
        for (Iterator i = nodesTag.getNodeContext().elementIterator("node"); i.hasNext();) {
            Node node = (Node) i.next();
            String nodeId = node.getAttribute("id");

            Object[] nodeAtts = new Object[3];
            nodeMap.put(nodeId, nodeAtts);
            nodeAtts[0] = node.getAttribute("class");
            nodeAtts[1] = Parameters.EMPTY_PARAMETERS;

            for (Iterator j = node.getNodeContext().elementIterator(); j.hasNext();) {
                Node n = (Node) j.next();
                String nm = n.getNodeContext().getNodeName();
                if ("attributes".equals(nm)) {
                    nodeAtts[1] = new Parameters((Map) n.getNodeValue());
                } else if ("substitutor".equals(nm)) {
                    nodeAtts[2] = n.getNodeValue();
                }
            }
        }
    }

    /* TAGSタグの情報を格納します。 */
    private static void tagTag(Node tagsTag, HashMap tagMap) {
        for (Iterator i = tagsTag.getNodeContext().elementIterator("tag"); i.hasNext();) {
            Node info = (Node) i.next();
            tagMap.put(info.getAttribute("name"), info.getAttribute("node"));
        }
    }
}
