/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.helpers;

import java.util.IdentityHashMap;
import java.util.Iterator;
import java.util.Map;

import org.xml.sax.DTDHandler;
import org.xml.sax.EntityResolver;
import org.xml.sax.ErrorHandler;

import shohaku.composer.CompositeException;
import shohaku.composer.CompositeRule;
import shohaku.composer.Node;
import shohaku.composer.NodeRule;
import shohaku.core.collections.Parameters;

/**
 * XML情報を構造化されたオブジェクトに変換するルールを定義するデフォルト実装を提供します。
 */
public class DefaultCompositeRule implements CompositeRule {

    /*
     * implement CompositeRule
     */

    private Boolean validating;

    private Boolean namespaceAware;

    private EntityResolver entityResolver;

    private DTDHandler dtdHandler;

    private ErrorHandler errorHandler;

    private Map features = new IdentityHashMap();

    /**
     * ノードの構成情報を返却します(名前空間には現在未対応)。
     * 
     * @param namespaceURI
     *            解析対象の名前空間URI
     * @param nodeURI
     *            解析対象のXML階層URI
     * @param tagName
     *            タグ名
     * @return ノード構成情報
     */
    public NodeRule findNodeRule(String namespaceURI, String nodeURI, String tagName){

        for (Iterator i = nodeRuleParameters.iterator(); i.hasNext();) {
            Parameters.Entry e = (Parameters.Entry) i.next();
            String suffix = e.getName();
            if (suffix.equals("*") || nodeURI.endsWith(suffix)) {
                return (NodeRule) e.getValue();
            }
        }
        return null;
    }

    /**
     * ドキュメントの公開識別子を返却します。
     * 
     * @param root
     *            ルートノード
     * @return ドキュメントの公開識別子
     */
    public String getPublicId(Node root) {
        return root.getAttribute(getPublicIdAttribute());
    }

    /**
     * ノードの識別子を返却します。
     * 
     * @param node
     *            ノード
     * @return ノードの識別子
     */
    public String getNodeId(Node node) {
        return node.getAttribute(getNodeIdAttribute());
    }

    /**
     * 指定されたIDが示す構造化の拡張機能を返却します。 <br>
     * 指定されたIDに対応する拡張機能が存在しない場合Nullを返します。
     * 
     * @param id
     *            機能を示すクラス
     * @return 拡張機能
     */
    public Object getFeature(Class id) {
        synchronized (features) {
            return features.get(id);
        }
    }

    /*
     * SAX
     */

    /**
     * EntityResolver を格納します。
     * 
     * @param entityResolver
     *            EntityResolver、設定を行わない場合Null
     */
    public void setEntityResolver(EntityResolver entityResolver) {
        this.entityResolver = entityResolver;
    }

    /**
     * EntityResolver を返却します。
     * 
     * @return EntityResolver、設定を行わない場合Null
     */
    public EntityResolver getEntityResolver() {
        return this.entityResolver;
    }

    /**
     * DTDHandler を格納します。
     * 
     * @param dtdHandler
     *            DTDHandler、設定を行わない場合Null
     */
    public void setDTDHandler(DTDHandler dtdHandler) {
        this.dtdHandler = dtdHandler;
    }

    /**
     * DTDHandler を返却します。
     * 
     * @return DTDHandler、設定を行わない場合Null
     */
    public DTDHandler getDTDHandler() {
        return this.dtdHandler;
    }

    /**
     * ErrorHandler を格納します。
     * 
     * @param errorHandler
     *            ErrorHandler、設定を行わない場合Null
     */
    public void setErrorHandler(ErrorHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    /**
     * ErrorHandler を返却します。
     * 
     * @return ErrorHandler、設定を行わない場合Null
     */
    public ErrorHandler getErrorHandler() {
        return this.errorHandler;
    }

    /**
     * SAXParserFactory#validating へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合Null
     */
    public Boolean getValidating() {
        return this.validating;
    }

    /**
     * SAXParserFactory#validating へ設定する値を格納します。
     * 
     * @param validating
     *            設定値、設定を行わない場合Null
     */
    public void setValidating(Boolean validating) {
        this.validating = validating;
    }

    /**
     * SAXParserFactory#namespaceAware へ設定する値を返却します。
     * 
     * @return 設定値、設定を行わない場合Null
     */
    public Boolean getNamespaceAware() {
        return this.namespaceAware;
    }

    /**
     * SAXParserFactory#namespaceAware へ設定する値を格納します。
     * 
     * @param namespaceAware
     *            設定値、設定を行わない場合Null
     */
    public void setNamespaceAware(Boolean namespaceAware) {
        this.namespaceAware = namespaceAware;
    }

    /**
     * 指定されたIDが示す構造化の拡張機能を追加します。
     * 
     * @param id
     *            機能を示すクラス
     * @param feature
     *            拡張機能
     */
    public void addFeature(Class id, Object feature) {
        if (!id.isInstance(feature)) {
            throw new CompositeException("not assignable class, id:" + id + " , feature:" + feature);
        }
        synchronized (features) {
            features.put(id, feature);
        }
    }

    /*
     * 実装固有の機能とプロパティ
     */

    private Parameters nodeRuleParameters;

    private String nodeIdAttribute = "id";

    private String publicIdAttribute = "publicId";

    /*
     * NodeRule
     */

    /**
     * ノードの構成ルールをXMLURIのパターンで保管するパラメータリストを取得します。
     * 
     * @return ノードの構成ルールをXMLURIのパターンで保管するパラメータリスト
     */
    public Parameters getNodeRuleParameters() {
        return nodeRuleParameters;
    }

    /**
     * ノードの構成ルールをXMLURIのパターンで保管するパラメータリストを格納します。
     * 
     * @param params
     *            ノードの構成ルールをXMLURIのパターンで保管するパラメータリスト
     */
    public void setNodeRuleParameters(Parameters params) {
        nodeRuleParameters = params;
    }

    /*
     * Attribute Name
     */

    /**
     * @return
     */
    public String getNodeIdAttribute() {
        return nodeIdAttribute;
    }

    /**
     * @param string
     */
    public void setNodeIdAttribute(String string) {
        nodeIdAttribute = string;
    }

    /**
     * @return
     */
    public String getPublicIdAttribute() {
        return publicIdAttribute;
    }

    /**
     * @param string
     */
    public void setPublicIdAttribute(String string) {
        publicIdAttribute = string;
    }
}