/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer.helpers;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import shohaku.composer.CompositeRule;
import shohaku.core.lang.NoSuchResourceException;
import shohaku.core.lang.ResourceLoader;

/**
 * <code>CompositeRule</code> を <code>XMLCompositeRuleFactory</code> で生成し保管する機能を提供します。 <br>
 * 一度読み込まれたリソースはキャッシュされるため２度目以降には読み取りは行われません。
 */
public class CompositeRuleFactory {

    /* 構成ルールをキャシュします。 */
    private static final Map composeRuleCache = new HashMap();

    /**
     * 構成ルールを生成して返却します。
     * 
     * @param rulePath
     *            構成ルールファイルへのクラスパス
     * @param classLoader
     *            生成に使用するクラスローダ
     * @return 構成ルール
     * @throws NoSuchResourceException
     *             リソースが見つからなかった場合
     */
    public static CompositeRule getComposeRule(String rulePath, ClassLoader classLoader) throws NoSuchResourceException {
        synchronized (composeRuleCache) {
            CompositeRule rule = (CompositeRule) composeRuleCache.get(rulePath);
            if (rule == null) {
                rule = loadComposeRule(rulePath, classLoader);
                composeRuleCache.put(rulePath, rule);
            }
            return rule;
        }
    }

    /* 構成ルールを読み取ります。 */
    private static CompositeRule loadComposeRule(String rulePath, ClassLoader classLoader)
            throws NoSuchResourceException {
        InputStream inStream = ResourceLoader.getResourceAsStream(rulePath, classLoader);
        XMLCompositeRuleFactory factory = new XMLCompositeRuleFactory();
        factory.setClassLoader(classLoader);
        return factory.create(inStream);
    }

}
