package shohaku.composer;

import org.xml.sax.Attributes;

import shohaku.core.lang.Eval;

/**
 * タグの属性情報を保管する機能を実装する。
 */
public class TagAttributes implements Attributes {

    /** タグの属性情報。 */
    private String[] values;

    /**
     * 空の属性情報で初期化する。
     */
    public TagAttributes() {
        this.values = new String[0];
    }

    /**
     * 属性情報を内部形式に変換して初期化する。
     * 
     * @param attributes
     *            属性情報
     */
    public TagAttributes(Attributes attributes) {
        int size = attributes.getLength();
        this.values = new String[size * 5];
        for (int i = 0, k = 0; i < size; i++) {
            this.values[k++] = attributes.getLocalName(i);
            this.values[k++] = attributes.getQName(i);
            this.values[k++] = attributes.getType(i);
            this.values[k++] = attributes.getURI(i);
            this.values[k++] = attributes.getValue(i);
        }
    }

    /**
     * タグ属性情報を内部形式に変換して初期化する。
     * 
     * @param tagAttributes
     *            タグ属性情報
     */
    public TagAttributes(TagAttributes tagAttributes) {
        this.values = new String[tagAttributes.values.length];
        System.arraycopy(tagAttributes.values, 0, this.values, 0, this.values.length);
    }

    /*
     * Attributes
     */

    /**
     * 属性数を返却する。
     * 
     * @return 属性数
     */
    public int getLength() {
        return (this.values.length / 5);
    }

    /**
     * 名前空間の属性のローカル名を返却する。
     * 
     * @param index
     *            インデックス
     * @return 名前空間の属性のローカル名
     */
    public String getLocalName(int index) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            return null;
        }
        return this.values[index * 5 + 0];
    }

    /**
     * 属性名を返却する。
     * 
     * @param index
     *            インデックス
     * @return 属性名
     */
    public String getQName(int index) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            return null;
        }
        return this.values[index * 5 + 1];
    }

    /**
     * 属性のタイプを返却する。
     * 
     * @param index
     *            インデックス
     * @return 属性のタイプ
     */
    public String getType(int index) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            return null;
        }
        return this.values[index * 5 + 2];
    }

    /**
     * 名前空間URIを返却する。
     * 
     * @param index
     *            インデックス
     * @return 名前空間URI
     */
    public String getURI(int index) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            return null;
        }
        return this.values[index * 5 + 3];
    }

    /**
     * 属性値を返却する。
     * 
     * @param index
     *            インデックス
     * @return 属性値
     */
    public String getValue(int index) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            return null;
        }
        return this.values[index * 5 + 4];
    }

    /**
     * 属性名のインデックスを返却する。
     * 
     * @param qName
     *            属性名
     * @return 属性名のインデックス
     */
    public int getIndex(String qName) {
        return findIndexByQName(qName);
    }

    /**
     * 属性名のタイプを返却する。
     * 
     * @param qName
     *            属性名
     * @return 属性名のタイプ
     */
    public String getType(String qName) {
        int index = findIndexByQName(qName);
        if (index < 0) {
            return null;
        }
        return getType(index);
    }

    /**
     * 属性名の値を返却する。
     * 
     * @param qName
     *            属性名
     * @return 属性名の値
     */
    public String getValue(String qName) {
        int index = findIndexByQName(qName);
        if (index < 0) {
            return null;
        }
        return getValue(index);
    }

    /**
     * 属性名が示すインデックスを返却する。
     * 
     * @param qName
     *            属性名
     * @return インデックス
     */
    private int findIndexByQName(String qName) {
        int size = getLength();
        for (int i = 0; i < size; i++) {
            if (qName.equals(getQName(i))) {
                return i;
            }
        }
        return -1;
    }

    /**
     * 属性名の値を変更する。
     * 
     * @param index
     *            インデックス
     * @param value
     *            属性名の値
     * @throws IllegalArgumentException
     *             インデックスが範囲外の場合
     */
    public void setValue(int index, String value) {
        if (!Eval.isRange(index, 0, (getLength() - 1))) {
            throw new IllegalArgumentException();
        }
        this.values[index * 5 + 4] = value;
    }

    /**
     * 属性名の値を変更する。
     * 
     * @param qName
     *            属性名
     * @param value
     *            属性名の値
     * @throws IllegalArgumentException
     *             属性名が存在しない場合
     */
    public void setValue(String qName, String value) {
        int index = findIndexByQName(qName);
        if (index < 0) {
            throw new IllegalArgumentException();
        }
        this.values[index * 5 + 4] = value;
    }

    /*
     * Name Space
     */

    /**
     * 名前空間URIとローカル名が示すインデックスを返却する。
     * 
     * @param uri
     *            名前空間URI
     * @param localName
     *            ローカル名
     * @return インデックス
     */
    public int getIndex(String uri, String localName) {
        return findIndexByUriAndLocalName(uri, localName);
    }

    /**
     * 名前空間URIとローカル名が示す属性のタイプを返却する。
     * 
     * @param uri
     *            名前空間URI
     * @param localName
     *            ローカル名
     * @return 属性のタイプ
     */
    public String getType(String uri, String localName) {
        int index = findIndexByUriAndLocalName(uri, localName);
        return (-1 != index) ? getType(index) : null;
    }

    /**
     * 名前空間URIとローカル名が示す属性値を返却する。
     * 
     * @param uri
     *            名前空間URI
     * @param localName
     *            ローカル名
     * @return 属性値
     */
    public String getValue(String uri, String localName) {
        int index = findIndexByUriAndLocalName(uri, localName);
        return (-1 != index) ? getValue(index) : null;
    }

    /**
     * 名前空間URIとローカル名が示すインデックスを返却する。
     * 
     * @param uri
     *            名前空間URI
     * @param localName
     *            ローカル名
     * @return インデックス
     */
    private int findIndexByUriAndLocalName(String uri, String localName) {
        int size = getLength();
        for (int i = 0; i < size; i++) {
            if (uri.equals(getURI(i)) && localName.equals(getLocalName(i))) {
                return i;
            }
        }
        return -1;
    }
}