/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 *
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer;

import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

import shohaku.core.collections.Parameters;

/**
 * 構造化に使用するコンテンツハンドラを実装します。
 */
class SAXContentHandler implements ContentHandler {

    /* スキップカウンタの無効を示す値。 */
    private static final int NO_SKIP_BODY = -1;

    /* Composer. */
    private final Composer composer;

    /* 解析中のノード構成ルールを一時保管します。 */
    private final LinkedList stackCreateRule = new LinkedList();

    /* ノード構成ルールを保管します。 */
    private final LinkedList createRuleList = new LinkedList();

    /* XML階層を下るSubstitutor. */
    private final Substitutor hierarchicalSubstitutor = new HierarchicalSubstitutor(this.stackCreateRule);

    /* 解析中の名前階層URIを保管します。 */
    private String currentUri;

    /* コンテンツの解析を無視する場合のスキップカウンタ。 */
    private int skipBody;

    /* コンテンツのキャッシュを行うかを示すフラグ。 */
    private boolean isCacheNode;

    SAXContentHandler(Composer composer) {
        this.composer = composer;
        initialize();
    }

    /* このオブジェクトを初期化します。 */
    private void initialize() {
        this.currentUri = "";
        this.skipBody = NO_SKIP_BODY;
        this.isCacheNode = false;
    }

    ////////////////////////////////////////////////////////////////////
    // SAX ContentHandler.
    ////////////////////////////////////////////////////////////////////

    /**
     * Receive a Locator object for document events.
     * 
     * @see org.xml.sax.ContentHandler#setDocumentLocator
     */
    public void setDocumentLocator(Locator locator) {
        // no op
    }

    /**
     * 文書の開始通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#startDocument()
     */
    public void startDocument() throws SAXException {
        // no op
    }

    /**
     * Receive notification of the start of a Namespace mapping.
     * 
     * @see org.xml.sax.ContentHandler#startPrefixMapping
     */
    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        // no op
    }

    /**
     * Receive notification of the end of a Namespace mapping.
     * 
     * @see org.xml.sax.ContentHandler#endPrefixMapping
     */
    public void endPrefixMapping(String prefix) throws SAXException {
        // no op
    }

    /**
     * 要素の開始通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String,
     *      org.xml.sax.Attributes)
     */
    public void startElement(String namespaceURI, String localName, String qName, Attributes atts) throws SAXException {

        if (this.skipBody != NO_SKIP_BODY) {
            this.skipBody++;
            return;
        }
        try {

            String nodeName = getNodeName(localName, qName);

            startElement(namespaceURI, nodeName, localName, qName, new TagAttributes(atts));

        } catch (Exception e) {
            handleException("Composer.startElement: ", e);
        }
    }

    /**
     * 要素の開始通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#startElement(java.lang.String, java.lang.String, java.lang.String,
     *      org.xml.sax.Attributes)
     */
    void startElement(String namespaceURI, String nodeName, String localName, String qName, TagAttributes atts)
            throws SAXException {

        try {

            downCurrentURI(nodeName);

            NodeCreateRule createRule = createNodeCreateRule(namespaceURI, nodeName);

            NodeCreateRule parentRule = (!empty()) ? peek() : null;

            push(createRule);

            createRule.begin(this.currentUri, namespaceURI, nodeName, localName, qName, atts, parentRule);

            Node node = createRule.getNodeObject();

            if (size() == 1) {
                DocumentContext context = composer.getDocumentContext();
                context.setRoot(node);
                String publicId = this.composer.getCompositeRule().getPublicId(node);
                context.setPublicId(publicId);
            }

            if (node instanceof CacheContentNode) {
                this.isCacheNode = true;
            }

            if (node instanceof EvaluationNode) {
                startEvaluation(node);
            }

            fireContextStartElement(node);

        } catch (Exception e) {
            handleException("Composer.startElement: ", e);
        }
    }

    /**
     * 要素内の文字データの通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#characters(char[], int, int)
     */
    public void characters(char[] buffer, int start, int length) throws SAXException {

        if (this.skipBody != NO_SKIP_BODY) {
            return;
        }

        try {

            if (!empty()) {

                NodeCreateRule createRule = peek();
                char[] chars = new char[length];
                System.arraycopy(buffer, start, chars, 0, length);
                createRule.addChars(this.currentUri, chars);

            }

        } catch (Exception e) {
            handleException("Composer.characters: ", e);
        }
    }

    /**
     * 要素の終了通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#endElement(java.lang.String, java.lang.String, java.lang.String)
     */
    public void endElement(String namespaceURI, String localName, String qName) throws SAXException {

        if (this.skipBody != NO_SKIP_BODY) {
            if (this.skipBody == 0) {
                this.skipBody = NO_SKIP_BODY;
            } else {
                this.skipBody--;
                return;
            }
        }

        try {
            //if (!empty()) {

            NodeCreateRule createRule = pop();

            Node node = createRule.getNodeObject();

            if (node instanceof CacheContentNode) {
                this.isCacheNode = false;
            }

            if (node instanceof EvaluationNode) {
                endEvaluation(node);
            }

            String name = getNodeName(localName, qName);
            createRule.end(this.currentUri, namespaceURI, name, localName, qName);
            //}

            upCurrentURI();

            fireContextEndElement(node);

        } catch (Exception e) {
            handleException("Composer.endElement: ", e);
        }
    }

    /**
     * 文書の終了通知を受け取ります。
     * 
     * @see org.xml.sax.ContentHandler#endDocument()
     */
    public void endDocument() throws SAXException {
        try {

            for (Iterator i = this.createRuleList.iterator(); i.hasNext();) {
                NodeCreateRule createRule = (NodeCreateRule) i.next();
                createRule.finish();
            }

        } catch (Exception e) {
            handleException("Composer.endDocument: ", e);
        }

    }

    /**
     * Receive notification of ignorable whitespace in element content.
     * 
     * @see org.xml.sax.ContentHandler#ignorableWhitespace
     */
    public void ignorableWhitespace(char ch[], int start, int length) throws SAXException {
        // no op
    }

    /**
     * Receive notification of a processing instruction.
     * 
     * @see org.xml.sax.ContentHandler#processingInstruction
     */
    public void processingInstruction(String target, String data) throws SAXException {
        // no op
    }

    /**
     * Receive notification of a skipped entity.
     * 
     * @see org.xml.sax.ContentHandler#processingInstruction
     */
    public void skippedEntity(String name) throws SAXException {
        // no op
    }

    ////////////////////////////////////////////////////////////////////
    // Stack NodeCreateRule.
    ////////////////////////////////////////////////////////////////////

    private boolean empty() {
        return (size() == 0);
    }

    private int size() {
        return this.stackCreateRule.size();
    }

    private void push(NodeCreateRule createRule) {
        this.stackCreateRule.addLast(createRule);
        this.createRuleList.add(createRule);
    }

    private NodeCreateRule pop() {
        return (NodeCreateRule) this.stackCreateRule.removeLast();
    }

    private NodeCreateRule peek() {
        return (NodeCreateRule) this.stackCreateRule.getLast();
    }

    ////////////////////////////////////////////////////////////////////
    // EvaluationNode.
    ////////////////////////////////////////////////////////////////////

    private void startEvaluation(Node node) {
        int eval = ((EvaluationNode) node).doInitBody();
        if (eval == EvaluationNode.SKIP_BODY) {
            this.skipBody = 0;
        }
    }

    private void endEvaluation(Node node) throws SAXException {
        EvaluationNode bodyEvalNode = ((EvaluationNode) node);
        while (EvaluationNode.EVAL_BODY == bodyEvalNode.doEvalBody()) {
            //children or chars
            evaluationChilds(node);
        }
    }

    private void evaluationContent(Node node) throws SAXException {
        NodeContext n = node.getNodeContext();
        String namespaceURI = n.getNodeNamespaceURI();
        String nodeName = n.getNodeName();
        String localName = n.getNodeLocalName();
        String qName = n.getNodeQName();
        TagAttributes atts = n.getElemAttributes();

        //startElement
        startElement(namespaceURI, nodeName, localName, qName, atts);
        //children or chars
        evaluationChilds(node);
        //endElement
        endElement(namespaceURI, localName, qName);
    }

    private void evaluationChilds(Node parent) throws SAXException {
        //children or chars
        for (Iterator i = parent.getNodeContext().childIterator(); i.hasNext();) {
            Node child = (Node) i.next();
            if (child.isType(Node.TYPE_TEXT)) {
                String text = child.getText();
                characters(text.toCharArray(), 0, text.length());
            } else {
                evaluationContent(child);
            }
        }
    }

    ////////////////////////////////////////////////////////////////////
    // helper mathod.
    ////////////////////////////////////////////////////////////////////

    /*
     * Creater
     */

    private NodeCreateRule createNodeCreateRule(String namespaceURI, String nodeName) throws SAXException {

        if (isCacheNode) {

            NodeRule nodeRule = createCacheNodeRule();
            return new NodeCreateRule(this.composer.getDocumentContext(), nodeRule, null);

        } else {

            NodeRule nodeRule = findNodeRule(namespaceURI, nodeName);
            if (nodeRule == null) {
                throw new SAXException("ノードの構成ルールの取得に失敗しました uri:" + this.currentUri + ", name:" + nodeName);
            }
            return new NodeCreateRule(this.composer.getDocumentContext(), nodeRule, this.hierarchicalSubstitutor);
        }
    }

    private NodeRule createCacheNodeRule() {
        NodeRule nodeRule;
        NodeRule set = new NodeRule();
        set.setNodeClass(CacheNode.class.getName());
        set.setExAttributes(Parameters.EMPTY_PARAMETERS);
        set.setPattern(this.currentUri);
        nodeRule = set;
        return nodeRule;
    }

    /*
     * Current URI
     */

    private void downCurrentURI(String name) {
        StringBuffer sb = new StringBuffer(this.currentUri);
        sb.append('/');
        sb.append(name);
        this.currentUri = sb.toString();
    }

    private void upCurrentURI() {
        String uri = this.currentUri;
        int slash = uri.lastIndexOf('/');
        if (slash >= 0) {
            this.currentUri = uri.substring(0, slash);
        } else {
            this.currentUri = "";
        }
    }

    /*
     * Node
     */

    private void fireContextStartElement(Node node) {
        this.composer.getDocumentContext().startElement(node);
        if (Node.SCOPE_PUBLIC == node.getScope()) {
            this.composer.getCompositeContext().startElement(node);
        }
    }

    private void fireContextEndElement(Node node) {
        this.composer.getDocumentContext().endElement(node);
        if (Node.SCOPE_PUBLIC == node.getScope()) {
            this.composer.getCompositeContext().endElement(node);
        }
    }

    private NodeRule findNodeRule(String namespaceURI, String nodeName) {
        return this.composer.getCompositeRule().findNodeRule(namespaceURI, this.currentUri, nodeName);
    }

    /*
     * error handler
     */

    private void handleException(String msg, Exception e) throws SAXException {

        if (this.composer.isErrThrowable()) {
            if (e instanceof SAXException) {
                throw (SAXException) e;
            } else {
                throw new SAXException(msg, e);
            }
        } else {
            error(msg, e);
        }

    }

    /*
     * Logger
     */

    private void error(Object message, Throwable t) {
        this.composer.getLogger().error(message, t);
    }

    /*
     * static
     */

    private static String getNodeName(String localName, String qName) {
        String name = localName;
        if ((name == null) || (name.length() < 1)) {
            name = qName;
        }
        return name;
    }

    //    private static PlainMap attributesToPlainMap(Attributes atts) {
    //        int length = atts.getLength();
    //        String[] a = new String[length * 2];
    //        for (int i = 0, k = 0; i < length; i++) {
    //            a[k++] = getNodeName(atts.getLocalName(i), atts.getQName(i));
    //            a[k++] = atts.getValue(i);
    //        }
    //        return PlainMapUtils.segmentPlainMap(a);
    //    }

    private static class HierarchicalSubstitutor implements Substitutor {
        private List stack;

        private HierarchicalSubstitutor(List stack) {
            this.stack = stack;
        }

        public TagAttributes substitute(DocumentContext context, String uri, String nodeName, TagAttributes atts) {
            Iterator i = this.stack.listIterator();
            while (i.hasNext()) {
                Substitutor substitutor = ((NodeCreateRule) i.next()).getNodeRule().getSubstitutor();
                if (substitutor != null) {
                    atts = substitutor.substitute(context, uri, nodeName, atts);
                }
            }
            return atts;
        }

        public String substitute(DocumentContext context, String uri, String nodeName, String text) {
            String rtext = text;
            Iterator i = this.stack.listIterator();
            while (i.hasNext()) {
                Substitutor substitutor = ((NodeCreateRule) i.next()).getNodeRule().getSubstitutor();
                if (substitutor != null) {
                    rtext = substitutor.substitute(context, uri, nodeName, rtext);
                }
            }
            return rtext;
        }
    }

}