/*
 * shohaku Copyright (C) 2005 tomoya nagatani
 * 
 * This library is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */
package shohaku.composer;

import shohaku.core.lang.ObjectCreationException;
import shohaku.core.lang.ResourceLoader;

/**
 * ノードを生成する機能を提供します。
 */
class NodeCreateRule {

    /*
     * private
     */

    /* Composer. */
    private final Composer composer;

    /* DocumentContext. */
    private final DocumentContext documentContext;

    /* 値の設定情報。 */
    private final NodeRule nodeRule;

    /* ノードの属性やテキスト文字列を代入する機能。 */
    private final Substitutor substitutor;

    /* ノード。 */
    private Node nodeObject;

    /* ノード属性情報。 */
    private NodeContext nodeContext;

    /* 本体のテキストを一時保管します。 */
    private StringBuffer charBuffer = new StringBuffer();

    /**
     * 情報を格納して初期化します。
     * 
     * @param context
     * @param nodeRule
     * @param substitutor
     */
    NodeCreateRule(DocumentContext context, NodeRule nodeRule, Substitutor substitutor) {
        super();
        this.composer = context.getCompositeContext().getComposer();
        this.documentContext = context;
        this.nodeRule = nodeRule;
        this.substitutor = substitutor;
    }

    /**
     * @param uri
     * @param namespace
     * @param nodeName
     * @param localName
     * @param qName
     * @param attributes
     * @param parentRule
     * @throws Exception
     */
    void begin(String uri, String namespace, String nodeName, String localName, String qName,
            TagAttributes attributes, NodeCreateRule parentRule) throws Exception {

        if (composer.getLogger().isDebugEnabled()) {
            composer.getLogger().debug(
                    "NodeCreateRule#begin():uri:" + uri + ", namespace:" + namespace + ", nodeName:" + nodeName
                            + ", localName:" + localName + ", qName:" + qName + ", attributes:" + attributes);
        }

        //substitutor 属性の代入
        if (null != this.substitutor) {
            attributes = this.substitutor.substitute(this.documentContext, uri, nodeName, attributes);
        }

        this.nodeObject = createNode();

        boolean isNotHierarchical = (this.nodeObject instanceof NotHierarchyNode);

        if (!isNotHierarchical && parentRule != null) {
            this.nodeContext = new NodeContext(this.documentContext, this.nodeRule, parentRule.getNodeObject(), uri,
                    namespace, localName, qName, nodeName, attributes);
            parentRule.addChild(uri, this.nodeObject);
        } else {
            this.nodeContext = new NodeContext(this.documentContext, this.nodeRule, null, uri, namespace, localName,
                    qName, nodeName, attributes);
        }

        //init
        this.nodeObject.initialize(this.nodeContext);

        //Node.begin
        this.nodeObject.begin();
    }

    /**
     * @param uri
     * @param chars
     * @throws Exception
     */
    void addChars(String uri, char[] chars) throws Exception {

        if (composer.getLogger().isDebugEnabled()) {
            composer.getLogger().debug("NodeCreateRule#chars():uri:" + uri + ":" + "chars:" + String.valueOf(chars));
        }

        this.charBuffer.append(chars);

    }

    /**
     * @param uri
     * @param child
     * @throws Exception
     */
    void addChild(String uri, Node child) throws Exception {

        if (composer.getLogger().isDebugEnabled()) {
            composer.getLogger().debug(
                    "NodeCreateRule#addChild():uri:" + uri + ", child:" + child.getNodeContext().getNodeName());
        }

        addIfTextNode();

        //NodeContext.addChild
        this.nodeContext.addChild(child);

        //Node.child
        this.nodeObject.child(child);
    }

    private void addIfTextNode() throws Exception {
        if (charBuffer.length() > 0) {

            //get text
            String text = this.charBuffer.toString();

            //init chars
            charBuffer.setLength(0);

            //substitutor テキストの代入
            if (null != this.substitutor) {
                text = this.substitutor.substitute(this.documentContext, this.nodeContext.getNodeURI(),
                        this.nodeContext.getNodeQName(), text);
            }

            //create TextNode
            TextNode child = new TextNode(text);

            //NodeContext.addChild
            this.nodeContext.addChild(child);

            //Node.child
            this.nodeObject.child(child);

        }
    }

    /**
     * @param uri
     * @param namespace
     * @param nodeName
     * @param localName
     * @param qName
     * @throws Exception
     */
    void end(String uri, String namespace, String nodeName, String localName, String qName) throws Exception {

        if (composer.getLogger().isDebugEnabled()) {
            composer.getLogger().debug(
                    "NodeCreateRule#end():uri:" + uri + ", namespace:" + namespace + ", nodeName:" + nodeName
                            + "localName:" + localName + ", qName:" + qName);
        }

        addIfTextNode();

        //Node.end
        this.nodeObject.end();
    }

    /**
     * @throws Exception
     */
    void finish() throws Exception {

        if (composer.getLogger().isDebugEnabled()) {
            composer.getLogger().debug("NodeCreateRule#finish():" + nodeObject.getClass());
        }

        //Node.finish
        this.nodeObject.finish();

        //clear
        this.nodeObject = null;
        this.nodeContext = null;
        this.charBuffer = null;
    }

    private Node createNode() throws ObjectCreationException {
        Node ret;
        String className = this.nodeRule.getNodeClass();
        ClassLoader classLoader = this.composer.getClassLoader();
        ret = (Node) ResourceLoader.getInstance(className, classLoader);
        if (ret == null) {
            throw new ObjectCreationException("className:" + this.nodeRule.getNodeClass());
        }
        return ret;
    }

    Node getNodeObject() {
        return nodeObject;
    }

    NodeRule getNodeRule() {
        return nodeRule;
    }
}